package ru.yandex.calendar.logic.sending.param;

import org.jdom.Element;
import org.joda.time.DateTimeZone;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;
import org.joda.time.LocalTime;
import org.joda.time.format.DateTimeFormat;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.sharing.MailType;
import ru.yandex.calendar.logic.user.NameI18n;
import ru.yandex.calendar.util.dates.DateTimeFormatter;
import ru.yandex.calendar.util.resources.UStringLiteral;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.inside.passport.PassportSid;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.annotation.FillWithSomething;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.random.Random2;

/**
 * @author Stepan Koltsov
 */
@BenderBindAllFields
public class NotificationMessageParameters extends EventMessageParameters {

    public static final int SMS_MAX_FIELD_LENGTH = 100;

    @FillWithSomething
    public NotificationMessageParameters(
            CommonEventMessageParameters commonMessageParameters, EventMessageInfo eventMessageInfo)
    {
        super(commonMessageParameters, eventMessageInfo);
    }

    private static final org.joda.time.format.DateTimeFormatter readableDatePatternY;

    static {
        // workaround for:
        // DateTimeFormat.parsePatternTo calls new DateTimeZone().getYear()
        DateTimeZone tz = DateTimeZone.getDefault();
        DateTimeZone.setDefault(DateTimeZone.UTC);
        readableDatePatternY = DateTimeFormat.forPattern(DateTimeFormatter.READABLE_DATE_PATTERN_Y);
        DateTimeZone.setDefault(tz);
    }

    private String formatTime(boolean printToday) {
        LocalDate nowLocalDate = meta.getNow().toLocalDate();
        LocalDate eventLocalDate = event.getEventStartTs().toLocalDate();

        Option<NameI18n> when;

        if (nowLocalDate.equals(eventLocalDate)) {
            when = Option.when(printToday || event.time.isAllDay, UStringLiteral.TODAY);

        } else if (nowLocalDate.plusDays(1).equals(eventLocalDate)) {
            when = Option.of(UStringLiteral.TOMORROW);

        } else {
            when = Option.of(new NameI18n(readableDatePatternY.print(eventLocalDate), Option.empty()));
        }

        if (!event.time.isAllDay) {
            LocalTime eventLocalTime = event.getEventStartTs().toLocalTime();

            return when.map(n -> n.concat(new NameI18n(" в ", " at "))).getOrElse(new NameI18n("В ", "At "))
                    .getName(meta.getLanguage()) + DateTimeFormatter.formatLocalTimeForHuman(eventLocalTime);
        } else {
            return when.get().getName(meta.getLanguage());
        }
    }

    private Option<String> formatLocation() {
        return StringUtils.notEmptyO(event.getLocation().asTextForSms(meta.getLanguage()));
    }

    private String trim(String field) {
        return trim(field, SMS_MAX_FIELD_LENGTH);
    }

    private String trim(String field, int length) {
        if (field.length() <= length) {
            return field;
        } else {
            return field.substring(0, length - 3) + "...";
        }
    }

    public String formatYamb() {
        return Cf.list(formatTime(false), "«" + trim(event.name) + "»")
                .plus(formatLocation().map(s -> "(" + s + ")")).mkString(" ")
                + "\n" + meta.getCalendarUrl() + "/event?event_id=" + event.eventId;
    }

    public String formatSms() {
        String r = formatTime(true) + " \"" + trim(event.name) + "\"";

        r += formatLocation().map(s -> " (" + trim(s) + ")").getOrElse("");

        r += !event.sid.isSome(PassportSid.MAIL) ? ". Я.Календарь" : ". Яндекс.Почта";

        return r;
    }

    @Override
    public NotificationMessageParameters withMeta(CommonEventMessageParameters meta) {
        return new NotificationMessageParameters(meta, event);
    }

    @Override
    public MailType mailType() {
        return MailType.EVENT_NOTIFICATION;
    }

    public Element toOldStyleXml() {
        Element e = super.toOldStyleXml();

        if (meta.getRecipient().getUid().isMatch(PassportUid::isYandexTeamRu)) {
            Option<String> where = StringUtils.notEmptyO(
                    event.getLocation().asTextForMailSubject(meta.getLanguage()));

            CalendarXmlizer.appendElm(e, "subject", Cf.<String>list()
                    .plus(formatTime(false))
                    .plus("«" + trim(event.name, 50) + "»")
                    .plus(where.map(s -> "(" + s + ")")).mkString(" "));
        }
        return e;
    }

    public MessageParameters withGeneratedMessageId(String url) {
        return withOverrides(getMessageOverrides().withMessageId(
                Random2.R.nextLetters(20) + "@" + url));
    }

    public LocalDateTime getEventStartTs() {
        return event.getEventStartTs();
    }
} //~
