package ru.yandex.calendar.logic.sending.param;

import org.jdom.Element;
import org.joda.time.DateTimeZone;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.beans.generated.TodoItem;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsCalendar;
import ru.yandex.calendar.logic.sharing.MailType;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.commune.mail.MailAddress;
import ru.yandex.misc.annotation.FillWithSomething;
import ru.yandex.misc.annotation.SampleValue;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;

/**
 * @author gutman
 */
@BenderBindAllFields
public class TodoListMessageParameters extends OverridableMessageParameters {

    private final Language language;
    private final IcsCalendar todoListIcsCalendar;
    private final String todoListName;
    private final ListF<TodoItemParameters> todoItems;
    private final MailAddress sender;
    private final MailAddress recipient;
    private final boolean toYandexUser;
    private final DateTimeZone tz;
    private final String calendarUrl;

    @FillWithSomething
    public TodoListMessageParameters(
            Language language,
            IcsCalendar todoListIcsCalendar, String todoListName, ListF<TodoItem> todoItems,
            MailAddress sender, MailAddress recipient, boolean toYandexUser, DateTimeZone tz,
            @SampleValue("http://calendar.yandex-team.ru/") String calendarUrl)
    {
        this(MessageOverrides.EMPTY,
                language, todoListIcsCalendar, todoListName, todoItems.map(TodoItemParameters::of),
                sender, recipient, toYandexUser, tz, calendarUrl.replaceFirst("/$", ""));
    }

    private TodoListMessageParameters(
            MessageOverrides overrides, Language language,
            IcsCalendar todoListIcsCalendar, String todoListName,
            ListF<TodoItemParameters> todoItems, MailAddress sender, MailAddress recipient, boolean toYandexUser,
            DateTimeZone tz, String calendarUrl)
    {
        super(overrides);
        this.language = language;
        this.todoListIcsCalendar = todoListIcsCalendar;
        this.todoListName = todoListName;
        this.todoItems = todoItems;
        this.sender = sender;
        this.recipient = recipient;
        this.toYandexUser = toYandexUser;
        this.tz = tz;
        this.calendarUrl = calendarUrl;
    }

    @Override
    public MailType mailType() {
        return MailType.TODO_LIST;
    }

    @Override
    public Element toOldStyleXml() {
        Element message = new Element("message");
        CalendarXmlizer.appendElm(message, "language", language.value());
        CalendarXmlizer.appendElm(message, "ics", todoListIcsCalendar.serializeToString());
        CalendarXmlizer.appendElm(message, "todo-list-name", todoListName);
        CalendarXmlizer.appendElm(message, "to", recipient.getEmail().getEmail());
        CalendarXmlizer.appendElm(message, "nick", recipient.getPersonal());
        CalendarXmlizer.appendElm(message, "sender-email", sender.getEmail().getEmail());
        CalendarXmlizer.appendElm(message, "sender-nick", sender.getPersonal());
        CalendarXmlizer.appendElm(message, "to-yandex-user", toYandexUser);

        CalendarXmlizer.appendElm(message, "calendar-url", calendarUrl);

        Element todoItemsElement = new Element("todo-items");
        for (TodoItemParameters todoItem : todoItems) {
            Element todoItemElement = new Element("todo-item");
            CalendarXmlizer.appendElm(todoItemElement, "title", todoItem.getTitle());
            CalendarXmlizer.appendElm(todoItemElement, "completed",
                    Boolean.toString(todoItem.getCompletionTs().isPresent()));
            // XXX ssytnik: bug, is serialized as Option<Instant>.toString()
            CalendarXmlizer.appendDtfElm(todoItemElement, "due-ts", todoItem.getDueTs(), tz);
            todoItemsElement.addContent(todoItemElement);
        }
        message.addContent(todoItemsElement);

        return message;
    }

    @Override
    public Recipient getRecipient() {
        return Recipient.of(recipient, Option.empty());
    }

    @Override
    public OverridableMessageParameters withOverrides(MessageOverrides overrides) {
        return new TodoListMessageParameters(
                overrides, language, todoListIcsCalendar,
                todoListName, todoItems, sender, recipient, toYandexUser, tz, calendarUrl);
    }

    @Override
    public String toString() {
        return getClass().getSimpleName() + "{"
                + "items=" + todoItems.size()
                + ", to=" + getRecipientEmail()
                + "}";
    }
}
