package ru.yandex.calendar.logic.sending.param;

import org.jdom.Element;
import org.joda.time.LocalDateTime;
import org.joda.time.LocalTime;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.beans.generated.TodoItem;
import ru.yandex.calendar.logic.beans.generated.TodoList;
import ru.yandex.calendar.logic.contact.UnivContact;
import ru.yandex.calendar.logic.sharing.MailType;
import ru.yandex.calendar.logic.todo.TodoMailType;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.util.email.Emails;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.commune.mail.MailAddress;
import ru.yandex.misc.annotation.FillWithSomething;
import ru.yandex.misc.annotation.SampleValue;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.lang.StringUtils;

@BenderBindAllFields
public class TodoMessageParameters extends OverridableMessageParameters {

    public final LocalDateTime now;

    public final ListF<TodoListParameters> lists;
    public final ListF<TodoItemParameters> items;

    public final MailAddress sender;
    public final UnivContact recipient;

    public final Language language;
    public final String calendarUrl;

    public final TodoMailType mailType;

    @FillWithSomething
    public TodoMessageParameters(
            LocalDateTime now, ListF<TodoList> lists, ListF<TodoItem> items,
            MailAddress sender, UnivContact recipient, Language language,
            @SampleValue("http://calendar.yandex-team.ru/") String calendarUrl, TodoMailType mailType)
    {
        this(MessageOverrides.EMPTY, now,
                lists.sortedBy(TodoList::getCreationTs).map(TodoListParameters::of),
                items.sortedBy(TodoItem::getCreationTs).map(TodoItemParameters::of),
                sender, recipient, language, calendarUrl, mailType);
    }

    private TodoMessageParameters(
            MessageOverrides overrides,
            LocalDateTime now, ListF<TodoListParameters> lists,
            ListF<TodoItemParameters> items, MailAddress sender, UnivContact recipient,
            Language language, String calendarUrl, TodoMailType mailType)
    {
        super(overrides);
        this.now = now;
        this.lists = lists;
        this.items = items;
        this.sender = sender;
        this.recipient = recipient;
        this.language = language;
        this.calendarUrl = calendarUrl;
        this.mailType = mailType;
    }

    @Override
    public MailType mailType() {
        return MailType.TODO_NOTIFICATION;
    }

    @Override
    public Element toOldStyleXml() {
        Element message = new Element("message");

        CalendarXmlizer.appendElm(message, "type", mailType.name().toLowerCase());

        Element todoItemsElement = new Element("todo-items");
        for (TodoItemParameters item : items) {
            Element todoItemElement = new Element("todo-item");
            CalendarXmlizer.appendElm(todoItemElement, "title", item.getTitle());
            todoItemsElement.addContent(todoItemElement);
        }
        message.addContent(todoItemsElement);

        Element todoListsElement = new Element("todo-lists");
        for (TodoListParameters list : lists) {
            Element todoItemElement = new Element("todo-list");
            CalendarXmlizer.appendElm(todoItemElement, "title", list.getTitle());
            todoListsElement.addContent(todoItemElement);
        }
        message.addContent(todoListsElement);

        CalendarXmlizer.appendElm(message, "date-time-now", now.toString());
        CalendarXmlizer.appendElm(message, "event-start-ts", now.withFields(LocalTime.MIDNIGHT));
        CalendarXmlizer.appendElm(message, "event-end-ts", now.plusDays(1).withFields(LocalTime.MIDNIGHT));
        CalendarXmlizer.appendElm(message, "show-is-all-day", true);

        CalendarXmlizer.appendElm(message, "language", language.value());

        CalendarXmlizer.appendElm(message, "to", recipient.getEmail().getEmail());
        CalendarXmlizer.appendElm(message, "nick", StringUtils.defaultIfEmpty(
                recipient.getName(), Emails.getUnicoded(recipient.getEmail())));

        CalendarXmlizer.appendElm(message, "sender-nick", sender.getPersonal());
        CalendarXmlizer.appendElm(message, "sender-email", sender.getEmail());
        CalendarXmlizer.appendElm(message, "email-from", sender.getEmail());
        CalendarXmlizer.appendElm(message, "calendar-url", calendarUrl);

        recipient.getUid().forEach(uid -> CalendarXmlizer.appendElm(message, "uid", uid.getUid()));

        return message;
    }

    @Override
    public Recipient getRecipient() {
        return Recipient.of(recipient, Option.empty());
    }

    @Override
    public OverridableMessageParameters withOverrides(MessageOverrides overrides) {
        return new TodoMessageParameters(overrides, now, lists, items, sender, recipient, language, calendarUrl, mailType);
    }

    @Override
    public String toString() {
        return getClass().getSimpleName() + "{"
                + "items=" + items.size()
                + ", lists=" + lists.size()
                + ", to=" + getRecipientEmail()
                + "}";
    }
}
