package ru.yandex.calendar.logic.sending.real;

import java.util.Optional;
import java.util.Properties;

import javax.mail.Address;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.MimeMessage;

import com.sun.mail.smtp.SMTPAddressFailedException;
import com.sun.mail.smtp.SMTPAddressSucceededException;
import com.sun.mail.smtp.SMTPSendFailedException;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.commune.mail.MailMessage;
import ru.yandex.commune.mail.MailUtils;

@Slf4j
public class ExtendedJavaMailSender {
    private final Session session;

    public ExtendedJavaMailSender() {
        Properties properties = new Properties();
        // https://javaee.github.io/javamail/docs/api/com/sun/mail/smtp/package-summary.html
        properties.setProperty("mail.smtp.host", "localhost");
        //properties.setProperty("mail.debug", "true");
        properties.setProperty("mail.smtp.reportsuccess", "true");
        this.session = Session.getDefaultInstance(properties, null);
    }

    public ExtendedJavaMailSender(Properties properties) {
        Properties patchedProperties = new Properties(properties);
        //patchedProperties.setProperty("mail.debug", "true");
        patchedProperties.setProperty("mail.smtp.reportsuccess", "true");
        this.session = Session.getInstance(patchedProperties);
    }

    public String send(MailMessage message) {
        MimeMessage mimeMessage = null;
        try {
            mimeMessage = MailUtils.toMimeMessage(session, message);
            Transport.send(mimeMessage);
        } catch (SMTPSendFailedException e) {
            boolean hasFailed = false;
            Exception chain = e.getNextException();
            while (chain != null) {
                if (chain instanceof SMTPAddressFailedException) {
                    hasFailed = true;
                    SMTPAddressFailedException smtpAddressFailedException = (SMTPAddressFailedException) chain;
                    log.error("ADDRESS FAILED MailMessage: " + message.getMessageId()
                            + ", mimeMessage: " + getMimeMessageMessageID(mimeMessage)
                            + ", address: " + smtpAddressFailedException.getAddress()
                            + ", command: " + smtpAddressFailedException.getCommand()
                            + ", returnCode:" + smtpAddressFailedException.getReturnCode()
                            + ", response: " + smtpAddressFailedException.getMessage());
                    chain = ((SMTPAddressFailedException) chain).getNextException();
                } else if (chain instanceof SMTPAddressSucceededException) {
                    SMTPAddressSucceededException succeededException = (SMTPAddressSucceededException) chain;
                    log.info("MailMessage: " + message.getMessageId()
                            + ", mimeMessage: " + getMimeMessageMessageID(mimeMessage)
                            + ", message: " + (e.getMessage() + " " + succeededException.getMessage()).replaceAll("\n", "\t")
                            + ", address: " + succeededException.getAddress()
                            + ", command: " + succeededException.getCommand()
                            + ", returnCode:" + succeededException.getReturnCode());
                    chain = ((SMTPAddressSucceededException) chain).getNextException();
                } else {
                    hasFailed = true;
                    log.warn(chain.getClass().toString());
                    chain = null;
                }
            }
            if (hasFailed) {
                log.error("MailMessage: " + getMimeMessageMessageID(mimeMessage)
                        + ", Recipients:" + getAllRecipients(mimeMessage), e);
                throw MailUtils.translate(e);
            } else {
                return e.getMessage().trim();
            }
        } catch (Exception e) {
            if (mimeMessage != null) {
                log.info("MailMessage: " + getMimeMessageMessageID(mimeMessage) + ", Recipients:" + getAllRecipients(mimeMessage));
            }
            throw MailUtils.translate(e);
        }
        return "";
    }

    private Optional<String> getMimeMessageMessageID(MimeMessage mimeMessage) {
        try {
            return Optional.of(mimeMessage.getMessageID());
        } catch (Exception e) {
            //nothing
        }
        return Optional.empty();
    }

    private Optional<String> getAllRecipients(MimeMessage mimeMessage) {
        try {
            final Address[] allRecipients = mimeMessage.getAllRecipients();
            final StringBuilder result = new StringBuilder();
            for (Address allRecipient : allRecipients) {
                result.append(allRecipient).append(",");
            }
            return Optional.of(result.toString());
        } catch (Exception e) {
            //nothing
        }
        return Optional.empty();
    }

}
