package ru.yandex.calendar.logic.sending.real;

import java.util.Optional;
import java.util.UUID;
import java.util.function.BiConsumer;

import io.micrometer.core.instrument.MeterRegistry;
import lombok.val;
import org.jetbrains.annotations.NotNull;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.worker.task.SendMailTask;
import ru.yandex.calendar.logic.beans.generated.SendingMail;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.sending.bazinga.SendingMailDao;
import ru.yandex.calendar.logic.sending.param.MessageParameters;
import ru.yandex.calendar.logic.update.DistributedSemaphore;
import ru.yandex.commune.bazinga.BazingaBender;
import ru.yandex.commune.bazinga.impl.FullJobId;
import ru.yandex.commune.bazinga.impl.JobId;
import ru.yandex.commune.bazinga.impl.JobInfoValue;
import ru.yandex.commune.bazinga.impl.OnetimeJob;
import ru.yandex.commune.bazinga.pg.storage.PgBazingaStorageConfiguration;
import ru.yandex.commune.bazinga.pg.storage.dao.JobJdbcDao;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.mail.MailException;
import ru.yandex.commune.mail.MailMessage;
import ru.yandex.inside.passport.blackbox.PassportAuthDomain;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.TimeUtils;

public class JavaMailSender extends MailSenderSupport implements MailSender {
    private static final Logger logger = LoggerFactory.getLogger(JavaMailSender.class);
    private static final String SEND_MAIL_METRIC = "application.worker.send_single_mail.";

    public enum Status {
        INITIAL(true),
        RATE_LIMIT(true),
        SPAM(false),
        SUCCESSFUL(false),
        ERROR(true);

        Status(boolean retriable) {
            this.retriable = retriable;
        }

        boolean retriable;
    }

    private final DynamicProperty<Boolean> sendEnabled = new DynamicProperty<>("mailSender.send.enabled", true);

    private final ExtendedJavaMailSender publicSender;
    private final ExtendedJavaMailSender corpSender;
    @Autowired
    private PgBazingaStorageConfiguration storage;
    @Autowired
    private SendingMailDao sendingMailDao;
    @Autowired
    private MeterRegistry registry;
    @Autowired
    private DistributedSemaphore distributedSemaphore;

    public JavaMailSender(ExtendedJavaMailSender publicSender, ExtendedJavaMailSender corpSender) {
        this.publicSender = publicSender;
        this.corpSender = corpSender;
    }

    private static OnetimeJob makeJob(SendMailTask task) {
        return new OnetimeJob(
                new FullJobId(task.id(), new JobId(UUID.randomUUID())),
                Option.of(Instant.now()),
                Instant.now(), Option.empty(),
                new String(BazingaBender.mapper.serializeJson(task.getParameters())), JobInfoValue.ready(),
                Option.empty(), Cf.set(), Option.empty(),
                task.priority(), Option.empty());
    }

    @Override
    public void send(DestinedMessage message, ActionInfo actionInfo) {
        val mail = prepareMail(message, actionInfo);
        send(message.getParameters().getRecipientEmail(), mail,
                (status, response) -> logSentToEventsLog(message, mail, status, response, actionInfo));
    }

    @Override
    public void send(Email recipientEmail, MailMessage mail) {
        send(recipientEmail, mail, (status, response) -> {});
    }

    @Override
    public void sendEmailsViaTask(ListF<? extends MessageParameters> parameters, ActionInfo actionInfo) {
        val destined = parameters.flatMap(this::destineMessage);

        val jobs = destined.zipWith(m -> makeJob(new SendMailTask(m, actionInfo)));

        sendingMailDao.insert(jobs.map(job -> {
            val mail = new SendingMail();
            mail.setJobId(job.get2().getJobId().getUuid());
            mail.setCreationTs(actionInfo.getNow());
            mail.setParameters(job.get1().getParameters());
            return mail;
        }));

        Check.equals(jobs.size(), new JobJdbcDao(storage.dataSource, false).insertIgnoreBatch(jobs.get2()));
    }

    private void send(Email recipient, MailMessage mail, BiConsumer<Status, Optional<String>> r) {
        if (!sendEnabled.get()) {
            logger.warn("Sending mail disabled via dynamic property");
            return;
        }
        val start = System.currentTimeMillis();
        Status status = Status.INITIAL;
        try (var l1 = distributedSemaphore.forceTryAcquire("mailSender");
             var l2 = distributedSemaphore.forceTryAcquire("mailSender_" + recipient.getLocalPart())) {

            Optional<String> response = Optional.empty();
            try {
                response = Optional.of(chooseSender(recipient).send(mail));
            } catch (MailException mailExcepion) {
                status = handleStatus(mailExcepion);
                logger.warn("Send mail message fail with id {}, reason {}", mail.getMessageId().get(),
                        mailExcepion.getMessage());
                if (status.retriable) {
                    throw mailExcepion;
                }
            }
            status = status == Status.INITIAL ? Status.SUCCESSFUL : status;
            r.accept(status, response);
        } finally {
            registry.counter(SEND_MAIL_METRIC + status).increment();
            logger.info("Sending from {} to {} with message-id {} {}; took {}",
                    MailSenderUtils.parseSenderEmailsSafe(mail).stableUnique(),
                    MailSenderUtils.parseRecipientEmailsSafe(mail).stableUnique(),
                    mail.getMessageId().get(),
                    status,
                    TimeUtils.millisecondsToSecondsStringToNow(start)
            );
        }
    }

    private ExtendedJavaMailSender chooseSender(Email recipient) {
        if (PassportAuthDomain.byEmail(recipient) == PassportAuthDomain.YANDEX_TEAM_RU) {
            return corpSender;
        }
        return publicSender;
    }

    @NotNull
    private Status handleStatus(MailException mailExcepion) {
        if (mailExcepion.getMessage().contains("Message rejected under suspicion of SPAM")) {
            return Status.SPAM;
        } else if (mailExcepion.getMessage().contains("Invalid Addresses")) {
            return Status.RATE_LIMIT;
        }
        return Status.ERROR;
    }
}
