package ru.yandex.calendar.logic.sending.real;

import java.util.Optional;

import javax.activation.CommandMap;
import javax.activation.MailcapCommandMap;
import javax.activation.MimetypesFileTypeMap;
import javax.xml.transform.Source;

import org.jdom.Document;
import org.jdom.Element;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.PropName;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.log.EventsLogger;
import ru.yandex.calendar.logic.sending.EventMailLogEventJson;
import ru.yandex.calendar.logic.sending.LayerInvitationMailEventJson;
import ru.yandex.calendar.logic.sending.TodoMailLogEventJson;
import ru.yandex.calendar.logic.sending.param.EventMessageParameters;
import ru.yandex.calendar.logic.sending.param.LayerInvitationMessageParameters;
import ru.yandex.calendar.logic.sending.param.MessageDestination;
import ru.yandex.calendar.logic.sending.param.MessageOverrides;
import ru.yandex.calendar.logic.sending.param.MessageParameters;
import ru.yandex.calendar.logic.sending.param.Recipient;
import ru.yandex.calendar.logic.sending.param.TodoMessageParameters;
import ru.yandex.calendar.util.xml.XslUtils;
import ru.yandex.commune.mail.MailMessage;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.xml.jdom.JdomUtils;
import ru.yandex.misc.xml.support.XmlWriteFormat;

/**
 * @author dbrylev
 */
public abstract class MailSenderSupport {

    private static final Logger logger = LoggerFactory.getLogger(MailSenderSupport.class);

    static {
        MimetypesFileTypeMap mt = (MimetypesFileTypeMap) MimetypesFileTypeMap.getDefaultFileTypeMap();
        mt.addMimeTypes("text/calendar ics ICS");

        MailcapCommandMap mc = (MailcapCommandMap) CommandMap.getDefaultCommandMap();
        mc.addMailcap("text/calendar;; x-java-content-handler=com.sun.mail.handlers.text_html");
    }

    @Value("${ews.domain}")
    private String ewsDomain;

    @Value(PropName.DUMP_XSL_INPUT_ANNOT_VALUE)
    private boolean dumpXslInput;
    @Value(PropName.DUMP_XSL_OUTPUT_ANNOT_VALUE)
    private boolean dumpXslOutput;

    @Autowired
    private EventsLogger eventsLogger;


    protected ListF<DestinedMessage> destineMessage(MessageParameters parameters) {
        if (MailSenderUtils.isInacceptableAddress(parameters.getRecipient().address)) {
            logger.warn("Ignored mail with invalid recipient: " + parameters.getRecipientEmail());

            return Cf.list();
        }
        DestinedMessage.XslSource normalSource = DestinedMessage.XslSource.NORMAL;

        Option<DestinedMessage.XslSource> outlookSource = Option.when(
                parameters.mailType().getOutlookerXslName().isPresent(), DestinedMessage.XslSource.OUTLOOK);

        if (Option.of(parameters).filterByType(EventMessageParameters.class).exists(p -> !p.getIcs().isPresent())) {
            outlookSource = Option.empty();
        }

        MessageDestination destination = parameters.getMessageOverrides()
                .destination.getOrElse(MessageDestination.ANYWHERE);

        Recipient recipient = parameters.getRecipient();

        boolean hasLdEmail = recipient.login.isPresent() && recipient.isOutlooker;

        if (destination == MessageDestination.NOT_FOR_EXCHANGE) {
            return Cf.list(new DestinedMessage(DestinedMessage.XslSource.NORMAL, destination, parameters));
        }
        if (destination == MessageDestination.ONLY_FOR_EXCHANGE) {
            return Cf.list(new DestinedMessage(outlookSource.getOrElse(normalSource),
                    hasLdEmail ? destination : MessageDestination.ANYWHERE, parameters));
        }
        if (!(hasLdEmail && outlookSource.isPresent())) {
            return Cf.list(new DestinedMessage(normalSource, destination, parameters));
        }
        return Cf.list(
                new DestinedMessage(normalSource, MessageDestination.NOT_FOR_EXCHANGE, parameters),
                new DestinedMessage(outlookSource.get(), MessageDestination.ONLY_FOR_EXCHANGE, parameters));
    }

    protected MailMessage prepareMail(DestinedMessage destined, ActionInfo actionInfo) {
        MessageParameters messageParameters = destined.getParameters();

        Source xsl = destined.getXsl() == DestinedMessage.XslSource.OUTLOOK
                ? messageParameters.mailType().getOutlookerXslName().get().getSS()
                : messageParameters.mailType().getXslName().getSS();

        Document outDoc = applyXsl(messageParameters.toOldStyleXml(), xsl);

        MailMessage message = MessageXml.parseMessage(outDoc);
        message = MailHacks.addCalendarHeaders(message, messageParameters.mailType(), actionInfo);

        MessageOverrides overrides = messageParameters.getMessageOverrides();

        message = overrides.messageId.map(message::withMessageId).getOrElse(message::withMessageId);
        message = overrides.subject.map(message::withSubject).getOrElse(message);
        message = overrides.date.map(message::withDate).getOrElse(message);

        if (destined.getDestination() == MessageDestination.MAILLIST_SHARED_FOLDER) {
            message = MailSenderUtils.withMaillistSharedFolderDestination(
                    message, MailSenderUtils.parseMailAddress(message.getHeader(MailHeaders.TO).get()).getEmail());

        } else if (destined.getDestination() == MessageDestination.ONLY_FOR_EXCHANGE) {
            message = message.withTo(new Email(destined.getParameters().getRecipient().login.get() + "@" + ewsDomain));
            message = message.getHeader(MailHeaders.REPLY_TO).flatMapO(MailSenderUtils::parseMailAddressSafe)
                    .map(message::withFrom).getOrElse(message);

            message = message.withHeader(MailHeaders.X_CALENDAR_ONLY_FOR_EXCHANGE, "yes");

        } else if (destined.getDestination() == MessageDestination.NOT_FOR_EXCHANGE) {
            message = message.withHeader(MailHeaders.X_CALENDAR_NOT_FOR_EXCHANGE, "yes");
        }
        return message;
    }

    protected void logSentToEventsLog(
            DestinedMessage destined, MailMessage mail,
            JavaMailSender.Status status, Optional<String> response,
            ActionInfo actionInfo
    ) {
        MessageParameters parameters = destined.getParameters();

        if (parameters instanceof EventMessageParameters) {
            eventsLogger.log(new EventMailLogEventJson(
                    (EventMessageParameters) parameters,
                    MailSenderUtils.parseMailAddress(mail.getHeader(MailHeaders.TO).get()).getEmail(),
                    destined, mail.getMessageId().get(),
                    status.name(), response
            ), actionInfo);

        } else if (parameters instanceof TodoMessageParameters) {
            eventsLogger.log(new TodoMailLogEventJson(
                    (TodoMessageParameters) parameters,
                    mail.getMessageId().get(), status.name(), response
            ), actionInfo);

        } else if (parameters instanceof LayerInvitationMessageParameters) {
            eventsLogger.log(new LayerInvitationMailEventJson(
                    (LayerInvitationMessageParameters) parameters,
                    mail.getMessageId().get(), status.name(), response
            ), actionInfo);
        }
    }

    private Document applyXsl(Element eSrcXml, Source xsl) {
        Document inDoc = toDocument(eSrcXml);
        if (dumpXslInput) {
            String xml = JdomUtils.I.writeDocumentToString(inDoc, XmlWriteFormat.prettyPrintFormat());
            logger.debug("Sender.xslSend(): source xml document is: " + xml);
        }

        Document outDoc = XslUtils.applyXslt(eSrcXml, xsl);
        if (dumpXslOutput) {
            String xml = JdomUtils.I.writeDocumentToString(outDoc, XmlWriteFormat.prettyPrintFormat());
            logger.debug("Sender.xslSend(): xml to be mailed is: " + xml);
        }
        return outDoc;
    }

    private static Document toDocument(Element eSrcXml) {
        Document inDoc = eSrcXml.getDocument();
        if (inDoc == null) {
            inDoc = new Document(eSrcXml);
        }
        return inDoc;
    }
}
