package ru.yandex.calendar.logic.sending.real;

import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.commune.mail.MailAddress;
import ru.yandex.commune.mail.MailMessage;
import ru.yandex.misc.email.Email;

/**
 * @author dbrylev
 */
public class MailSenderUtils {

    public static ListF<Email> parseRecipientEmailsSafe(MailMessage message) {
        return message.getHeader(MailHeaders.TO)
                .plus(message.getHeader(MailHeaders.CC))
                .plus(message.getHeader(MailHeaders.BCC))
                .flatMap(MailSenderUtils::parseEmailsSafe);
    }

    public static ListF<Email> parseSenderEmailsSafe(MailMessage message) {
        return message.getHeader(MailHeaders.FROM)
                .plus(message.getHeader(MailHeaders.REPLY_TO))
                .flatMap(MailSenderUtils::parseEmailsSafe);
    }

    public static Option<MailAddress> parseMailAddressSafe(String address) {
        try {
            return Option.of(parseMailAddress(address));

        } catch (IllegalArgumentException e) {
            return Option.empty();
        }
    }

    public static MailAddress parseMailAddress(String address) {
        try {
            InternetAddress parsed = new InternetAddress(address);
            return new MailAddress(new Email(parsed.getAddress()), parsed.getPersonal());

        } catch (AddressException e) {
            throw new IllegalArgumentException("Invalid address: " + address, e);
        }
    }

    public static boolean isInacceptableAddress(String address) { // CAL-7357
        try {
            return new Email(new InternetAddress(address).getAddress()).getDomain().getDomain().indexOf('.') <= 0;
        } catch (IllegalArgumentException | AddressException e) {
            return true;
        }
    }

    public static boolean isInacceptableAddress(MailAddress address) {
        try {
            address.toInternetAddress().validate();
            return address.getEmail().getDomain().getDomain().indexOf('.') <= 0;

        } catch (AddressException e) {
            return false;
        }
    }

    public static ListF<Email> parseEmailsSafe(String addresses) {
        return parseAddressesSafe(addresses).flatMap(a -> Email.parseSafe(a.getAddress()));
    }

    public static ListF<InternetAddress> parseAddressesSafe(String addresses) {
        try {
            return Cf.x(InternetAddress.parse(addresses));
        } catch (AddressException e) {
            return Cf.list();
        }
    }

    public static ListF<InternetAddress> parseAddressHeaderSafe(MailMessage message, String header) {
        return message.getHeaders().filter(h -> h.nameIs(header)).flatMap(h -> {
            try {
                return Cf.x(InternetAddress.parseHeader(h.getValue(), true));
            } catch (AddressException e) {
                return Cf.list();
            }
        });
    }

    public static MailMessage withMaillistSharedFolderDestination(MailMessage message, Email to) {
        return message
                .withHeader(MailHeaders.TO, to.getLocalPart() + "@mail.yandex-team.ru")
                .withHeader(MailHeaders.BCC, to.getLocalPart() + UserManager.YT_PG_MAILLIST_SUFFIX + "@yandex-team.ru");
    }
}
