package ru.yandex.calendar.logic.sending.real;

import java.io.IOException;
import java.io.StringWriter;

import javax.mail.Message;
import javax.mail.MessagingException;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.commune.mail.BodyPart;
import ru.yandex.commune.mail.ContentType;
import ru.yandex.commune.mail.DefaultContent;
import ru.yandex.commune.mail.MailAddress;
import ru.yandex.commune.mail.MailMessage;
import ru.yandex.commune.mail.MailUtils;
import ru.yandex.commune.mail.Multipart;
import ru.yandex.commune.mail.sendmail.SendmailException;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.email.Emails;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.xml.XmlUtils;

/**
 * @author Stepan Koltsov
 */
public class MessageXml {

    // Aux

    // Element names
    private static final String E_FIELD = "field";
    private static final String E_FROM = "from";
    private static final String E_REPLY_TO = "reply-to";
    private static final String E_HEADER = "header";
    private static final String E_PART = "part";
    private static final String E_SUBJECT = "subject";
    // Attribute names
    private static final String A_ADDRESS = "address";
    private static final String A_CONTENT_TYPE = "content-type";
    private static final String A_NAME__FIELD = "name";
    private static final String A_NAME__PART = "name";
    private static final String A_PERSONAL = "personal";
    private static final String A_SYNTAX_CHECK = "syntax-check";
    private static final String A_TEXT = "text";
    private static final String A_TYPE = "type";
    private static final String A_VALUE = "value";
    // Just literals
    private static final String L_ALTERNATIVE = "alternative";
    private static final String L_MESSAGE = "message";
    private static final String L_MIXED = "mixed";
    private static final String L_RELATED = "related";
    private static final String L_SINGLE = "single";
    private static final String[] KNOWN_MULTIPART_TYPES = { L_ALTERNATIVE, L_MIXED, L_RELATED };

    private static final Message.RecipientType[] RCPT_HEADER_INFOS = {
        Message.RecipientType.TO, Message.RecipientType.CC, Message.RecipientType.BCC
    };

    public static MailMessage parseMessage(Document messageDocument) {
        return parseMessage(messageDocument.getRootElement());
    }

    /**
     * <pre>
     * Example:
     * &lt;br&gt;&lt;message&gt;
     * &lt;br&gt;  &lt;header&gt;
     * &lt;br&gt;   &lt;from address=&quot;from@test.com&quot; syntax-check=&quot;yes&quot; /&gt;
     * &lt;br&gt;   &lt;to address=&quot;to@test.com&quot; /&gt;
     * &lt;br&gt;   &lt;reply-to address=&quot;reply-to@test.com&quot; /&gt;
     * &lt;br&gt;   &lt;subject text=&quot;test subject&quot; /&gt;
     * &lt;br&gt;  &lt;/header&gt;
     * &lt;br&gt;  &lt;part id=&quot;1&quot; content-type=&quot;text/plain; charset=koi8-r&quot;&gt;
     * &lt;br&gt;   Message body
     * &lt;br&gt;  &lt;/part&gt;
     * &lt;br&gt;&lt;/message&gt;
     * </pre>
     */
    private static MailMessage parseMessage(Element eMsg) {
        try {
            if (!L_MESSAGE.equalsIgnoreCase(eMsg.getQualifiedName())) {
                throw new RuntimeException("In the input XML root tag must be 'message'");
            }

            Element eHeader = eMsg.getChild(E_HEADER);
            MailMessage mailMessage = parseHeaderElement(eHeader);

            BodyPart bodyPart = MessageXml.addParts(eMsg);
            mailMessage = new MailMessage(mailMessage.getHeaders().plus(bodyPart.getHeaders()), bodyPart.getContent());

            mailMessage = mailMessage.withDate(new Instant());

            return mailMessage;
        } catch (Exception e) {
            throw MailUtils.translate(e);
        }
    }

    private static MailMessage parseHeaderElement(Element eHeader) {
        MailMessage msg = MailMessage.empty();

        Validate.V.equals("header", eHeader.getName());

        Element eFrom = eHeader.getChild(E_FROM);
        String fromStr = MessageXml.getAndSyntaxCheckIfNeeded(eFrom, A_ADDRESS);
        msg = msg.withFrom(new Email(fromStr), eFrom.getAttributeValue(A_PERSONAL));

        for (Element eReplyTo : Option.ofNullable(eHeader.getChild(E_REPLY_TO))) {
            String emailStr = MessageXml.getAndSyntaxCheckIfNeeded(eReplyTo, A_ADDRESS);
            MailAddress mailAddress = new MailAddress(new Email(emailStr), eReplyTo.getAttributeValue(A_PERSONAL));

            msg = msg.withHeader("Reply-To", mailAddress.getValue());
        }

        // add to, cc and bcc fields
        for (Message.RecipientType rt : RCPT_HEADER_INFOS) {
            String rtStr = rt.toString().toLowerCase(); // ssytnik: WAS: corresponding String[] {"to", "cc", "bcc"};
            for (Object o : eHeader.getChildren(rtStr)) {
                Element eAddress = (Element) o;
                String recStr = MessageXml.getAndSyntaxCheckIfNeeded(eAddress, A_ADDRESS);
                String rcptPersonal = eAddress.getAttributeValue(A_PERSONAL);
                msg = msg.addRecipient(rt, new Email(recStr), rcptPersonal);
            }
        }

        Element eSubject = eHeader.getChild(E_SUBJECT);
        String srcSubjectValue = eSubject.getAttributeValue(A_TEXT);
        if (srcSubjectValue == null) { srcSubjectValue = eSubject.getText(); }
        String subject = srcSubjectValue; // ssytnik: tr.conv, then truncate
        // ssytnik: WAS: msg.setSubject(MimeUtility.encodeText(subject, charset, ENC_QP));
        //          NOW: we prefer that java mail chooses the best way of subject encoding;
        //               we just specify mime charset (given from message xml template)
        msg = msg.withSubject(subject); // unless charset was needed, we could just say: setSubject(subject)

        // <field name="Sender" value="test@test.com" syntax-check="yes"/>
        // <field>Precedence: junk</field>
        for (Object o : eHeader.getChildren(E_FIELD)) {
            Element eField = (Element) o;
            String name = eField.getAttributeValue(A_NAME__FIELD);
            String value;
            if (name != null) {
                value = MessageXml.getAndSyntaxCheckIfNeeded(eField, A_VALUE);
                msg = msg.addHeader(name, value);
            } else {
                value = eField.getText();
                msg = msg.addHeaderLine(value);
            }
        }

        return msg;
    }

    // Adds all parts. Possibly, with nested multiparts:
    // - <message @type={single part type}> <part ... /> </message>
    // - <message @type={multipart type}> <part ... /> [<part ... /> ...] </message>
    // where <part> is:
    // - <part @type={single part type} content-type="...">...</part>
    // - <part @type={multpart type}> <part ... /> [<part ... /> ...] </part>
    // and @type is:
    // - non-specified or "single" for single part;
    // - "mixed", "alternative" or "related" for multipart.
    private static BodyPart addParts(
            Element eParentPart)
            throws MessagingException, IOException, SendmailException
    {
        // Get inner parts, parent part (<message> or <part>) type, checks
        final boolean isMultipart;
        String parentPartType = eParentPart.getAttributeValue(A_TYPE);
        if (StringUtils.isEmpty(parentPartType) || L_SINGLE.equalsIgnoreCase(parentPartType))
            isMultipart = false;
        else if (Cf.x(KNOWN_MULTIPART_TYPES).exists(parentPartType::equalsIgnoreCase))
            isMultipart = true;
        else
            throw new SendmailException("Unknown message or part type: " + parentPartType);

        ListF<?> eInnerParts = Cf.x(eParentPart.getChildren(E_PART));
        int innerPartsCount = eInnerParts.size();
        if (isMultipart) {
            if (innerPartsCount == 0) {
                throw new SendmailException("Multipart tags should contain 1+ inner part(s)");
            }
            Multipart innerMp = Multipart.empty();
            for (Object eInnerPart1 : eInnerParts) {
                Element eInnerPart = (Element) eInnerPart1;
                //MimeBodyPart innerPart = new MimeBodyPart();
                innerMp = innerMp.addPart(addParts(eInnerPart));
                // XXX: filename?
            }
            return BodyPart.create(innerMp, ContentType.valueOf("multipart/" + parentPartType));
        } else {
            boolean isMessageTag = eParentPart.getName().equals(L_MESSAGE); // do we deal with top level (message) tag?
            if ((isMessageTag && innerPartsCount != 1) || (!isMessageTag && innerPartsCount != 0)) {
                throw new SendmailException("Singlepart tags should contain 1 (for <message>) / 0 (for <part>) inner part(s)");
            }
            Element ePart = (innerPartsCount == 0 ? eParentPart : (Element) eInnerParts.get(0));
            String contentType = ePart.getAttributeValue(A_CONTENT_TYPE);
            if (StringUtils.isEmpty(contentType))
                throw new SendmailException("Singlepart tag MUST contain @content-type");

            String bodyText = contentType.startsWith("text/html") ? writeElementContent(ePart) : ePart.getText();
            BodyPart bodyPart = BodyPart.create(new DefaultContent(bodyText), ContentType.valueOf(contentType));
            //BodyPart part = BodyPart.create();
            //parentPart.setHeader(H_CONTENT_TRANSFER_ENCODING, "8bit");
            String partName = ePart.getAttributeValue(A_NAME__PART);
            if (StringUtils.isNotEmpty(partName))
                bodyPart = bodyPart.withAttachmentFileName(partName);
            return bodyPart;
        }
    }

    // XXX add to DomApiUtils
    private static String writeElementContent(Element ePart) {
        StringWriter writer = new StringWriter();
        // ssytnik@: raw format here is important (no space changes):
        // compact format removes spaces between text and tags in xml fields with tags;
        // pretty format adds spaces between 1st letter and other part of nick.
        XMLOutputter xw = new XMLOutputter(Format.getRawFormat());
        try {
            xw.output(ePart.getContent(), writer);
        } catch (Exception e) {
            throw XmlUtils.translate(e);
        }
        return writer.toString();
    }

    // Obtains attribute and performs syntax check on it if needed
    private static String getAndSyntaxCheckIfNeeded(Element e, String attrName) throws SendmailException {
        String attrValue = e.getAttributeValue(attrName);
        if ("yes".equals(e.getAttributeValue(A_SYNTAX_CHECK)) && !checkEmailAddressSyntax(attrValue)) {
            final String msg = "Invalid " + e.getName() + " syntax: " + attrValue;
            throw new RuntimeException(msg + ": " + e.getName());
        }
        return attrValue;
    }

    private static boolean checkEmailAddressSyntax(String address) throws SendmailException {
        if (address != null)
            address = address.trim();
        return Emails.isValid(address);
    }



} //~
