package ru.yandex.calendar.logic.sending.real;

import java.io.IOException;

import io.micrometer.core.instrument.MeterRegistry;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.jdom.DocType;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;
import org.joda.time.DateTimeZone;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.resource.OfficeManager;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.resource.schedule.EventIntervalInfo;
import ru.yandex.calendar.logic.resource.schedule.ResourceScheduleInfo;
import ru.yandex.calendar.logic.resource.schedule.ResourceScheduleManager;
import ru.yandex.calendar.logic.sharing.participant.Participants;
import ru.yandex.calendar.logic.sharing.participant.YandexUserParticipantInfo;
import ru.yandex.calendar.logic.staff.DefaultStaffHttpProvider;
import ru.yandex.calendar.logic.staff.StaffV3;
import ru.yandex.calendar.logic.svc.SvcRoutines;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.HttpClientConfiguration;
import ru.yandex.commune.mail.ContentType;
import ru.yandex.commune.mail.DefaultContent;
import ru.yandex.commune.mail.MailMessage;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox.PassportAuthDomain;
import ru.yandex.inside.utils.AuthenticatingClient;
import ru.yandex.inside.utils.OAuthAuthenticationHandler;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.StringUtils;

@Slf4j
public class ScheduleMailSender {

    @Autowired
    private MailSender sender;
    @Autowired
    private SvcRoutines svcRoutines;
    @Autowired
    private ResourceScheduleManager resourceScheduleManager;
    @Autowired
    protected UserManager userManager;
    @Value("${yt.staff.token:-}")
    private String staffApiToken;
    @Value("${yt.staffV3.url}")
    private String staffUrl;
    @Autowired
    private MeterRegistry meterRegistry;


    public void sendSchedule(ListF<ResourceInfo> resources, ListF<String> scheduleDocuments, Email recipient, LocalDate date){
        for (int i = 0; i < scheduleDocuments.size(); ++i) {

            String content = scheduleDocuments.get(i);

            ResourceInfo resource = resources.get(i);
            MailMessage message = MailMessage.empty()
                    .withSubject(String.format("%s, %s, расписание на %s",
                            resource.getOffice().getName(),
                            resource.getName().orElse("?"),
                            DateTimeFormat.forPattern("dd.MM.yyyy").print(date)))
                    .withFrom(svcRoutines.getCalendarInfoEmail(PassportAuthDomain.YANDEX_TEAM_RU))
                    .withTo(recipient)
                    .withMessageId()
                    .withContent(new DefaultContent(content))
                    .withContentType(ContentType.TEXT_HTML_UTF8);


            log.info("Sending schedule for room with id {} ({}) to {}",
                    resource.getResourceId(),
                    resource.getOffice().getName() + ", " + resource.getName().orElse("?"),
                    recipient.getEmail());

            sender.send(recipient, message);

        }
    }

    public String scheduleHtml(ResourceInfo resource, LocalDate date) {
        Document document = new Document();
        DocType type = new DocType("html", "-//W3C//DTD HTML 4.01//EN",
                "http://www.w3.org/TR/html4/strict.dtd");
        document.setDocType(type);


        DateTimeZone timeZone = OfficeManager.getOfficeTimeZone(resource.getOffice());

        ResourceScheduleInfo schedule = resourceScheduleManager.getResourceScheduleForDayAndResource(
                new PassportUid(1), date, timeZone, resource, ActionInfo.adminManager().getActionSource());


        Element root = new Element("body");
        document.setRootElement(root);

        Element officeNameEl = new Element("h1");
        officeNameEl.addContent(resource.getOffice().getName());
        root.addContent(officeNameEl);

        Element resourceNameEl = new Element("h2");
        resourceNameEl.setAttribute("style", "font-weight:300");
        resourceNameEl.addContent(resource.getName().orElse(""));
        root.addContent(resourceNameEl);

        Element tableEl = new Element("table");
        tableEl.setAttribute("border", "1");
        tableEl.setAttribute("width", "600");
        tableEl.setAttribute("style", "border-collapse:collapse;");
        root.addContent(tableEl);

        Element headerEl = new Element("tr");
        tableEl.addContent(headerEl);
        headerEl.addContent(new Element("th").addContent("Время"));
        headerEl.addContent(new Element("th").addContent("Что"));
        headerEl.addContent(new Element("th").addContent("Кто"));
        headerEl.addContent(new Element("th").addContent("Телефон"));
        headerEl.addContent(new Element("th").addContent("Телеграм"));
        headerEl.addContent(new Element("th").addContent("Подпись"));

        var httpConf = HttpClientConfiguration.fromProperties("yt.center");
        var httpClient = httpConf.consTrustAllClient(meterRegistry, "staff");
        var staffV3 = new StaffV3(staffUrl, new DefaultStaffHttpProvider(staffApiToken,
                new AuthenticatingClient(httpClient, new OAuthAuthenticationHandler(staffApiToken))));

        for (EventIntervalInfo event : schedule.getEvents()) {
            Element eventEl = new Element("tr");
            tableEl.addContent(eventEl);


            eventEl.addContent(new Element("td").addContent(
                    DateTimeFormat.forPattern("HH:mm").print(event.getInterval().getStart().toDateTime(timeZone)) + " - " +
                            DateTimeFormat.forPattern("HH:mm").print(event.getInterval().getEnd().toDateTime(timeZone))
            ));
            eventEl.addContent(new Element("td").addContent(event.getEventWithRelations().getEvent().getName()));

            Option<String> userName = Option.empty();
            Option<Integer> userPhone = Option.empty();
            Option<String> telegramLogin = Option.empty();
            Participants participants = event.getEventWithRelations().getParticipants();
            Option<YandexUserParticipantInfo> userO = participants.getLastYandexUser();
            if (userO.isPresent()) {
                userName = userO.get().getSettings().getUserName();
                Option<String> userLogin = userO.get().getSettings().getUserLogin();
                if (StringUtils.isBlank(userName.getOrElse(""))) {
                    userName = userLogin.orElse(Option.of("?"));
                }
                if (userLogin.isPresent()) {
                    val userDetails = userManager.getYtUserByLogin(userLogin.get());
                    if (userDetails.isPresent()) {
                        userPhone = Option.x(userDetails.get().getInfo().getWorkPhone().map(Long::intValue));
                    }
                    try {
                        telegramLogin = staffV3.getTelegramLogin(userLogin.get());
                    } catch (IOException e) {
                        log.error("Failed to fetch telegram login from staff for username " + userLogin.get(), e);
                    }

                }
            }

            eventEl.addContent(new Element("td").addContent(userName.orElse("")));
            eventEl.addContent(new Element("td").addContent(userPhone.map(Cf.Integer.toStringF()).orElse("")));
            eventEl.addContent(new Element("td").addContent(telegramLogin.map(l -> "@" + l).orElse("")));
            eventEl.addContent(new Element("td").addContent(""));
        }

        Element dateEl = new Element("h3");
        dateEl.setAttribute("style", "font-weight:300");
        dateEl.addContent(date.toString());
        root.addContent(dateEl);


        Format fmt = Format.getPrettyFormat();
        XMLOutputter serializer = new XMLOutputter(fmt);

        return serializer.outputString(document);
    }


}
