package ru.yandex.calendar.logic.sending.real;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.logic.beans.generated.SettingsFields;
import ru.yandex.calendar.logic.domain.DomainManager;
import ru.yandex.calendar.logic.sending.param.NotificationMessageParameters;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.util.base.AuxBase;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.sms.PassportSmsService;
import ru.yandex.misc.TranslitUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * Concrete sms sender
 * @author ssytnik
 * @author akirakozov
 * @author Stepan Koltsov
 */
public class SmsSender {
    private static final Logger logger = LoggerFactory.getLogger(SmsSender.class);

    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private PassportSmsService passportSmsService;
    @Autowired
    private DomainManager domainManager;

    private String smsPassportSender;

    public String sendNotificationSms(NotificationMessageParameters smsNotificationParameters, PassportUid uid) {
        String text = smsNotificationParameters.formatSms();
        return send(text, uid);
    }

    public String send(String text, PassportUid uid) {
        boolean translitSms = settingsRoutines.getSettingsField(uid, SettingsFields.TRANSLIT_SMS);

        if (translitSms) {
            // Apply transliteration for the whole text
            logger.debug("SmsSender.send(): *** Text before transliteration is [" + text + "] ***");
            text = TranslitUtils.translit(text);
            // Post-transliteration corrections
            text = text.replaceAll("Yandeks", "Yandex").replaceAll("Kalendar'", "Calendar");
            text = AuxBase.toSingleQuotedStringExceptQuot(text); // https://jira.yandex-team.ru/browse/CAL-3133
        }

        logger.debug(
            "SmsSender.send(): *** Sending to user [" + uid +
            "] text [" + text + "]; sms length = " + text.length() + " ***"
        );

        String messageId;
        if (uid.isYandexTeamRu()) {
            Option<String> phone = domainManager.getPhoneNumber(uid);
            logger.debug("Yandex-team phone number for uid " + uid + " is: " + phone);
            if (phone.isPresent()) {
                messageId = passportSmsService.sendToPhone(phone.get(), text, "yandex-team");
            } else {
                throw new CommandRunException("User " + uid + " has incorrect phone number");
            }
        } else {
            messageId = passportSmsService.sendToUid(uid.getUid(), text, smsPassportSender);
        }
        return messageId;
    }

    @Required
    @Value("${sms.passport.sender}")
    public void setSmsPassportSender(String smsPassportSender) {
        this.smsPassportSender = smsPassportSender;
    }

} //~
