package ru.yandex.calendar.logic.sending.so;

import java.util.List;
import java.util.Objects;
import java.util.Optional;

import io.micrometer.core.instrument.MeterRegistry;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import one.util.streamex.StreamEx;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.util.EntityUtils;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.forhuman.Comparator;
import ru.yandex.calendar.CalendarRequest;
import ru.yandex.calendar.CalendarRequestHandle;
import ru.yandex.calendar.RemoteInfo;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.frontend.web.cmd.run.Situation;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.model.EventData;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsVTimeZones;
import ru.yandex.calendar.logic.ics.iv5j.ical.component.IcsVEvent;
import ru.yandex.calendar.logic.sending.param.EventMessageInfo;
import ru.yandex.calendar.logic.sending.param.InvitationMessageParameters;
import ru.yandex.calendar.logic.sending.param.LayerInvitationMessageParameters;
import ru.yandex.calendar.logic.sending.param.MessageParameters;
import ru.yandex.calendar.logic.sending.param.Sender;
import ru.yandex.calendar.logic.sharing.MailType;
import ru.yandex.calendar.micro.so.Form;
import ru.yandex.calendar.micro.so.SoCheckClient;
import ru.yandex.calendar.util.email.Emails;
import ru.yandex.calendar.util.exception.ExceptionUtils;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.commune.json.JsonString;
import ru.yandex.commune.mapObject.MapField;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.ip.IpAddress;
import ru.yandex.misc.lang.CharsetUtils;
import ru.yandex.misc.time.TimeUtils;

import static org.joda.time.DateTimeZone.UTC;

@Slf4j
public class SoChecker {
    @Autowired
    private MeterRegistry registry;
    @Autowired
    private EnvironmentType environmentType;

    private static final String SO_CHECKER_METRIC_PREFIX = "application.so-checker.";
    static final String SO_CHECKER_METRIC_SPAM = SO_CHECKER_METRIC_PREFIX + "spam";
    static final String SO_CHECKER_METRIC_HAM = SO_CHECKER_METRIC_PREFIX + "ham";
    static final String SO_CHECKER_METRIC_ERROR = SO_CHECKER_METRIC_PREFIX + "error";

    private final DynamicProperty<Boolean> checkEnabled = new DynamicProperty<>("so.checkSpam",
            Cf.set(EnvironmentType.DEVELOPMENT, EnvironmentType.TESTS, EnvironmentType.PRODUCTION).containsTs(EnvironmentType.getActive()));
    private final DynamicProperty<Boolean> checkFormEnabled = new DynamicProperty<>("so.checkForm", EnvironmentType.TESTS.equals(EnvironmentType.getActive()));

    private final String url;
    private final Timeout timeout;
    private SoCheckClient soCheckClient;

    public SoChecker(String url, Timeout timeout, SoCheckClient soCheckClient) {
        this.url = url;
        this.timeout = timeout;
        this.soCheckClient = soCheckClient;
    }

    /**
     * @deprecated Will be removed after completion of integration with check-form interface  ({@link SoChecker#checkEventInternal}).
     */
    @Deprecated
    public void checkNoSpam(ListF<MessageParameters> eventMails, ActionInfo actionInfo) {
        if (url.isEmpty()) {
            return;
        }
        long start = System.currentTimeMillis();

        MapF<Tuple2<Option<PassportUid>, Long>, ListF<InvitationMessageParameters>> groups;

        ListF<InvitationMessageParameters> mails = eventMails.filterByType(InvitationMessageParameters.class);
        mails = mails.iterator()
                .filterNot(m -> m.getRecipientUid().exists(m.getSender().getUid()::isSome))
                .filter(m -> m.getSender().getUid().isPresent())
                .filter(m -> m.mailType() == MailType.EVENT_UPDATE || m.mailType() == MailType.EVENT_INVITATION)
                .toList();
        groups = mails
                .groupBy(p -> Tuple2.tuple(p.getSender().getUid(), p.getMainEventId()));
        try {
            groups.values().forEach(ms -> {
                InvitationMessageParameters mail = ms.max(Comparator.<InvitationMessageParameters>constEqualComparator()
                        .thenComparing(m -> m.mailType() == MailType.EVENT_UPDATE)
                        .thenComparing(m -> !m.getOccurrenceId().isPresent()));

                Sender sender = mail.getSender();
                EventMessageInfo eventInfo = mail.getEventMessageInfo();
                SoRequestData.User user = new SoRequestData.User(sender);
                ListF<Email> recipients = ms.map(InvitationMessageParameters::getRecipientEmail).stableUnique();
                SoRequestData.Extended extended = new SoRequestData.Extended(recipients.size(),
                        actionInfo.getActionSource());
                checkNoSpam(sender, recipients, Option.of(eventInfo),
                        new SoRequestData(mail.mailType(), remoteInfo(), user, eventInfo, extended), actionInfo);
            });
        } finally {
            log.debug("checkNoSpam; rc={}; took {}", groups.values().size(), TimeUtils.millisecondsToSecondsString(System.currentTimeMillis() - start));
        }
    }

    @Deprecated
    public void checkNoSpamLayers(ListF<LayerInvitationMessageParameters> mails, ActionInfo actionInfo) {
        if (url.isEmpty()) return;

        mails.groupBy(p -> Tuple2.tuple(p.getSender().getUid(), p.getLayerId())).values().forEach(ms -> {
            Sender sender = ms.first().getSender();

            ListF<Email> recipients = ms.map(MessageParameters::getRecipientEmail);

            SoRequestData data = new SoRequestData(
                    MailType.LAYER_INVITATION, remoteInfo(), new SoRequestData.User(sender),
                    ms.first().getLayerName(), "", "",
                    new SoRequestData.Extended(ms.size(), actionInfo.getActionSource()));

            checkNoSpam(sender, recipients, Option.empty(), data, actionInfo);
        });
    }

    @Deprecated
    public void checkNoSpam(
            Sender sender, ListF<Email> recipients, Option<EventMessageInfo> eventInfo,
            SoRequestData data, ActionInfo actionInfo
    ) {
        if (url.isEmpty() || !checkEnabled.get())
            return;

        val to = recipients.map(Emails::getUnicoded);

        val parameters = Tuple2List.<String, Object>tuple2List()
                .plus1("add_headers", 0)
                .plus1("uid", data.user.uid.getUid())
                .plus1("from", Emails.getUnicoded(sender.getEmail()))
                .plus1("to", StreamEx.of(to).limit(50).joining(";"))
                .plus1("client_ip", data.ip.getOrElse(""))
                .plus1("request_id", actionInfo.getRequestIdWithHostId())
                .plus(eventInfo.flatMap(ei -> Tuple2List.fromPairs(
                        "event_id", ei.getEventId(),
                        "is_repeating", ei.getIsRepeating() ? "1" : "0")
                ))
                .plus1("source", "calendar");
        val request = new HttpPost(UrlUtils.addParameters(url, parameters));
        request.setEntity(new StringEntity("From: " + Emails.getUnicoded(sender.getEmail()) + "\r\n"
                + "To: " + String.join(",", to) + "\r\n"
                + "Subject: " + data.subject + "\r\n\n"
                + data.location + "\r\n"
                + data.description, CharsetUtils.UTF8_CHARSET));

        boolean spam;

        try {
            spam = ApacheHttpClientUtils.execute(request, response -> {
                int code = response.getStatusLine().getStatusCode();

                if (!Cf.list(200).containsTs(code)) {
                    throw new RuntimeException(code + " response: " + EntityUtils.toString(response.getEntity()));
                }
                return JsonObject.parseObject(EntityUtils.toString(response.getEntity()))
                        .getO("resolution").filterByType(JsonString.class).exists(s -> s.getString().equals("SPAM"));

            }, timeout, url.startsWith("https"));

        } catch (RuntimeException e) {
            log.error("Failed to check spam: {}", ExceptionUtils.getAllMessages(e));
            spam = false;
        }
        log.debug("Spam resolution = " + spam);
        if (spam) {
            throw CommandRunException.createSituation("Spam detected from " + sender.getEmail(), Situation.SPAM_DETECTED);
        }
    }

    private static RemoteInfo remoteInfo() {
        return CalendarRequest.getCurrentO().map(CalendarRequestHandle::getRemoteInfo)
                .getOrElse(new RemoteInfo(Option.empty(), Option.empty()));
    }

    public void validateEvent(PassportUid uid, EventData eventData, ActionInfo actionInfo) {
        val form = constructForm(uid, eventData, actionInfo);
        if (checkEventInternal(form, actionInfo)) {
            throw new SoCheckFailedException("Event is spam");
        }
    }

    public boolean isEventSpam(PassportUid uid, IcsVEvent event, ActionInfo actionInfo, Optional<String> url) {
        val form = constructForm(uid, event, actionInfo, url);
        return checkEventInternal(form, actionInfo);
    }

    @SneakyThrows
    private boolean checkEventInternal(Form form, ActionInfo actionInfo) {
        if (!checkFormEnabled.get()) {
            log.info("Spam check is disabled.");
            return false;
        }
        log.info("Spam check formID = {} uid = {}", form.getFormFields().getId(), form.getFormAuthor());

        long start = System.currentTimeMillis();
        try {
            val spam = soCheckClient.checkForm("CALENDAR", actionInfo.getRequestIdWithHostId(), form.getFormFields().getId().getValue().toString(), form).get();
            log.info("Check event for spam was success. Spam resolution is = {}", spam);
            registry.counter(spam ? SO_CHECKER_METRIC_SPAM : SO_CHECKER_METRIC_HAM).increment();
            return spam;
        } catch (Exception e) {
            log.error("Error when event check for spam request", e);
            registry.counter(SO_CHECKER_METRIC_ERROR).increment();
            return false;
        } finally {
            log.debug("checkEventInternalSpam; took {}", TimeUtils.millisecondsToSecondsString(System.currentTimeMillis() - start));
        }
    }

    Form constructForm(PassportUid uid, EventData eventData, ActionInfo actionInfo) {
        val event = eventData.getEvent();
        val formFields = new Form.FormFields(
                getFieldValue(event, EventFields.LOCATION),
                getFieldValue(event, EventFields.DESCRIPTION),
                getFieldValue(event, EventFields.NAME),
                getFieldValue(event, EventFields.ID),
                getFieldValue(event, EventFields.START_TS),
                getFieldValue(event, EventFields.END_TS),
                unwrapEmails(eventData.getParticipantEmails()));
        return constructForm(
                environmentType,
                actionInfo.getActionSource(),
                actionInfo.getAction(),
                event.getFieldValueO(EventFields.NAME).toOptional(),
                uid,
                event.getFieldValueO(EventFields.URL).toOptional(),
                formFields);
    }

    private static String getFieldValue(Event event, MapField<?> eventField) {
        return event.getFieldValueO(eventField).map(Objects::toString).toOptional().orElse("");
    }

    Form constructForm(PassportUid uid, IcsVEvent event, ActionInfo actionInfo, Optional<String> url) {
        val formFields = new Form.FormFields(
            event.getLocation().orElse(""),
            event.getDescription().orElse(""),
            event.getSummary().orElse(""),
            event.getUid().toOptional().orElse(""),
            Optional.of(event.getStart().getInstant(IcsVTimeZones.fallback(UTC)).toString()).orElse(""),
            Optional.of(event.getEnd().getInstant(IcsVTimeZones.fallback(UTC)).toString()).orElse(""),
            unwrapEmails(event.getParticipantEmailsSafe()));
        return constructForm(
                environmentType,
                actionInfo.getActionSource(),
                actionInfo.getAction(),
                event.getSummary(),
                uid,
                url,
                formFields);
    }

    static Form constructForm(EnvironmentType environment, ActionSource actionSource, String action, Optional<String> subject, PassportUid formAuthor, Optional<String> formRealpath, Form.FormFields formFields) {
        val clientIp = CalendarRequest.getCurrentO().map(CalendarRequestHandle::getRemoteInfo)
                .getOrElse(new RemoteInfo(Option.empty(), Option.empty()))
                .ip.map(IpAddress::format).getOrElse("");
        return new Form(environment.name(), actionSource.name(), action, clientIp, subject.orElse(""), formAuthor.getUid(), formRealpath.orElse("unknown"), formFields);
    }

    private List<String> unwrapEmails(List<Email> emailList) {
        return StreamEx.of(emailList).map(Email::getEmail).toImmutableList();
    }
}
