package ru.yandex.calendar.logic.sharing;

import lombok.Getter;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.sharing.participant.ParticipantBasics;
import ru.yandex.calendar.logic.sharing.participant.ParticipantData;
import ru.yandex.calendar.logic.sharing.participant.ParticipantId;
import ru.yandex.calendar.logic.sharing.participant.ParticipantInfo;
import ru.yandex.inside.passport.PassportUid;

/**
 * Stores information about invitation changes
 */
@Getter
public class EventParticipantsChangesInfo {
    private final Tuple2List<ParticipantId, ParticipantData> newParticipants;
    private final Tuple2List<ParticipantId, ParticipantChangesInfo> updatedInvitations;
    private final ListF<ParticipantInfo> removedParticipants;

    private final Option<ParticipantId> newOrganizer;
    private final ListF<ParticipantId> oldOrganizers;

    private final boolean changedToNotMeeting;

    public static final EventParticipantsChangesInfo EMPTY =
            new EventParticipantsChangesInfo(Tuple2List.tuple2List(), Tuple2List.tuple2List(), Cf.list(), false);

    private EventParticipantsChangesInfo(
            Tuple2List<ParticipantId, ParticipantData> newParticipants,
            Tuple2List<ParticipantId, ParticipantChangesInfo> updatedInvitations,
            ListF<ParticipantInfo> removedParticipants,
            boolean changedToNotMeeting)
    {
        this.newParticipants = newParticipants;
        this.updatedInvitations = updatedInvitations;
        this.removedParticipants = removedParticipants;
        this.changedToNotMeeting = changedToNotMeeting;

        ListF<ParticipantId> newParticipantIds = newParticipants.get1();
        ListF<ParticipantId> updatedParticipantIds = updatedInvitations.get1();
        ListF<ParticipantId> removedParticipantIds = removedParticipants.map(ParticipantInfo.getIdF());
        ListF<ParticipantId> affectedParticipantIds = newParticipantIds.plus(updatedParticipantIds).plus(removedParticipantIds);
        if (affectedParticipantIds.unique().size() != affectedParticipantIds.size()) {
            throw new IllegalArgumentException("participant ids intersect: new: " + newParticipantIds + ", updated: " + updatedParticipantIds + ", removed: " + removedParticipantIds);
        }
        this.newOrganizer = newParticipants.filterBy2(ParticipantData.isOrganizerF()).get1().singleO()
                .orElse(updatedInvitations.filterBy2(ParticipantChangesInfo.becomesOrganizerF()).get1().singleO());

        this.oldOrganizers = removedParticipants.filter(ParticipantInfo.isOrganizerF()).map(ParticipantInfo.getIdF())
                .plus(updatedInvitations.filterBy2(ParticipantChangesInfo.becomesNotOrganizerF()).get1());
    }

    public static EventParticipantsChangesInfo changes(
            Tuple2List<ParticipantId, ParticipantData> newParticipants,
            Tuple2List<ParticipantId, ParticipantChangesInfo> updatedInvitations,
            ListF<ParticipantInfo> removedParticipants)
    {
        return new EventParticipantsChangesInfo(newParticipants, updatedInvitations, removedParticipants, false);
    }

    public EventParticipantsChangesInfo withOnlySubjectMightBeRemoved(UidOrResourceId subjectId) {
        ParticipantId participantId = subjectId.toParticipantId();

        ListF<ParticipantInfo> onlyMaybeSubjectRemoved =
                removedParticipants.filter(ParticipantInfo.getIdF().andThenEquals(participantId));

        return new EventParticipantsChangesInfo(newParticipants, updatedInvitations, onlyMaybeSubjectRemoved,
                changedToNotMeeting);
    }

    public EventParticipantsChangesInfo withoutRemovedParticipants() {
        return new EventParticipantsChangesInfo(newParticipants, updatedInvitations, Cf.list(), false);
    }

    public boolean participantsRemoved() {
        return !removedParticipants.isEmpty();
    }

    public EventParticipantsChangesInfo withParticipantRemoved(ParticipantInfo participant) {
        return new EventParticipantsChangesInfo(
                newParticipants.filterBy1(ParticipantId.sameAsF2().bind2(participant.getId())),
                updatedInvitations.filterBy1(ParticipantId.sameAsF2().bind2(participant.getId())),
                removedParticipants.plus1(participant).stableUniqueBy(ParticipantInfo.getIdF()),
                changedToNotMeeting);
    }

    public EventParticipantsChangesInfo exceptSubject(UidOrResourceId subjectId) {
        ParticipantId participantId = subjectId.toParticipantId();

        return new EventParticipantsChangesInfo(
                newParticipants.filterBy1(participantId.sameAsF().notF()),
                updatedInvitations.filterBy1(participantId.sameAsF().notF()),
                removedParticipants.filter(ParticipantInfo.getIdF().andThenEquals(participantId).notF()),
                changedToNotMeeting
        );
    }

    public EventParticipantsChangesInfo applyParticipantDecision(UidOrResourceId subjectId, Decision decision) {
        ParticipantId participantId = subjectId.toParticipantId();

        Tuple2List<ParticipantId, ParticipantData> added = Tuple2List.arrayList();
        Tuple2List<ParticipantId, ParticipantChangesInfo> updated = Tuple2List.arrayList();

        for (Tuple2<ParticipantId, ParticipantData> n : newParticipants) {
            added.add(n._1.sameAs(participantId) ? Tuple2.tuple(n._1, n._2.withDecision(decision)) : n);
        }
        for (Tuple2<ParticipantId, ParticipantChangesInfo> u : updatedInvitations) {
            updated.add(u._1.sameAs(participantId) ? Tuple2.tuple(u._1, u._2.withDecision(decision)) : u);
        }
        return new EventParticipantsChangesInfo(added, updated, removedParticipants, changedToNotMeeting);
    }

    public static EventParticipantsChangesInfo changedToNotMeeting(ListF<ParticipantInfo> removedParticipants) {
        return new EventParticipantsChangesInfo(
                Tuple2List.tuple2List(), Tuple2List.tuple2List(),
                removedParticipants, true);
    }

    public ListF<ParticipantData> getNewParticipantsExclOrganizer() {
        return getNewParticipants().get2().filter(ParticipantData.isOrganizerF().notF());
    }

    public ListF<ParticipantId> getNewAndRemovedParticipantIds() {
        return newParticipants.get1().plus(removedParticipants.map(ParticipantInfo.getIdF()));
    }

    public ListF<Long> getNewAndRemovedResources() {
        return getNewAndRemovedParticipantIds().filterMap(ParticipantId.getResourceIdIfResourceF());
    }

    public ListF<Long> getNewResources() {
        return getNewParticipants().filterMap(p -> p.get1().getResourceIdIfResource());
    }

    public ListF<PassportUid> getNewAndRemovedUids() {
        return getNewAndRemovedParticipantIds().filterMap(ParticipantId.getUidIfYandexUserF());
    }

    public ListF<ParticipantInfo> getRemovedOrganizers() {
        return getRemovedParticipants().filter(ParticipantInfo.isOrganizerF());
    }

    public ListF<ParticipantInfo> getRemovedParticipantsButNotOrganizer() {
        return getRemovedParticipants().filter(ParticipantInfo.isOrganizerF().notF());
    }

    public boolean wasChange() {
        return newParticipants.isNotEmpty() || updatedInvitations.isNotEmpty() ||
            removedParticipants.isNotEmpty();
    }

    public boolean wasOrganizerChange() {
        return newOrganizer.isPresent() || oldOrganizers.isNotEmpty();
    }

    public boolean wasNewParticipantWithUid(PassportUid uid) {
        return newParticipants.get1().exists(ParticipantId.isYandexUserWithUidF(uid));
    }

    public boolean wasResourcesChange() {
        return newParticipants.get1().plus(updatedInvitations.get1()).plus(removedParticipants.map(ParticipantInfo.getIdF()))
                .exists(ParticipantId.isResourceF());
    }

    public boolean wasOnlyUserAttendeesChange() {
        return wasChange() && !wasResourcesChange() && !wasOrganizerChange();
    }

    public boolean wasOnlyAttendeesChange() {
        return wasChange() && !wasOrganizerChange();
    }

    public boolean wasParticipantsOrDecisionChange() {
        return newParticipants.isNotEmpty() || updatedInvitations.filterBy2(ParticipantChangesInfo.decisionChangedF()).isNotEmpty() ||
            removedParticipants.isNotEmpty();
    }

    public boolean wasParticipantsChange() {
        return newParticipants.isNotEmpty() || removedParticipants.isNotEmpty();
    }

    public boolean wasNewOrRemovedResources() {
        return wasNewResources() || removedParticipants.exists(ParticipantInfo.isResourceF());
    }

    public boolean wasNewResources() {
        return newParticipants.findBy1(ParticipantId.isResourceF()).isPresent();
    }

    public Option<ParticipantId> getNewOrganizerNotJustInvited() {
        return getNewOrganizer().filter(newParticipants.get1().containsF().notF());
    }

    public ListF<ParticipantId> getOldOrganizersNotRemoved() {
        return getOldOrganizers().filter(removedParticipants.map(ParticipantInfo.getIdF()).unique().containsF().notF());
    }

    public Option<ParticipantBasics> getNewOrganizerParticipant() {
        return newOrganizer.flatMapO(id -> Option.<ParticipantBasics>empty()
                .orElse(newParticipants.findBy1(id::equals).map(Tuple2::get2))
                .orElse(updatedInvitations.findBy1(id::equals).map(t -> t.get2().getOldParticipantInfo())));
    }
}
