package ru.yandex.calendar.logic.sharing;

import net.fortuna.ical4j.model.property.Method;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsMethod;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.PropertiesMeta;
import ru.yandex.calendar.logic.resource.ResourceType;
import ru.yandex.calendar.logic.sending.param.ApartmentMessageParameters;
import ru.yandex.calendar.logic.sending.param.BrokenDisplayReportMessageParameters;
import ru.yandex.calendar.logic.sending.param.CancelEventMessageParameters;
import ru.yandex.calendar.logic.sending.param.EventOnLayerChangeMessageParameters;
import ru.yandex.calendar.logic.sending.param.EventOnResourceChangeMessageParameters;
import ru.yandex.calendar.logic.sending.param.InvitationMessageParameters;
import ru.yandex.calendar.logic.sending.param.LayerInvitationMessageParameters;
import ru.yandex.calendar.logic.sending.param.MessageParameters;
import ru.yandex.calendar.logic.sending.param.NotificationMessageParameters;
import ru.yandex.calendar.logic.sending.param.RepetitionConfirmationMessageParameters;
import ru.yandex.calendar.logic.sending.param.ReplyMessageParameters;
import ru.yandex.calendar.logic.sending.param.TodoListMessageParameters;
import ru.yandex.calendar.logic.sending.param.TodoMessageParameters;
import ru.yandex.calendar.util.resources.CalendarClassPathResourceUtils.ResourceName;
import ru.yandex.misc.enums.EnumResolver;
import ru.yandex.misc.reflection.ClassX;

/**
 * @author akirakozov
 * @author shinderuk
 */
public enum MailType {
    EVENT_CANCEL(ResourceName.EMAIL_EVENT_CANCEL_XSL, CancelEventMessageParameters.class, Method.CANCEL),
    EVENT_REORGANIZATION(ResourceName.EMAIL_EVENT_REORGANIZATION, InvitationMessageParameters.class, Method.CANCEL),
    EVENT_INVITATION(ResourceName.EMAIL_EVENT_INVITATION_XSL, InvitationMessageParameters.class, Method.REQUEST),
    EVENT_NOTIFICATION(ResourceName.EMAIL_EVENT_NOTIFICATION_XSL, NotificationMessageParameters.class),
    EVENT_REPLY(ResourceName.EMAIL_EVENT_REPLY_XSL, ReplyMessageParameters.class, Method.REPLY),
    EVENT_UPDATE(ResourceName.EMAIL_EVENT_UPDATE_XSL, InvitationMessageParameters.class, Method.REQUEST),
    RESOURCE_UNCHECKIN(ResourceName.EMAIL_EVENT_REPLY_XSL, ReplyMessageParameters.class, Method.REPLY),
    LAYER_INVITATION(ResourceName.EMAIL_LAYER_INVITATION_XSL, LayerInvitationMessageParameters.class),
    TODO_LIST(ResourceName.EMAIL_TODO_LIST_XSL, TodoListMessageParameters.class),
    TODO_NOTIFICATION(ResourceName.EMAIL_TODO_NOTIFICATION_XSL, TodoMessageParameters.class),
    REPETITION_CONFIRMATION(ResourceName.EMAIL_REPETITION_CONFIRMATION_XSL, RepetitionConfirmationMessageParameters.class),
    APARTMENT_INVITATION(ResourceName.EMAIL_APARTMENT_INVITATION_XSL, ApartmentMessageParameters.class),
    APARTMENT_UPDATE(ResourceName.EMAIL_APARTMENT_UPDATE_XSL, ApartmentMessageParameters.class),
    APARTMENT_CANCEL(ResourceName.EMAIL_APARTMENT_CANCEL_XSL, ApartmentMessageParameters.class),
    CAMPUS_INVITATION(ResourceName.EMAIL_CAMPUS_XSL, ApartmentMessageParameters.class),
    CAMPUS_UPDATE(ResourceName.EMAIL_CAMPUS_XSL, ApartmentMessageParameters.class),
    CAMPUS_CANCEL(ResourceName.EMAIL_CAMPUS_XSL, ApartmentMessageParameters.class),
    HOTEL_INVITATION(ResourceName.EMAIL_HOTEL_XSL, ApartmentMessageParameters.class),
    HOTEL_UPDATE(ResourceName.EMAIL_HOTEL_XSL, ApartmentMessageParameters.class),
    HOTEL_CANCEL(ResourceName.EMAIL_HOTEL_XSL, ApartmentMessageParameters.class),
    EVENT_RESOURCE_ADDED(ResourceName.EMAIL_EVENT_RESOURCE_ADDED_XSL, EventOnResourceChangeMessageParameters.class),
    EVENT_RESOURCE_UPDATED(ResourceName.EMAIL_EVENT_RESOURCE_UPDATED_XSL, EventOnResourceChangeMessageParameters.class),
    EVENT_RESOURCE_CANCELLED(ResourceName.EMAIL_EVENT_RESOURCE_REMOVED_XSL, EventOnResourceChangeMessageParameters.class),
    EVENT_ON_LAYER_ADDED(ResourceName.EMAIL_EVENT_LAYER_ADDED_XSL, EventOnLayerChangeMessageParameters.class),
    EVENT_ON_LAYER_UPDATED(ResourceName.EMAIL_EVENT_LAYER_UPDATED_XSL, EventOnLayerChangeMessageParameters.class),
    EVENT_ON_LAYER_REMOVED(ResourceName.EMAIL_EVENT_LAYER_REMOVED_XSL, EventOnLayerChangeMessageParameters.class),
    BROKEN_DISPLAY_REPORT(ResourceName.EMAIL_BROKEN_DISPLAY_XSL, BrokenDisplayReportMessageParameters.class),
    ;

    public static final EnumResolver<MailType> R = EnumResolver.er(MailType.class);

    private final ResourceName xslName;
    private final Option<Method> method;
    private final ClassX<? extends MessageParameters> dataClass;

    private MailType(ResourceName xslName, Class<? extends MessageParameters> dataClass) {
        this.xslName = xslName;
        this.method = Option.empty();
        this.dataClass = ClassX.wrap(dataClass);
    }

    private MailType(ResourceName xslName, Class<? extends MessageParameters> dataClass, Method method) {
        this.xslName = xslName;
        this.method = Option.of(method);
        this.dataClass = ClassX.wrap(dataClass);
    }

    public ResourceName getXslName() {
        return xslName;
    }

    public Option<ResourceName> getOutlookerXslName() {
        switch (this) {
            case EVENT_CANCEL:
            case EVENT_REORGANIZATION:
                return Option.of(ResourceName.EMAIL_FOR_OUTLOOKER_CANCEL_XSL);
            case EVENT_REPLY:
                return Option.of(ResourceName.EMAIL_FOR_OUTLOOKER_REPLY_XSL);
            case EVENT_INVITATION:
            case EVENT_UPDATE:
                return Option.of(ResourceName.EMAIL_FOR_OUTLOOKER_DEFAULT_XSL);
            default:
                return Option.empty();
        }
    }

    public Option<Method> getMethod() {
        return this.method;
    }

    public Option<IcsMethod> getIcsMethod() {
        return this.method.map(PropertiesMeta.M.fromIcal4jF()).uncheckedCast();
    }

    public ClassX<? extends MessageParameters> getDataClass() {
        return dataClass;
    }

    public boolean in(MailType... values) {
        return Cf.x(values).containsTs(this);
    }

    public boolean isCancel() {
        return this == EVENT_CANCEL;
    }

    public boolean isResourceDecline() {
        return this == RESOURCE_UNCHECKIN;
    }

    public MailType forResource(ResourceType type) {
        if (type == ResourceType.APARTMENT) {
            switch (this) {
                case EVENT_INVITATION:
                case APARTMENT_INVITATION:
                    return APARTMENT_INVITATION;
                case EVENT_UPDATE:
                case APARTMENT_UPDATE:
                    return APARTMENT_UPDATE;
                case APARTMENT_CANCEL:
                case EVENT_CANCEL:
                    return APARTMENT_CANCEL;
            }
            throw new IllegalStateException("Unexpected type " + this + " for resource " + type);
        }
        if (type == ResourceType.CAMPUS) {
            switch (this) {
                case EVENT_INVITATION:
                case CAMPUS_INVITATION:
                    return CAMPUS_INVITATION;
                case EVENT_UPDATE:
                case CAMPUS_UPDATE:
                    return CAMPUS_UPDATE;
                case EVENT_CANCEL:
                case CAMPUS_CANCEL:
                    return CAMPUS_CANCEL;
            }
            throw new IllegalStateException("Unexpected type " + this + " for resource " + type);
        }
        if (type == ResourceType.HOTEL) {
            switch (this) {
                case EVENT_INVITATION:
                case HOTEL_INVITATION:
                    return HOTEL_INVITATION;
                case EVENT_UPDATE:
                case HOTEL_UPDATE:
                    return HOTEL_UPDATE;
                case EVENT_CANCEL:
                case HOTEL_CANCEL:
                    return HOTEL_CANCEL;
            }
            throw new IllegalStateException("Unexpected type " + this + " for resource " + type);
        }
        switch (this) {
            case EVENT_INVITATION:
            case EVENT_RESOURCE_ADDED:
                return EVENT_RESOURCE_ADDED;
            case EVENT_UPDATE:
            case EVENT_RESOURCE_UPDATED:
                return EVENT_RESOURCE_UPDATED;
            case EVENT_CANCEL:
            case EVENT_RESOURCE_CANCELLED:
                return EVENT_RESOURCE_CANCELLED;
        }
        throw new IllegalStateException("Unexpected type " + this + " for resource " + type);
    }
}
