package ru.yandex.calendar.logic.sharing.participant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.logic.beans.generated.EventInvitation;
import ru.yandex.calendar.logic.beans.generated.EventResource;
import ru.yandex.calendar.logic.event.EventUserWithRelations;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.sharing.perm.ResourceInfoForPermsCheck;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.inside.passport.PassportUid;

/**
 * @author dbrylev
 */
public class EventParticipants {
    private final long eventId;
    private final MapF<PassportUid, EventUserWithRelations> eventUserByUid;
    private final ListF<EventInvitation> invitations;
    private final ListF<ResourceInfo> resources;
    private final ListF<EventResource> eventResources;

    public EventParticipants(
            long eventId,
            ListF<EventUserWithRelations> eventUsers,
            ListF<EventInvitation> invitations,
            ListF<ResourceInfo> resources,
            ListF<EventResource> eventResources)
    {
        this.eventId = eventId;
        this.eventUserByUid = Cf2.stableMapToKey(
                eventUsers.sortedBy(eu -> eu.getEventUser().getId()), EventUserWithRelations::getUid);
        this.invitations = invitations.sortedBy(EventInvitation::getCreationTs);
        this.resources = resources;
        this.eventResources = eventResources;
    }

    private MapF<Long, ResourceInfo> resourceById;

    public MapF<Long, ResourceInfo> getResourceByIdMap() {
        if (resourceById == null) {
            resourceById = resources.toMapMappingToKey(ResourceInfo.resourceIdF());
        }
        return resourceById;
    }

    private Participants participants = null;

    public Participants getParticipants() {
        if (participants == null) {
            ListF<ResourceParticipantInfo> resources = eventResources.map(
                    r -> new ResourceParticipantInfo(getResourceByIdMap().getOrThrow(r.getResourceId()), r));

            ListF<ExternalUserParticipantInfo> invitations = this.invitations.map(ExternalUserParticipantInfo.consF());

            ListF<YandexUserParticipantInfo> users = eventUserByUid.values()
                    .filter(EventUserWithRelations::isAttendeeOrOrganizerOrSubscriber)
                    .map(e -> new YandexUserParticipantInfo(e.getEventUser(), e.getSettings()));

            ListF<ParticipantInfo> result = Cf.arrayListWithCapacity(
                    resources.size() + invitations.size() + users.size());

            result.addAll(resources);
            result.addAll(users);
            result.addAll(invitations);

            participants = Participants.maybeMeetingWithSubscribers(result);
        }
        return participants;
    }

    public ListF<ResourceInfoForPermsCheck> getResourcesForPermsCheck() {
        return getResources().map(Cf2.f(ResourceInfoForPermsCheck::fromResource).compose(ResourceInfo.resourceF()));
    }

    public Option<ParticipantId> getOrganizerIdWithInconsistent() {
        return getParticipants().getOrganizerIdWithInconsistent().find(Function1B.trueF());
    }

    public Option<ParticipantInfo> getOrganizerWithInconsistent() {
        Option<ParticipantId> id = getOrganizerIdWithInconsistent();
        return id.isPresent() ? getParticipants().getByIdSafeWithInconsistent(id.get()) : Option.empty();
    }

    public boolean userIsAttendeeWithInconsistent(PassportUid uid) {
        return eventUserByUid.getO(uid).exists(eu -> eu.getEventUser().getIsAttendee() || eu.getEventUser().getIsOrganizer());
    }

    public EventParticipants withoutResources(ListF<Long> resourceIds) {
        if (resourceIds.isEmpty()) return this;

        return new EventParticipants(
                eventId, eventUserByUid.values().toList(), invitations,
                resources.filterNot(r -> resourceIds.containsTs(r.getResourceId())),
                eventResources.filterNot(r -> resourceIds.containsTs(r.getResourceId())));
    }

    public long getEventId() {
        return eventId;
    }

    public Option<EventUserWithRelations> getEventUser(PassportUid uid) {
        return eventUserByUid.getO(uid);
    }

    public CollectionF<EventUserWithRelations> getEventUsers() {
        return eventUserByUid.values();
    }

    public ListF<EventInvitation> getInvitations() {
        return invitations;
    }

    public ListF<ResourceInfo> getResources() {
        return resources;
    }

    public ListF<EventResource> getEventResources() {
        return eventResources;
    }
}
