package ru.yandex.calendar.logic.sharing.participant;

import lombok.Getter;

import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.logic.beans.generated.EventInvitation;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;

@Getter
public class ParticipantData extends DefaultObject implements ParticipantBasics {
    private final Email email;
    private final String name;
    private final Decision decision;
    private final boolean attendee;
    private final boolean organizer;
    private final boolean isOptional;

    public ParticipantData(Email email, String name, Decision decision, boolean attendee, boolean organizer, boolean isOptional) {
        Validate.isTrue(attendee || organizer);

        this.email = email;
        this.name = StringUtils.defaultIfEmpty(name, "");
        this.decision = decision;
        this.attendee = attendee;
        this.organizer = organizer;
        this.isOptional = isOptional;
    }

    public ParticipantData withAttendee(boolean attendee) {
        return new ParticipantData(email, name, decision, attendee, organizer, isOptional);
    }

    public ParticipantData withOptional(boolean isOptional) {
        return new ParticipantData(email, name, decision, attendee, organizer, isOptional);
    }

    public ParticipantData withAttendeeAndOrganizer(boolean attendee, boolean organizer) {
        return new ParticipantData(email, name, decision, attendee, organizer, isOptional);
    }

    public ParticipantData withDecision(Decision decision) {
        return new ParticipantData(email, name, decision, attendee, organizer, isOptional);
    }

    public EventInvitation toInvitation() {
        EventInvitation invitation = new EventInvitation();
        invitation.setEmail(email.normalize());
        invitation.setName(name);
        invitation.setDecision(decision);
        invitation.setIsOrganizer(organizer);
        return invitation;
    }

    public static Function1B<ParticipantData> isOrganizerF() {
        return a -> a.organizer;
    }

    public static Function1B<ParticipantData> isAttendeeF() {
        return a -> a.attendee;
    }

    public static Function<ParticipantData, Email> getEmailF() {
        return p -> p.email;
    }

    public static Function<ParticipantData, Decision> getDecisionF() {
        return ParticipantData::getDecision;
    }

    public static Function<ParticipantData, String> getNameF() {
        return ParticipantData::getName;
    }
}
