package ru.yandex.calendar.logic.sharing.participant;

import java.util.Objects;

import javax.annotation.Nullable;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.bolts.function.Function2B;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.lang.Validate;

/**
 * @see UidOrResourceId
 */
public class ParticipantId {
    private final ParticipantKind kind;
    private final long longValue;
    @Nullable
    private final Email emailValue;

    private ParticipantId(ParticipantKind kind, long longValue) {
        Validate.notNull(kind);
        this.kind = kind;
        this.longValue = longValue;
        this.emailValue = null;
    }

    private ParticipantId(ParticipantKind kind, Email emailValue) {
        Validate.notNull(kind);
        this.kind = kind;
        this.longValue = 121212;
        this.emailValue = emailValue;
    }

    public ParticipantKind getKind() {
        return kind;
    }

    public boolean isResource() {
        return getKind() == ParticipantKind.RESOURCE;
    }

    public boolean isYandexUser() {
        return getKind() == ParticipantKind.YANDEX_USER;
    }

    public boolean isYandexUserWithUid(PassportUid uid) {
        return sameAs(ParticipantId.yandexUid(uid));
    }

    public boolean isExternalUser() {
        return getKind() == ParticipantKind.EXTERNAL_USER;
    }

    public boolean isAnyUser() {
        return isYandexUser() || isExternalUser();
    }

    public static ParticipantId yandexUid(PassportUid uid) {
        return new ParticipantId(ParticipantKind.YANDEX_USER, uid.getUid());
    }

    public static ParticipantId resourceId(long resourceId) {
        return new ParticipantId(ParticipantKind.RESOURCE, resourceId);
    }

    public static ParticipantId invitationIdForExternalUser(Email invitationId) {
        return new ParticipantId(ParticipantKind.EXTERNAL_USER, invitationId.normalize());
    }

    private void checkKind(ParticipantKind kind) {
        Check.C.equals(kind, this.kind);
    }

    public PassportUid getUid() {
        checkKind(ParticipantKind.YANDEX_USER);
        return new PassportUid(longValue);
    }

    public Option<PassportUid> getUidIfYandexUser() {
        if (kind == ParticipantKind.YANDEX_USER)
            return Option.of(getUid());
        else
            return Option.empty();
    }

    public Option<Email> getEmailIfExternalUser() {
        return kind == ParticipantKind.EXTERNAL_USER ? Option.of(emailValue) : Option.empty();
    }

    public static Function<ParticipantId, Option<PassportUid>> getUidIfYandexUserF() {
        return ParticipantId::getUidIfYandexUser;
    }

    public static Function<ParticipantId, Option<Long>> getResourceIdIfResourceF() {
        return ParticipantId::getResourceIdIfResource;
    }

    public static Function<ParticipantId, Option<Email>> getEmailIfExternalUserF() {
        return ParticipantId::getEmailIfExternalUser;
    }

    public Option<Long> getResourceIdIfResource() {
        if (kind == ParticipantKind.RESOURCE) {
            return Option.of(getResourceId());
        } else {
            return Option.empty();
        }
    }

    public long getResourceId() {
        checkKind(ParticipantKind.RESOURCE);
        return longValue;
    }

    public Email getInviteeEmailForExternalUser() {
        checkKind(ParticipantKind.EXTERNAL_USER);
        return emailValue;
    }

    public Option<UidOrResourceId> getSubjectId() {
        if (isYandexUser()) {
            return Option.of(UidOrResourceId.user(getUid()));
        } else if (isResource()) {
            return Option.of(UidOrResourceId.resource(getResourceId()));
        } else if (isExternalUser()) {
            return Option.empty();
        } else {
            throw new IllegalStateException();
        }
    }

    public static Function<ParticipantId, Option<UidOrResourceId>> getSubjectIdF() {
        return ParticipantId::getSubjectId;
    }

    public static Function1B<ParticipantId> isResourceF() {
        return ParticipantId::isResource;
    }

    public static Function<Long, ParticipantId> consResourceIdF() {
        return ParticipantId::resourceId;
    }

    public static Function<PassportUid, ParticipantId> consPassportUidIdF() {
        return ParticipantId::yandexUid;
    }

    public static Function<Email, ParticipantId> consExternalUserIdF() {
        return ParticipantId::invitationIdForExternalUser;
    }

    @Override
    public int hashCode() {
        return Objects.hash(kind, longValue, emailValue);
    }

    public boolean sameAs(ParticipantId participantId) {
        if (this.kind != participantId.kind)
            return false;

        if (this.kind == ParticipantKind.EXTERNAL_USER) {
            return this.emailValue.equalsIgnoreCase(participantId.emailValue);
        } else {
            return this.longValue == participantId.longValue;
        }
    }

    @Override
    public boolean equals(Object obj) {
        return obj instanceof ParticipantId && this.sameAs((ParticipantId) obj);
    }

    @Override
    public String toString() {
        return this.kind.toString() + "(" + (emailValue != null ? emailValue.toString() : String.valueOf(longValue)) + ")";
    }

    public static Function<ParticipantId, Long> getResourceIdF() {
        return ParticipantId::getResourceId;
    }

    public Function1B<ParticipantId> sameAsF() {
        return sameAsF2().bind1(this);
    }

    public static Function2B<ParticipantId, ParticipantId> sameAsF2() {
        return ParticipantId::sameAs;
    }

    public static Function1B<ParticipantId> isYandexUserWithUidF(final PassportUid uid) {
        return id -> id.isYandexUserWithUid(uid);
    }
}
