package ru.yandex.calendar.logic.sharing.participant;

import java.util.Set;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.Validate;

/**
 * @author Stepan Koltsov
 */
public abstract class Participants {

    private final ListF<YandexUserParticipantInfo> subscribers;

    protected Participants(ListF<YandexUserParticipantInfo> subscribers) {
        this.subscribers = subscribers;
    }

    private static class NotMeeting extends Participants {
        public NotMeeting(ListF<YandexUserParticipantInfo> subscribers) {
            super(subscribers);
        }

        @Override
        public String toString() {
            return "not meeting";
        }
    }

    private static class Meeting extends Participants {
        // organizer first
        private final ListF<ParticipantInfo> participants;

        private final ParticipantInfo organizer;
        private final ListF<ParticipantInfo> attendees;
        private final ListF<ParticipantInfo> optionalAttendees;
        private final ListF<ParticipantInfo> attendeesButNotOrganizer;
        private final ListF<ParticipantInfo> optionalAttendeesButNotOrganizer;

        public Meeting(ListF<ParticipantInfo> participants, ListF<YandexUserParticipantInfo> subscribers) {
            super(subscribers);
            Validate.notEmpty(participants);

            ListF<ParticipantInfo> organizers = participants.filter(ParticipantInfo.isOrganizerF());
            Validate.V.hasSize(1, organizers);

            this.organizer = organizers.single();
            this.optionalAttendees = participants
                    .filter(ParticipantInfo.isAttendeeF())
                    .filter(ParticipantInfo.isOptionalF());
            this.attendees = participants
                    .filter(ParticipantInfo.isAttendeeF())
                    .filter(ParticipantInfo.isOptionalF().notF());
            this.attendeesButNotOrganizer = attendees.filter(ParticipantInfo.isOrganizerF().notF());
            this.optionalAttendeesButNotOrganizer = optionalAttendees.filter(ParticipantInfo.isOrganizerF().notF());
            this.participants = Cf.list(this.organizer).plus(attendeesButNotOrganizer).plus(optionalAttendeesButNotOrganizer)
                    .stableUniqueBy(ParticipantInfo::getId); // if some users containing in both lists (optional and required)

            SetF<Email> uniqueEmails = participants.map(ParticipantInfo.getEmailF()).unique();
            Validate.V.hasSize(uniqueEmails.size(), participants);

            // RFC5545 requires at least one attendee for meeting
            Validate.notEmpty(attendees);
        }

        @Override
        public String toString() {
            return "organizer: " + organizer + ", attendees: " + attendees + ", optional attendees: " + optionalAttendees;
        }
    }

    private static class InconsistentMeeting extends Participants {
        private final ListF<ParticipantInfo> participants;

        public InconsistentMeeting(ListF<ParticipantInfo> participants, ListF<YandexUserParticipantInfo> subscribers) {
            super(subscribers);
            Validate.notEmpty(participants);
            this.participants = participants;
        }

        @Override
        public String toString() {
            return "InconsistentMeeting" + participants.toString();
        }
    }

    public boolean isMeeting() {
        return this instanceof Meeting;
    }

    public boolean isMeetingOrInconsistent() {
        return isMeeting() || isInconsistent();
    }

    public boolean isInconsistent() {
        return this instanceof InconsistentMeeting;
    }

    /**
     * @see #isNotMeetingStrict()
     */
    public boolean isNotMeetingOrIsInconsistent() {
        return !isMeeting();
    }

    /**
     * @see #isNotMeetingOrIsInconsistent()
     */
    public boolean isNotMeetingStrict() {
        return this instanceof NotMeeting;
    }

    public boolean isExtMeeting() {
        return isMeeting() && getOrganizer() instanceof ExternalUserParticipantInfo;
    }

    private Meeting asMeeting() {
        if (!isMeeting()) {
            throw new IllegalStateException("not a meeting: " + this);
        }
        return (Meeting) this;
    }

    private InconsistentMeeting asInconsistentMeeting() {
        if (!isInconsistent()) {
            throw new IllegalStateException("not an inconcistent meeting: " + this);
        }
        return (InconsistentMeeting) this;
    }

    public ParticipantInfo getOrganizer() {
        return asMeeting().organizer;
    }

    public ListF<ParticipantInfo> getAllAttendees() {
        return asMeeting().attendees
                .plus(asMeeting().optionalAttendees)
                .unique().toList();
    }

    public ListF<ParticipantInfo> getAttendees() {
        return asMeeting().attendees;
    }

    public ListF<ParticipantInfo> getOptionalAttendees() {
        return asMeeting().optionalAttendees;
    }

    public ListF<ParticipantInfo> getParticipants() {
        return asMeeting().participants;
    }

    public Option<ParticipantInfo> getOrganizerSafe() {
        return isMeeting() ? Option.of(getOrganizer()) : Option.empty();
    }

    public ListF<ParticipantInfo> getAllAttendeesButNotOrganizerSafe() {
        return isMeeting() ? getAllAttendeesButNotOrganizer() : Cf.list();
    }

    public ListF<ParticipantInfo> getAttendeesButNotOrganizerSafe() {
        return isMeeting() ? getAttendeesButNotOrganizer() : Cf.list();
    }

    public ListF<ParticipantInfo> getOptionalAttendeesButNotOrganizerSafe() {
        return isMeeting() ? getOptionalAttendeesButNotOrganizer() : Cf.list();
    }

    public Option<ParticipantId> getOrganizerIdSafe() {
        return getOrganizerSafe().map(ParticipantInfo.getIdF());
    }

    public ListF<ParticipantId> getAllAttendeesButNotOrganizerIdsSafe() {
        return getAllAttendeesButNotOrganizerSafe().map(ParticipantInfo.getIdF());
    }

    public ListF<ParticipantId> getAttendeesButNotOrganizerIdsSafe() {
        return getAttendeesButNotOrganizerSafe().map(ParticipantInfo.getIdF());
    }

    public ListF<ResourceParticipantInfo> getResourceParticipants() {
        return getParticipants().filterByType(ResourceParticipantInfo.class);
    }

    public ListF<UserParticipantInfo> getUserParticipants() {
        return getParticipants().filterByType(UserParticipantInfo.class);
    }

    public ListF<UserParticipantInfo> getUserParticipantsSafe() {
        return isMeeting() ? getUserParticipants() : Cf.list();
    }

    public ListF<ParticipantId> getParticipantIds() {
        return getParticipants().map(ParticipantInfo.getIdF());
    }

    public ListF<Email> getParticipantEmailsSafe() {
        return getParticipantsSafe().map(ParticipantInfo.getEmailF());
    }

    public Option<ParticipantInfo> getParticipantByEmail(final Email email) {
        asMeeting();
        return getParticipantByEmailSafe(email);
    }

    public Option<ParticipantInfo> getParticipantByEmailSafe(final Email email) {
        return participantsByEmailSafe().getO(email);
    }

    public Option<ParticipantInfo> getParticipantById(final ParticipantId participantId) {
        asMeeting();
        return participantsByIdSafe().getO(participantId);
    }

    public Option<YandexUserParticipantInfo> getParticipantByUid(final PassportUid uid) {
        return getParticipantById(ParticipantId.yandexUid(uid)).uncheckedCast();
    }

    public Option<ExternalUserParticipantInfo> getExternalParticipantByEmail(Email email) {
        return getParticipantById(ParticipantId.invitationIdForExternalUser(email)).uncheckedCast();
    }

    public Option<YandexUserParticipantInfo> getLastYandexUser() {
        if (isMeeting()) {
            return getParticipants().filterByType(YandexUserParticipantInfo.class).lastO();
        } else {
            return Option.empty();
        }
    }

    public boolean isOrganizer(PassportUid uid) {
        return isMeeting() && getOrganizer().getUid().isSome(uid);
    }

    public ListF<ParticipantInfo> getAllAttendeesSafe() {
        return isMeeting() ? getAllAttendees() : Cf.list();
    }

    public ListF<ParticipantInfo> getAttendeesSafe() {
        return isMeeting() ? getAttendees() : Cf.list();
    }

    public ListF<ParticipantInfo> getOptionalAttendeesSafe() {
        return isMeeting() ? getOptionalAttendees() : Cf.list();
    }

    public ListF<ParticipantInfo> getParticipantsSafe() {
        if (isMeeting()) {
            return getParticipants();
        } else {
            return Cf.list();
        }
    }

    public ListF<YandexUserParticipantInfo> getSubscribers() {
        return subscribers;
    }

    public ListF<YandexUserParticipantInfo> getNotDeclinedSubscribers() {
        return subscribers.filter(ParticipantInfo.getDecisionF().andThenEquals(Decision.NO).notF());
    }

    public ListF<YandexUserParticipantInfo> getNotDeclinedOutlookerSubscribers() {
        return getNotDeclinedSubscribers().filter(u -> u.getSettings().isOutlookerOrEwser());
    }

    public ListF<ParticipantInfo> getParticipantsSafeWithInconsistent() {
        if (isMeeting()) {
            return getParticipants();
        } else if (isInconsistent()) {
            return asInconsistentMeeting().participants;
        } else {
            return Cf.list();
        }
    }

    public ListF<UserParticipantInfo> getUserParticipantsSafeWithInconsistent() {
        return getParticipantsSafeWithInconsistent().filterByType(UserParticipantInfo.class);
    }

    public ListF<ParticipantInfo> getAllAttendeesButNotOrganizer() {
        return asMeeting().attendeesButNotOrganizer
                .plus(asMeeting().optionalAttendeesButNotOrganizer)
                .unique().toList();
    }

    public ListF<ParticipantInfo> getAttendeesButNotOrganizer() {
        return asMeeting().attendeesButNotOrganizer;
    }

    public ListF<ParticipantInfo> getOptionalAttendeesButNotOrganizer() {
        return asMeeting().optionalAttendeesButNotOrganizer;
    }

    public ListF<UserParticipantInfo> getAllUserAttendeesButNotOrganizer() {
        return getAllAttendeesButNotOrganizer().filterByType(UserParticipantInfo.class);
    }

    public ListF<UserParticipantInfo> getUserAttendeesButNotOrganizer() {
        return getAttendeesButNotOrganizer().filterByType(UserParticipantInfo.class);
    }

    public ListF<UserParticipantInfo> getUserOptionalAttendeesButNotOrganizer() {
        return getOptionalAttendeesButNotOrganizer().filterByType(UserParticipantInfo.class);
    }

    public Option<ParticipantInfo> getByIdSafe(ParticipantId id) {
        return participantsByIdSafe().getO(id);
    }

    public Option<ParticipantInfo> getByIdSafeWithInconsistent(ParticipantId id) {
        return participantsByIdSafeWithInconsistent().getO(id);
    }

    public Option<YandexUserParticipantInfo> getByUidSafeWithInconsistent(PassportUid uid) {
        return getByIdSafeWithInconsistent(ParticipantId.yandexUid(uid)).uncheckedCast();
    }

    public Option<UserParticipantInfo> getByPrivateTokenSafe(final String privateToken) {
        return getParticipantsSafe().find(p -> p instanceof UserParticipantInfo
                && ((UserParticipantInfo) p).getPrivateToken().isSome(privateToken)).uncheckedCast();
    }

    public ListF<ResourceParticipantInfo> getResourceParticipantsSafeWithInconsistent() {
        return getParticipantsSafeWithInconsistent().filterByType(ResourceParticipantInfo.class);
    }

    public ListF<Long> getResourceIdsSafeWithInconsistent() {
        return getResourceParticipantsSafeWithInconsistent().map(ResourceParticipantInfo.getResourceIdF());
    }

    public ListF<PassportUid> getParticipantUidsSafe() {
        return getParticipantsSafe().filterMap(ParticipantInfo.getIdF().andThen(ParticipantId.getUidIfYandexUserF()));
    }

    public ListF<PassportUid> getParticipantUidsSafeWithInconsistent() {
        return getParticipantsSafeWithInconsistent().filterMap(ParticipantInfo::getUid);
    }

    private SetF<ParticipantId> participantIdsSafe;

    public boolean isParticipantWithInconsistent(PassportUid uid) {
        if (participantIdsSafe == null) {
            participantIdsSafe = getParticipantsSafeWithInconsistent().map(ParticipantInfo::getId).unique();
        }
        return participantIdsSafe.containsTs(ParticipantId.yandexUid(uid));
    }

    private SetF<ParticipantId> organizerIdsSafe;

    public SetF<ParticipantId> getOrganizerIdWithInconsistent() {
        if (organizerIdsSafe == null) {
            if (isMeeting()) {
                organizerIdsSafe = Cf.set(getOrganizer().getId());
            } else if (isInconsistent()) {
                organizerIdsSafe = asInconsistentMeeting()
                        .participants.find(ParticipantInfo::isOrganizer).map(ParticipantInfo::getId).unique();
            } else {
                organizerIdsSafe = Cf.set();
            }
        }
        return organizerIdsSafe;
    }

    private transient MapF<Email, ParticipantInfo> participantByEmailAny;

    private MapF<Email, ParticipantInfo> participantsByEmailSafe() {
        return isInconsistent() ? Cf.map() : participantsByEmailWithInconsistent();
    }

    private MapF<Email, ParticipantInfo> participantsByEmailWithInconsistent() {
        if (participantByEmailAny == null) {
            participantByEmailAny = getParticipantsSafeWithInconsistent()
                    .toMapMappingToKey(ParticipantInfo::getEmail);
        }
        return participantByEmailAny;
    }

    private transient MapF<ParticipantId, ParticipantInfo> participantByIdAny;

    private MapF<ParticipantId, ParticipantInfo> participantsByIdSafe() {
        return isInconsistent() ? Cf.map() : participantsByIdSafeWithInconsistent();
    }

    private MapF<ParticipantId, ParticipantInfo> participantsByIdSafeWithInconsistent() {
        if (participantByIdAny == null) {
            participantByIdAny = getParticipantsSafeWithInconsistent()
                    .toMapMappingToKey(ParticipantInfo::getId);
        }
        return participantByIdAny;
    }

    public boolean isOrganizerWithInconsistent(PassportUid uid) {
        return getOrganizerIdWithInconsistent().containsTs(ParticipantId.yandexUid(uid));
    }

    public boolean isAttendee(PassportUid uid) {
        return participantsByIdSafe().getO(ParticipantId.yandexUid(uid)).exists(p -> !p.isOrganizer());
    }

    public static Participants notMeeting() {
        return new NotMeeting(Cf.list());
    }

    public static Participants maybeMeetingWithSubscribers(ListF<ParticipantInfo> participants) {
        ListF<YandexUserParticipantInfo> subscribers = participants
                .filterByType(YandexUserParticipantInfo.class)
                .filter(UserParticipantInfo::isSubscriber);
        Set<ParticipantId> subscriberIds = subscribers.map(ParticipantInfo.getIdF()).unique();

        participants = participants.filter(participantInfo -> !subscriberIds.contains(participantInfo.getId()));

        if (participants.isEmpty()) {
            return new NotMeeting(subscribers);
        }
        try {
            return new Meeting(participants, subscribers);
        } catch (IllegalArgumentException e) {
            return new InconsistentMeeting(participants, subscribers);
        }
    }

} //~
