package ru.yandex.calendar.logic.sharing.participant;

import lombok.AccessLevel;
import lombok.NoArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.Validate;

/**
 * @see ParticipantInfo
 */
@NoArgsConstructor(access = AccessLevel.PRIVATE)
public class ParticipantsData {
    private static class NotMeeting extends ParticipantsData {
    }

    public static class Meeting extends ParticipantsData {
        private final ListF<ParticipantData> participants;

        private final ParticipantData organizer;

        public Meeting(ListF<ParticipantData> participants) {
            this.participants = participants;

            this.organizer = participants.filter(ParticipantData.isOrganizerF()).single();
        }

        public ParticipantData getOrganizer() {
            return organizer;
        }
    }

    public boolean isMeeting() {
        return this instanceof Meeting;
    }

    public Meeting asMeeting() {
        return (Meeting) this;
    }

    public ListF<ParticipantData> getParticipantsSafe() {
        if (isMeeting())
            return asMeeting().participants;
        else
            return Cf.list();
    }

    public ListF<Email> getParticipantEmailsSafe() {
        return getParticipantsSafe().map(ParticipantData.getEmailF());
    }

    public Option<Email> getOrganizerEmailSafe() {
        return getParticipantsSafe().find(ParticipantData.isOrganizerF()).map(ParticipantData.getEmailF());
    }

    public ListF<Email> getAttendeesButNotOrganizerEmailsSafe() {
        return getParticipantEmailsSafe().filter(getOrganizerEmailSafe().containsF().notF());
    }

    public ParticipantsData plusAttendee(Email email, Decision decision) {
        return new Meeting(asMeeting().getParticipantsSafe().plus(new ParticipantData(email, "", decision, true, false, false)));
    }

    public ParticipantsData excludeParticipants(ListF<Email> emails) {
        if (isMeeting()) {
            Function1B<ParticipantData> matchesF = Function1B.allOfF(
                    ParticipantData.isOrganizerF().notF(),
                    ParticipantData.getEmailF().andThen(emails.containsF()));

            return new Meeting(asMeeting().participants.filter(matchesF.notF()));
        } else {
            return this;
        }
    }

    public static ParticipantsData merge(ParticipantData organizer, ListF<ParticipantData> attendees) {
        Validate.isTrue(organizer.isOrganizer());

        Validate.V.forAll(attendees, ParticipantData.isOrganizerF().notF());
        Validate.V.forAll(attendees, ParticipantData.isAttendeeF());

        ParticipantData fixedOrganizer;
        Tuple2<ListF<ParticipantData>, ListF<ParticipantData>> t = attendees.partition(ParticipantData.getEmailF().andThenEquals(organizer.getEmail()));
        if (t._1.isEmpty()) {
            fixedOrganizer = organizer;
        } else {
            Validate.V.hasSize(1, t._1);
            fixedOrganizer = organizer.withAttendee(true);
        }
        return new Meeting(Cf.list(fixedOrganizer).plus(t._2));
    }

    public static ParticipantsData notMeeting() {
        return new NotMeeting();
    }
}
