package ru.yandex.calendar.logic.sharing.participant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.bolts.function.Function2;
import ru.yandex.calendar.logic.beans.generated.EventResource;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.Validate;

public class ResourceParticipantInfo extends ParticipantInfo {
    private final ResourceInfo resourceInfo;
    private final EventResource eventResource;
    private final long resourceId;
    private final String exchangeName;
    private final long officeId;

    public ResourceParticipantInfo(ResourceInfo resourceInfo, EventResource eventResource) {
        Validate.V.isTrue(resourceInfo.getResource().getId() == eventResource.getResourceId());
        this.resourceInfo = resourceInfo;
        this.eventResource = eventResource;
        this.resourceId = resourceInfo.getResource().getId();
        this.exchangeName = resourceInfo.getResource().getExchangeName().getOrElse("?");
        this.officeId = resourceInfo.getOffice().getId();
    }

    @Override
    public ParticipantId getId() {
        return ParticipantId.resourceId(resourceInfo.getResource().getId());
    }

    public ResourceInfo getResourceInfo() {
        return resourceInfo;
    }

    @Override
    public long getEventId() {
        return eventResource.getEventId();
    }

    public long getResourceId() {
        return eventResource.getResourceId();
    }

    @Override
    public String getHiddenName() {
        return resourceInfo.getName().getOrElse("");
    }

    @Override
    public Decision getDecision() {
        return Decision.YES;
    }

    @Override
    public Option<ActionSource> getDecisionSource() {
        return Option.empty();
    }

    @Override
    public boolean isAttendee() {
        return true;
    }

    @Override
    public boolean isOptional() {
        return false;
    }

    @Override
    public boolean isOrganizer() {
        return false;
    }

    @Override
    public Email getEmail() {
        return ResourceRoutines.getResourceEmail(resourceInfo.getResource());
    }

    @Override
    public Option<PassportUid> getUid() {
        return Option.empty();
    }

    @Override
    public String getReason() {
        return "";
    }

    @Override
    public Option<String> getExchangeId() {
        return eventResource.getExchangeId();
    }

    public Resource getResource() {
        return getResourceInfo().getResource();
    }

    @Override
    protected String specifics() {
        return resourceId + "," + exchangeName;
    }

    public static Function<ResourceParticipantInfo, ResourceInfo> resourceInfoF() {
        return ResourceParticipantInfo::getResourceInfo;
    }

    public static Function2<ResourceInfo, EventResource, ResourceParticipantInfo> consF() {
        return ResourceParticipantInfo::new;
    }

    public boolean hasExchangeId() {
        return getExchangeId().isPresent();
    }

    public static Function1B<ResourceParticipantInfo> hasExchangeIdF() {
        return ResourceParticipantInfo::hasExchangeId;
    }

    public static Function<ResourceParticipantInfo, Option<String>> getExchangeIdF() {
        return ResourceParticipantInfo::getExchangeId;
    }

    public static Function<ResourceParticipantInfo, Resource> resourceF() {
        return resourceInfoF().andThen(ResourceInfo.resourceF());
    }

    public static Function<ResourceParticipantInfo, Long> resourceIdF() {
        return resourceF().andThen(Resource.getIdF());
    }

    public static Function<ResourceParticipantInfo, Long> getResourceIdF() {
        return ResourceParticipantInfo::getResourceId;
    }

    public long getOfficeId() {
        return officeId;
    }

    public static Function1B<ResourceParticipantInfo> isYaTeamAndSyncWithExchangeF() {
        return resourceInfoF().andThen(ResourceInfo.isYaTeamAndSyncWithExchange());
    }

    public static Function1B<ResourceParticipantInfo> isAsyncWithExchangeF() {
        return resourceInfoF().andThen(ResourceInfo.isAsyncWithExchangeF());
    }
}
