package ru.yandex.calendar.logic.sharing.perm;

import java.util.EnumSet;
import java.util.Set;

import lombok.Getter;

import ru.yandex.calendar.micro.perm.LayerAction;
import ru.yandex.misc.enums.StringEnum;
import ru.yandex.misc.enums.StringEnumResolver;
import ru.yandex.misc.lang.CamelWords;

/**
 * Available values of Layer / LayerUser permission fields
 * that denote layer atomic permission classes.
 * They cover all possible atomic layer actions.
 */
public enum LayerActionClass implements StringEnum {
    /**
     * "Self-shared" (added via public link) layers.
     */
    ACCESS(LayerAction.ANONYMOUS_LIST),
    /**
     * List (have simple access to the layer contents, 'weak' events view)
     */
    LIST(LayerAction.LIST),
    /**
     * {@link #LIST} + create / attach new events
     */
    CREATE(LayerAction.LIST, LayerAction.VIEW_EVENT, LayerAction.CREATE_EVENT),
    /**
     * {@link #LIST} + 'strong' events view
     */
    VIEW(LayerAction.LIST, LayerAction.VIEW_EVENT),
    /**
     * Everything that allows both {@link #CREATE} and {@link #VIEW}
     * + attach, edit, delete and detach events,
     * and edit layer itself
     */
    EDIT(LayerAction.LIST, LayerAction.CREATE_EVENT,
         LayerAction.VIEW_EVENT, LayerAction.EDIT_EVENT,
         LayerAction.DETACH_EVENT, LayerAction.DELETE_EVENT,
         LayerAction.EDIT
    ),
    /**
     * Everything: {@link #EDIT} + delete layer;
     * grant and revoke permissions through layer-users
     */
    ADMIN(EnumSet.allOf(LayerAction.class));

    @Getter
    private final Set<LayerAction> actions;

    LayerActionClass() {
        this(EnumSet.noneOf(LayerAction.class));
    }

    LayerActionClass(EnumSet<LayerAction> actions) {
        this.actions = actions;
    }

    LayerActionClass(LayerAction firstAction, LayerAction... restActions) {
        this.actions = EnumSet.of(firstAction, restActions);
    }

    public boolean contains(LayerAction layerAction) { return actions.contains(layerAction); }

    public static final StringEnumResolver<LayerActionClass> R = StringEnumResolver.r(LayerActionClass.class);

    public String toDbValue() { return CamelWords.parse(this.name()).toDbName(); }

    @Override
    public String value() {
        return toDbValue();
    }
}
