package ru.yandex.calendar.logic.sharing.perm;


import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Stream;

import javax.inject.Inject;

import lombok.extern.slf4j.Slf4j;
import lombok.val;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectorsF;
import ru.yandex.bolts.collection.Lazy;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function2B;
import ru.yandex.calendar.frontend.web.cmd.run.EventOrLayerAction;
import ru.yandex.calendar.frontend.web.cmd.run.PermissionDeniedUserException;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventLayer;
import ru.yandex.calendar.logic.beans.generated.Layer;
import ru.yandex.calendar.logic.beans.generated.LayerUser;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.beans.generated.Settings;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.EventInvitationManager;
import ru.yandex.calendar.logic.event.EventUserWithRelations;
import ru.yandex.calendar.logic.event.EventWithRelations;
import ru.yandex.calendar.logic.event.dao.EventLayerDao;
import ru.yandex.calendar.logic.event.dao.EventResourceDao;
import ru.yandex.calendar.logic.event.dao.EventUserDao;
import ru.yandex.calendar.logic.event.model.EventType;
import ru.yandex.calendar.logic.event.model.ParticipantsOrInvitationsData;
import ru.yandex.calendar.logic.layer.LayerRoutines;
import ru.yandex.calendar.logic.layer.LayerType;
import ru.yandex.calendar.logic.layer.LayerUserDao;
import ru.yandex.calendar.logic.layer.UserLayersSharing;
import ru.yandex.calendar.logic.resource.ResourceDao;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.resource.ResourceType;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.sharing.participant.EventParticipants;
import ru.yandex.calendar.logic.sharing.participant.ParticipantId;
import ru.yandex.calendar.logic.sharing.participant.ParticipantInfo;
import ru.yandex.calendar.logic.user.SettingsInfo;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserDao;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.micro.perm.EventAction;
import ru.yandex.calendar.micro.perm.LayerAction;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox.PassportDomain;

import static java.util.Collections.emptyList;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static java.util.function.Function.identity;
import static java.util.function.Predicate.not;

/**
 * Permission manager answers the questions about user's possibility to
 * perform an atomic action at given shared resource: event or layer.
 */
@Slf4j
public class PermManager implements Authorizer {
    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private LayerRoutines layerRoutines;
    @Autowired
    private ResourceDao resourceDao;
    @Autowired
    private UserManager userManager;
    @Autowired
    private EventInvitationManager eventInvitationManager;
    @Autowired
    private EventResourceDao eventResourceDao;
    @Autowired
    private EventLayerDao eventLayerDao;
    @Autowired
    private EventUserDao eventUserDao;
    @Autowired
    private UserDao userDao;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;
    @Inject
    private SuperSecretsManager superSecretsManager;
    @Inject
    private LayerUserDao layerUserDao;

    @Override
    public void loadBatchAndCacheAllRequiredForPermsCheck(PassportUid uid, List<EventWithRelations> events) {
        val layerIds = StreamEx.of(events)
            .map(EventWithRelations::getEventLayers)
            .flatMap(layers -> layers.stream().map(EventLayer::getLayerId))
            .distinct()
            .collect(CollectorsF.toList());
        layerRoutines.getLayerUsersByUidAndLayerIds(uid, layerIds);
        layerRoutines.getLayersById(layerIds);
    }

    @Override
    public boolean canPerformLayerAction(LayerInfoForPermsCheck layer, LayerAction action, ActionSource actionSource) {
        if (actionSource.isTrusted()) {
            return true;
        }

        if (layer.getType().isAbsence() && !actionSource.isStaff() && action != LayerAction.LIST) {
            return false;
        }

        return action == LayerAction.LIST;
    }

    @Override
    public boolean canPerformLayerAction(UserInfo user, LayerInfoForPermsCheck layer, Optional<UserLayersSharing> sharing, LayerAction action,
                                         ActionSource source) {
        val uid = user.getUid();

        if (source.isTrusted()) {
            return true;
        }

        if (layer.getType().isAbsence() && !source.isStaff() && action != LayerAction.LIST) {
            return false;
        }

        val layerCreatorUid = layer.getCreatorUid();
        if (layerCreatorUid.sameAs(user.getUid())) {
            return true;
        }

        log.debug("User {} is not the same as layer creator {}", user, layerCreatorUid);

        val layerId = layer.getId();
        final var lazyPerm = Lazy.withSupplier(() -> {
            return sharing.map(s -> s.get(layerId))
                .orElseGet(() -> {
                    return layerRoutines.getLayerUser(layerId, uid)
                        .toOptional()
                        .map(LayerUser::getPerm)
                        .map(LayerActionClass::getActions)
                        .orElseGet(Collections::emptySet);
                });
        });

        if (action == LayerAction.LIST && (!user.isExternalYt() || !lazyPerm.get().isEmpty())) {
            return true;
        }

        // ...if not successful, try to use specific permissions in layer user
        return lazyPerm.get().contains(action);
    }

    @Override
    public void ensureCanPerformLayerAction(LayerInfoForPermsCheck layer, LayerAction action, ActionSource actionSource) {
        if (!canPerformLayerAction(layer, action, actionSource)) {
            throw permissionDeniedException(Optional.empty(), "layer: " + layer.getId(), EventOrLayerAction.layer(action));
        }
    }

    @Override
    public void ensureCanPerformLayerAction(UserInfo user, LayerInfoForPermsCheck layer, Optional<UserLayersSharing> sharing,
                                            LayerAction action, ActionSource source) {
        if (!canPerformLayerAction(user, layer, sharing, action, source)) {
            throw permissionDeniedException(user, "layer: " + layer.getId(), EventOrLayerAction.layer(action));
        }
    }

    @Override
    public void ensureLayerType(LayerInfoForPermsCheck layer, LayerType... types) {
        if (types.length == 0) {
            throw new IllegalArgumentException("types should contain at least one element");
        }

        val layerType = layer.getType();
        if (Arrays.asList(types).contains(layerType)) {
            return;
        }

        val msg = "Layer (id = " + layer.getId() + ") is of wrong type: " + layerType;
        throw new PermissionDeniedUserException(msg);
    }

    @Override
    public void ensureLayerProperties(LayerInfoForPermsCheck layer, PassportUid uid, LayerType... types) {
        ensureLayerType(layer, types);
        if (!uid.sameAs(layer.getCreatorUid())) {
            val msg = "Layer (id = " + layer.getId() + ") does not belong to uid = " + uid;
            throw new PermissionDeniedUserException(msg);
        }
    }

    @Override
    public void ensureCanPerformLayerActionType(UserInfo user, LayerInfoForPermsCheck layer, LayerAction action,
                                                ActionSource actionSource, LayerType... types) {
        ensureLayerType(layer, types);
        ensureCanPerformLayerAction(user, layer, Optional.empty(), action, actionSource);
    }

    @Override
    public boolean canViewEvent(UserInfo user, EventInfoForPermsCheck event, ActionSource actionSource) {
        if (actionSource.isTrusted()) {
            return true;
        }
        if (isAbsenceEvent(event.getPrimaryLayer().toOptional())) {
            return true;
        }
        if (event.isHasSuperSecretUserIsKeep()) {
            return true;
        }

        val hasSuperSecret = event.isHasSuperSecretAndUserIsNotKeeper();
        val resources = event.getResources();

        if (!hasSuperSecret
                && (event.getOrganizer().isPresent() || user.isSuperUser())
                && (canAdminSomeEventResources(user, resources)
                || canViewAnyEventForAllResources(user, resources))) {
            return true;
        }

        if (isOwner(user, event)) {
            return true;
        }
        if (!canViewAllEventResources(user, resources)) {
            return false;
        }

        if (!hasSuperSecret && canEditEventPermissions(user, event, actionSource)) {
            return true;
        }
        if (!user.isExternalYt()
                && event.getPermAll() == EventActionClass.VIEW
                && checkLayerPermissionForEvent(
                user, event.getPrimaryLayer().toOptional(), Optional.of(event.getUserLayersSharing()), LayerAction.LIST, actionSource)) {
            return true;
        }

        if (event.isUserHasEventLayer() || event.isUserIsAttendee()) {
            return true;
        }

        if (user.isExternalYt()) {
            return false;
        }

        // hack for iseg@
        // if i can't see event on main layer, but my friend can and he shared his calendar to me, show me the event
        return event.isUserHasEventOnLayerSharedWithViewPerm();
    }

    @Override
    public boolean canViewEvent(EventInfoForPermsCheck event, ActionSource actionSource) {
        if (actionSource.isTrusted()) {
            return true;
        }
        if (isAbsenceEvent(event.getPrimaryLayer().toOptional())) {
            return true;
        }
        if (event.isHasSuperSecretUserIsKeep()) {
            return true;
        }

        val hasSuperSecret = event.isHasSuperSecretAndUserIsNotKeeper();

        if (!hasSuperSecret && canEditEventPermissions(event, actionSource)) {
            return true;
        }

        return event.getPermAll() == EventActionClass.VIEW
                && checkLayerPermissionForEvent(
                event.getPrimaryLayer().toOptional(), LayerAction.LIST, actionSource);
    }

    @Override
    public boolean hasSuperSecret(EventWithRelations event) {
        return superSecretsManager.hasSuperSecretAndUserIsNotKeeper(Optional.empty(), event);
    }

    @Override
    public boolean canEditEvent(UserInfo user, EventInfoForPermsCheck event, ActionSource actionSource) {
        if (actionSource.isTrusted()) {
            if (actionSource.isFromMailOrHook()) {
                if (!canEditEventForNonTrustedSources(user, event, actionSource)) {
                    log.warn("GREG-171.canEditEvent: user:{}, event:{}", user.getUid(), event.getEventId());
                }
            }
            return true;
        }
        return canEditEventForNonTrustedSources(user, event, actionSource);
    }

    private boolean canEditEventForNonTrustedSources(UserInfo user, EventInfoForPermsCheck event, ActionSource actionSource) {
        if (isAbsenceEvent(event.getPrimaryLayer().toOptional()) && !actionSource.isStaff()) {
            return false;
        }

        if (user.isSuperUser()) {
            return true;
        }

        if (event.getOrganizer().exists(ParticipantId::isExternalUser)) {
            return false;
        }

        if (canAdminSomeEventResources(user, event.getResources())) {
            return true;
        }
        if (isOwner(user, event)) {
            return true;
        }
        if (!canViewAllEventResources(user, event.getResources())) {
            return false;
        }

        if (canEditEventPermissions(user, event, actionSource)) {
            return true;
        }
        if (canChangeOrganizer(user, event, actionSource)) {
            return true;
        }
        if (event.getPermParticipants() == EventActionClass.EDIT || event.isOrganizerLetToEditAnyMeeting()) {
            //...then check whether user participates in this meeting-event
            if (event.isUserIsAttendee()) {
                return true;
            }
        }
        return false;
    }

    @Override
    public boolean canEditEvent(EventInfoForPermsCheck event, ActionSource actionSource) {
        if (actionSource.isTrusted()) {
            return true;
        }
        if (isAbsenceEvent(event.getPrimaryLayer().toOptional()) && !actionSource.isStaff()) {
            return false;
        }

        if (event.getOrganizer().exists(ParticipantId::isExternalUser)) {
            return false;
        }

        if (canEditEventPermissions(event, actionSource)) {
            return true;
        }

        if (event.getPermParticipants() == EventActionClass.EDIT || event.isOrganizerLetToEditAnyMeeting()) {
            //...then check whether user participates in this meeting-event
            if (event.isUserIsAttendee()) {
                return true;
            }
        }
        return false;

    }

    @Override
    public boolean canEditEventPermissions(UserInfo user, EventInfoForPermsCheck event, ActionSource actionSource) {
        if (user.isSuperUser()) {
            return true;
        }

        if (actionSource.isTrusted()) {
            return true;
        }
        if (isAbsenceEvent(event.getPrimaryLayer().toOptional()) && !actionSource.isStaff()) {
            return false;
        }

        ListF<ResourceInfoForPermsCheck> resources = event.getResources();

        if ((event.getOrganizer().isPresent() || user.isSuperUser())
                && canAdminSomeEventResources(user, resources)) {
            return true;
        }
        if (isOwner(user, event)) {
            return true;
        }
        if (!canViewAllEventResources(user, resources)) {
            return false;
        }

        return checkLayerPermissionForEvent(user, event.getPrimaryLayer().toOptional(), Optional.of(event.getUserLayersSharing()),
            LayerAction.EDIT_EVENT, actionSource);
    }

    private boolean canEditEventPermissions(EventInfoForPermsCheck event, ActionSource actionSource) {
        if (actionSource.isTrusted()) {
            return true;
        }
        if (isAbsenceEvent(event.getPrimaryLayer().toOptional()) && !actionSource.isStaff()) {
            return false;
        }

        return checkLayerPermissionForEvent(event.getPrimaryLayer().toOptional(), LayerAction.EDIT_EVENT, actionSource);
    }

    @Override
    public boolean canDeleteEvent(UserInfo user, EventInfoForPermsCheck event, ActionSource actionSource) {
        if (event.getType() == EventType.FEED) {
            return false;
        }
        if (actionSource.isTrusted()) {
            return true;
        }
        if (user.isSuperUser()) {
            return true;
        }
        if (isAbsenceEvent(event.getPrimaryLayer().toOptional()) && !actionSource.isStaff()) {
            return false;
        }

        if (canAdminSomeEventResources(user, event.getResources())) {
            return true;
        }

        if (isOwner(user, event)) {
            return true;
        }
        if (!canViewAllEventResources(user, event.getResources())) {
            return false;
        }

        return checkLayerPermissionForEvent(user, event.getPrimaryLayer().toOptional(), Optional.of(event.getUserLayersSharing()),
            LayerAction.DELETE_EVENT, actionSource);
    }

    @Override
    public boolean canDeleteEvent(EventInfoForPermsCheck event, ActionSource actionSource) {
        if (actionSource.isTrusted()) {
            return true;
        }
        if (event.getType() == EventType.FEED) {
            return false;
        }
        if (isAbsenceEvent(event.getPrimaryLayer().toOptional()) && !actionSource.isStaff()) {
            return false;
        }

        return checkLayerPermissionForEvent(event.getPrimaryLayer().toOptional(), LayerAction.DELETE_EVENT, actionSource);
    }

    @Override
    public boolean canSplitEvent(UserInfo user, EventInfoForPermsCheck event, ActionSource actionSource) {
        return canSplitEvent(event, actionSource);
    }

    @Override
    public boolean canSplitEvent(EventInfoForPermsCheck event, ActionSource actionSource) {
        if (actionSource.isTrusted()) {
            return true;
        }
        if (isAbsenceEvent(event.getPrimaryLayer().toOptional())) {
            return false;
        }

        return EventType.FEED != event.getType()
                && !(event.getOrganizer().isPresent() && event.getOrganizer().get().isExternalUser());
    }

    @Override
    public boolean canInviteToEvent(UserInfo user, EventInfoForPermsCheck event, ActionSource actionSource) {
        if (actionSource.isTrusted()) {
            if (actionSource.isFromMailOrHook()) {
                if (!canInviteToEventForNonTrustedSources(user, event, actionSource)) {
                    log.warn("GREG-171.canInviteToEvent: user:{}, event:{}", user.getUid(), event.getEventId());
                }
            }
            return true;
        }

        return canInviteToEventForNonTrustedSources(user, event, actionSource);
    }

    private boolean canInviteToEventForNonTrustedSources(UserInfo user, EventInfoForPermsCheck event, ActionSource actionSource) {
        if (isAbsenceEvent(event.getPrimaryLayer().toOptional())) {
            return false;
        }

        if (canAdminSomeEventResources(user, event.getResources())) {
            return true;
        }
        if (isOwner(user, event)) {
            return true;
        }
        if (!canViewAllEventResources(user, event.getResources())) {
            return false;
        }

        if (canEditEvent(user, event, actionSource)) {
            return true;
        }

        if (!event.isUserHasEventLayer()) {
            return false;
        }

        return event.isParticipantsCanInvite() && event.isUserIsAttendee();
    }

    public boolean canChangeOrganizer(UserInfo user, Event event, ActionSource actionSource) {
        return canChangeOrganizer(user, loadInfoForPermsCheckByEvents(Optional.of(user), Option.of(event)).get(0), actionSource);
    }

    @Override
    public boolean canChangeOrganizer(UserInfo user, EventInfoForPermsCheck event, ActionSource actionSource) {
        PassportUid uid = user.getUid();
        val organizerUid = event.getOrganizer().filterMap(ParticipantId::getUidIfYandexUser);

        if (canAdminSomeEventResources(user, event.getResources()) && organizerUid.isPresent()) {
            return true;
        }
        if (!canViewAllEventResources(user, event.getResources())) {
            return false;
        }

        if (uid.isYandexTeamRu() && organizerUid.isSome(uid)) {
            return true; // CAL-6841
        }

        if (event.isUserIsAttendee() && organizerUid.isPresent()) {
            val organizer = userManager.getYtUserByUid(organizerUid.get());
            return organizer.stream().anyMatch(u -> u.getInfo().isDismissed()); // CAL-6263
        }
        return false;
    }

    @Override
    public boolean canAdminAllResources(UserInfo user, List<Resource> resources) {
        var resourceInfos = StreamEx.of(resources).map(ResourceInfoForPermsCheck::fromResource).toImmutableList();
        return isAllEventResourcesPermitted(user, resourceInfos, user::canAdminResource, false);
    }

    @Override
    public boolean canViewAnyEventForAllResources(UserInfo user, List<ResourceInfoForPermsCheck> resources) {
        return isAllEventResourcesPermitted(user, resources, user::canViewAnyEventWithResource, false);
    }

    @Override
    public boolean canAdminSomeEventResources(UserInfo user, List<ResourceInfoForPermsCheck> resources) {
        if (user.isSuperUser()) {
            return true;
        }
        if (resources.isEmpty()) {
            return user.isRoomAdmin();
        }
        return StreamEx.of(resources).anyMatch(r -> user.canAdminResource(r.getType(), r.getAccessGroup()));
    }


    @Override
    public boolean canViewAllEventResources(UserInfo user, List<ResourceInfoForPermsCheck> resources) {
        return isAllEventResourcesPermitted(user, resources, user::canViewEventWithResource, true);
    }

    private static boolean isAllEventResourcesPermitted(UserInfo user, List<ResourceInfoForPermsCheck> resources,
                                                        Function2B<ResourceType, Option<Integer>> permittedF,
                                                        boolean permitIfNoResources) {
        if (user.isSuperUser()) {
            return true;
        }
        if (resources.isEmpty()) {
            return user.isRoomAdmin() || permitIfNoResources;
        }

        return resources
            .stream()
            .allMatch(r -> permittedF.apply(r.getType(), r.getAccessGroup()));
    }

    private static boolean isAbsenceEvent(Optional<LayerInfoForPermsCheck> primaryLayer) {
        return primaryLayer.stream()
                .map(LayerInfoForPermsCheck::getType)
                .anyMatch(LayerType::isAbsence);
    }

    @Override
    public void ensureCanDeleteEvent(UserInfo user, EventInfoForPermsCheck event, ActionSource actionSource) {
        if (!canDeleteEvent(user, event, actionSource)) {
            throw permissionDeniedException(user, "event " + event.getEventId(),
                EventOrLayerAction.event(EventAction.DELETE));
        }
    }

    @Override
    public void ensureCanDeleteEvent(EventInfoForPermsCheck event, ActionSource actionSource) {
        if (!canDeleteEvent(event, actionSource)) {
            throw permissionDeniedException(Optional.empty(), "event " + event.getEventId(),
                EventOrLayerAction.event(EventAction.DELETE));
        }
    }

    @Override
    public void ensureCanEditEventPermissions(UserInfo user, EventInfoForPermsCheck event, ActionSource actionSource) {
        if (!canEditEventPermissions(user, event, actionSource)) {
            throw permissionDeniedException(user, "event " + event.getEventId(),
                EventOrLayerAction.event(EventAction.EDIT_PERMISSIONS));
        }
    }

    @Override
    public void ensureCanEditEvent(UserInfo user, EventInfoForPermsCheck event, ActionSource actionSource) {
        if (!canEditEvent(user, event, actionSource)) {
            throw permissionDeniedException(user, "event " + event.getEventId(),
                EventOrLayerAction.event(EventAction.EDIT));
        }
    }

    @Override
    public void ensureCanInviteToEvent(UserInfo user, EventInfoForPermsCheck event, ActionSource actionSource) {
        if (!canInviteToEvent(user, event, actionSource)) {
            throw permissionDeniedException(user, "event " + event.getEventId(),
                    EventOrLayerAction.event(EventAction.INVITE));
        }
    }

    @Override
    public void ensureCanSplitEvent(UserInfo user, EventInfoForPermsCheck event, ActionSource actionSource) {
        if (!canSplitEvent(user, event, actionSource)) {
            throw permissionDeniedException(user, "event " + event.getEventId(),
                    EventOrLayerAction.event(EventAction.SPLIT));
        }
    }

    @Override
    public void ensureCanViewEvent(UserInfo user, EventInfoForPermsCheck event, ActionSource actionSource) {
        if (!canViewEvent(user, event, actionSource)) {
            throw permissionDeniedException(user, "event " + event.getEventId(),
                EventOrLayerAction.event(EventAction.VIEW));
        }
    }

    @Override
    public void ensureCanViewEvent(EventInfoForPermsCheck event, ActionSource actionSource) {
        if (!canViewEvent(event, actionSource)) {
            throw permissionDeniedException(Optional.empty(), "event " + event.getEventId(),
                EventOrLayerAction.event(EventAction.VIEW));
        }
    }

    @Override
    public void ensureCanViewEvent(UserInfo user, EventInfoForPermsCheck event, boolean tokenPresent, ActionSource actionSource) {
        if (!tokenPresent) {
            ensureCanViewEvent(user, event, actionSource);
        }
    }

    @Override
    public void ensureCanViewEventForParticipant(UserInfo participant, PassportUid actorUid, EventInfoForPermsCheck event, boolean tokenPresent, ActionSource actionSource) {
        checkUserCanViewEventsForParticipant(actorUid);
        ensureCanViewEvent(participant, event, tokenPresent, actionSource);
    }

    @Override
    public boolean canViewAvailability(UserInfo user, UidOrResourceId subjectId) {
        if (subjectId.isUser()) {
            return canViewAvailability(user, singletonList(subjectId.getUid()), emptyList()).isAnythingAvailable();
        } else {
            return canViewAvailability(
                    user, emptyList(),
                    singletonList(resourceDao.findResourceById(subjectId.getResourceId()))).isAnythingAvailable();
        }
    }

    @Override
    public ParticipantsAvailability canViewAvailability(UserInfo user, List<UidOrResourceId> subjectIds) {
        val subjectUids = StreamEx.of(subjectIds)
            .flatMap(subject -> subject.getUidO().stream())
            .toImmutableList();
        val resourceIds = StreamEx.of(subjectIds)
            .flatMap(subject -> subject.getResourceIdO().stream())
            .collect(CollectorsF.toList());
        return canViewAvailability(user, subjectUids, resourceDao.findResourcesByIds(resourceIds));
    }

    @Override
    public ParticipantsAvailability canViewAvailability(UserInfo user, List<PassportUid> users, List<Resource> resources) {
        val canViewResourceAvailability = StreamEx.of(resources)
                .mapToEntry(resource -> canAccess(user, resource), identity())
                .mapValues(Resource::getId)
                .mapValues(UidOrResourceId::resource)
                .grouping();

        val settings = settingsRoutines.getSettingsByUidBatch(Cf.toList(users).plus1(user.getUid())).mapValues(SettingsInfo::getCommon);

        val canViewUsersAvailability = StreamEx.of(users)
                .mapToEntry(subjectUid -> canViewUserAvailability(user, subjectUid, settings), identity())
                .mapValues(UidOrResourceId::user)
                .grouping();

        val available = StreamEx.of(canViewResourceAvailability.getOrDefault(true, emptyList()))
                .append(canViewUsersAvailability.getOrDefault(true, emptyList()))
                .toImmutableSet();
        val unavailable = StreamEx.of(canViewResourceAvailability.getOrDefault(false, emptyList()))
                .append(canViewUsersAvailability.getOrDefault(false, emptyList()))
                .toImmutableSet();

        return new ParticipantsAvailability(available, unavailable);
    }

    private static boolean canViewUserAvailability(UserInfo user, PassportUid subjectUid, Map<PassportUid, Settings> settings) {
        val uid = user.getUid();

        if (uid.equals(subjectUid)) {
            return true;
        }

        if (uid.isYandexTeamRu() && subjectUid.isYandexTeamRu()) {
            return true;
        }

        if (subjectUid.isDomains() && !settings.get(subjectUid).getDomain()
                .exists(settings.get(uid).getDomain()::isSome)) {
            return false;
        }

        return settings.get(subjectUid).getShowAvailability();
    }

    /**
     * Checks whether specified user can perform a given layer action
     * on a layer which is the parent layer for the given event
     *
     * @return true if action is available on the event parent layer
     */
    private boolean checkLayerPermissionForEvent(UserInfo user, Optional<LayerInfoForPermsCheck> layer,
                                                 Optional<UserLayersSharing> userLayersSharing, LayerAction action,
                                                 ActionSource actionSource) {
        if (actionSource.isTrusted()) {
            return true;
        }

        if (layer.isEmpty()) {
            return false;
        }

        return canPerformLayerAction(user, layer.get(), userLayersSharing, action, actionSource);
    }

    /**
     * Checks whether specified user can perform a given layer action
     * on a layer which is the parent layer for the given event
     *
     * @return true if action is available on the event parent layer
     */
    private boolean checkLayerPermissionForEvent(Optional<LayerInfoForPermsCheck> layer, LayerAction action,
                                                 ActionSource actionSource) {
        if (actionSource.isTrusted()) {
            return true;
        }

        if (layer.isEmpty()) {
            return false;
        }

        return canPerformLayerAction(layer.get(), action, actionSource);
    }

    private static PermissionDeniedUserException permissionDeniedException(UserInfo userO, String target,
                                                                           EventOrLayerAction action) {
        return permissionDeniedException(Optional.of(userO), target, action);
    }

    private static PermissionDeniedUserException permissionDeniedException(Optional<UserInfo> userO, String target,
                                                                           EventOrLayerAction action) {
        val msg = target + ", user = " + userO + ", action = " + action;
        return new PermissionDeniedUserException(msg, action);
    }

    private List<EventInfoForPermsCheck> loadInfoForPermsCheck(Optional<UserInfo> userInfo, List<EventFieldsForPermsCheck> eventFields) {
        return userInfo.map(user -> loadInfoForPermsCheck(user, eventFields, Cf.list()))
            .orElseGet(() -> loadInfoForPermsCheck(eventFields, Cf.list()));
    }

    @Override
    public List<EventInfoForPermsCheck> loadInfoForPermsCheck(UserInfo userInfo, List<EventFieldsForPermsCheck> eventFields,
                                                              List<EventParticipants> eventParticipants) {
        return loadInfoForPermsCheck(Optional.of(userInfo), Cf.toList(eventFields), Cf.toList(eventParticipants));
    }

    @Override
    public List<EventInfoForPermsCheck> loadInfoForPermsCheck(List<EventFieldsForPermsCheck> eventFields, List<EventParticipants> eventParticipants) {
        return loadInfoForPermsCheck(Optional.empty(), Cf.toList(eventFields), Cf.toList(eventParticipants));
    }

    private ListF<EventInfoForPermsCheck> loadInfoForPermsCheck(Optional<UserInfo> userInfo, ListF<EventFieldsForPermsCheck> eventFields,
                                                                ListF<EventParticipants> eventParticipants) {
        ListF<Long> eventIds = eventFields.map(EventFieldsForPermsCheck::getId);

        Tuple2List<Long, ParticipantId> organizers;
        if (eventParticipants.isNotEmpty()) {
            organizers = Cf2.flatBy2(eventParticipants
                    .toTuple2List(EventParticipants::getEventId, EventParticipants::getOrganizerIdWithInconsistent));
        } else {
            organizers = Cf2.flatBy2(eventInvitationManager.findOrganizersByEventIds(eventIds));
        }

        MapF<Long, ParticipantId> organizersByEventId = organizers.toMap();

        Tuple2List<Long, PassportUid> userOrgs = Cf2.flatBy2(organizers.map2(ParticipantId.getUidIfYandexUserF()));

        Tuple2List<Long, LayerInfoForPermsCheck> eventIdPrimaryLayers =
                eventLayerDao.findPrimaryLayersForPersCheckByEventIds(eventIds);

        MapF<Long, LayerInfoForPermsCheck> foundPrimaryLayerByEventId = eventIdPrimaryLayers.toMap();

        Tuple2List<Long, LayerInfoForPermsCheck> userOrgsLayers = eventLayerDao.findOrganizerLayersByEventIdsAndUids(
                userOrgs.filterBy1Not(foundPrimaryLayerByEventId::containsKeyTs));

        MapF<Long, LayerInfoForPermsCheck> userOrgLayerByEventId = userOrgsLayers.toMap();

        Tuple2List<Long, ResourceInfoForPermsCheck> eventIdResources;
        if (eventParticipants.isNotEmpty()) {
            eventIdResources = Tuple2List.tuple2List(eventParticipants
                    .flatMap(p -> p.getResourcesForPermsCheck().map(rs -> Tuple2.tuple(p.getEventId(), rs))));
        } else {
            eventIdResources = eventResourceDao.findResourcesForPermsCheckByEventIds(eventIds);
        }

        Function<Long, Option<LayerInfoForPermsCheck>> primaryLayerByEventIdF = id ->
                foundPrimaryLayerByEventId.getO(id)
                        .orElse(() -> userOrgLayerByEventId.getO(id));

        MapF<Long, ListF<ResourceInfoForPermsCheck>> resourcesByEventId = eventIdResources.groupBy1();

        SetF<PassportUid> uidsLetToEditAnyMeeting;
        if (eventParticipants.isNotEmpty()) {
            uidsLetToEditAnyMeeting = eventParticipants.filterMap(ps -> {
                Option<ParticipantInfo> org = ps.getOrganizerWithInconsistent();
                return org.isPresent() && org.get().letParticipantsEdit() ? org.get().getUid() : Option.empty();
            }).unique();
        } else {
            ListF<PassportUid> ytOrganizerUids = userOrgs.get2().filter(PassportUid::isYandexTeamRu);
            uidsLetToEditAnyMeeting = ytOrganizerUids.isNotEmpty()
                    ? userDao.findUsersLetToEditAnyMeeting(ytOrganizerUids).unique()
                    : Cf.set();
        }

        SetF<Long> eventIdsOnUserCreatedLayers;
        SetF<Long> eventIdsOnUserViewSharedLayers;

        UserLayersSharing userLayersSharing = getLayersSharing(userInfo);

        if (userInfo.isPresent()) {
            val userCreatedLayerIds = userLayersSharing.getOwnedLayerIds(!passportAuthDomainsHolder.containsYandexTeamRu());
            val userViewSharedLayerIds = userLayersSharing.getLayerIdsSharedFor(LayerAction.VIEW_EVENT);

            // TODO: find all event layers (including layer creator)
            // TODO: request layers permissions for user
            val eventLayers = eventLayerDao.findEventLayerEventIdsAndLayerIds(
                    eventIds, StreamEx.of(userCreatedLayerIds).append(userViewSharedLayerIds).collect(CollectorsF.toList()));

            eventIdsOnUserCreatedLayers = eventLayers.filterMap(el ->
                    Option.when(userCreatedLayerIds.contains(el.getLayerId()), el.getEventId())).unique();
            eventIdsOnUserViewSharedLayers = eventLayers.filterMap(el ->
                    Option.when(userViewSharedLayerIds.contains(el.getLayerId()), el.getEventId())).unique();

        } else {
            eventIdsOnUserCreatedLayers = Cf.set();
            eventIdsOnUserViewSharedLayers = Cf.set();
        }

        val secretUsersAndKeepers = superSecretsManager.getSuperSecretUsersAndSecretKeepers();

        ListF<PassportUid> secretUsers = secretUsersAndKeepers.get1();
        ListF<PassportUid> notKeptSecretUsers = secretUsersAndKeepers.filterMap(
                t -> Option.when(userInfo.isEmpty() || !t.get2().containsTs(userInfo.get().getUid()), t.get1()));

        Function<ListF<PassportUid>, SetF<Long>> findSecretEventIds = (uids) -> {
            SetF<Long> result = Cf.hashSet();
            if (uids.isNotEmpty()) {
                result.addAll(eventIdPrimaryLayers.filterBy2(l -> secretUsers.containsTs(l.getCreatorUid())).get1());
                result.addAll(userOrgs.filterBy2(secretUsers::containsTs).get1());

                ListF<Long> restIds = eventIds.filterNot(result.containsF());

                if (eventParticipants.isNotEmpty()) {
                    result.addAll(eventParticipants.filterMap(
                            ep -> Option.when(uids.exists(ep::userIsAttendeeWithInconsistent), ep.getEventId())));
                } else {
                    result.addAll(eventUserDao.findUsersAttendingEventIds(secretUsers, restIds));
                }
            }
            return result;
        };
        SetF<Long> secretEventIds = findSecretEventIds.apply(secretUsers);
        SetF<Long> notKeptSecretEventIds = !secretUsers.equals(notKeptSecretUsers)
                ? findSecretEventIds.apply(notKeptSecretUsers)
                : secretEventIds;

        SetF<Long> userAttendingEventIds;
        if (eventParticipants.isNotEmpty() && userInfo.isPresent()) {
            userAttendingEventIds = eventParticipants.filterMap(ps -> Option.when(
                    ps.userIsAttendeeWithInconsistent(userInfo.get().getUid()), ps.getEventId())).unique();

        } else if (userInfo.isPresent()) {
            userAttendingEventIds = Cf.hashSet();
            Option<PassportUid> uid = Option.of(userInfo.get().getUid());
            userAttendingEventIds.addAll(userOrgs.filterBy2(uid.containsF()).get1());

            ListF<Long> restIds = eventIds.filterNot(secretEventIds.containsF());
            userAttendingEventIds.addAll(eventUserDao.findUsersAttendingEventIds(uid, restIds));
        } else {
            userAttendingEventIds = Cf.set();
        }

        final Predicate<EventParticipants> isNotAMeeting = participants -> participants.getResources().isEmpty() &&
                participants.getEventUsers().size() == 1 &&
                participants.getEventUsers().stream().noneMatch(u -> u.isAttendee() || u.isOrganizer());

        final Function<EventParticipants, Stream<PassportUid>> getAttendees = participants -> {
            val eventUsers = participants.getEventUsers().stream();

            if (isNotAMeeting.test(participants)) {
                return eventUsers.map(EventUserWithRelations::getUid);
            } else {
                return eventUsers
                        .filter(u -> u.isOrganizer() || u.isAttendee())
                        .map(EventUserWithRelations::getUid);
            }
        };

        return eventFields.map(event -> {
            val eventId = event.getId();

            final var isEventHasSuperSecretUserIsKeep = userInfo.map(info -> {
                val attendees = StreamEx.of(eventParticipants)
                    .filter(p -> p.getEventId() == eventId)
                    .flatMap(getAttendees)
                    .toImmutableSet();
                return superSecretsManager.hasSuperSecretUserIsKeep(info, attendees);
            }).orElse(false);

            return new EventInfoForPermsCheck(
                eventId, event.getType(), event.getCreatorUid(),
                event.getPermAll(), event.getPermParticipants(), event.isParticipantsInvite(),
                organizersByEventId.getO(eventId),
                organizersByEventId.getO(eventId)
                    .exists(p -> p.isYandexUser() && uidsLetToEditAnyMeeting.containsTs(p.getUid())),
                resourcesByEventId.getOrElse(eventId, Cf.list()),
                primaryLayerByEventIdF.apply(eventId),
                userLayersSharing,
                userAttendingEventIds.containsTs(eventId),
                eventIdsOnUserCreatedLayers.containsTs(eventId),
                eventIdsOnUserViewSharedLayers.containsTs(eventId),
                secretEventIds.containsTs(eventId), notKeptSecretEventIds.containsTs(eventId),
                isEventHasSuperSecretUserIsKeep);
        });
    }

    @Override
    public Map<Long, EventInfoForPermsCheck> loadEventsInfoForPermsCheck(UserInfo user, List<EventWithRelations> events) {
        if (events.isEmpty()) {
            return Map.of();
        }
        Optional<UserInfo> userInfo = Optional.of(user);
        UserLayersSharing sharing = getLayersSharing(userInfo);
        return StreamEx.of(events).toMap(EventWithRelations::getId, event -> loadEventInfoForPermsCheck(userInfo, event, Optional.empty(), sharing));
    }

    @Override
    public EventInfoForPermsCheck loadEventInfoForPermsCheck(UserInfo user, EventWithRelations event) {
        return loadEventInfoForPermsCheck(Optional.of(user), event, Optional.empty(), Optional.empty());
    }

    @Override
    public EventInfoForPermsCheck loadEventInfoForPermsCheck(EventWithRelations event) {
        return loadEventInfoForPermsCheck(Optional.empty(), event, Optional.empty(), Optional.empty());
    }

    @Override
    public EventInfoForPermsCheck loadEventInfoForPermsCheck(UserInfo user, EventWithRelations event,
                                                             UserLayersSharing userLayersSharing) {
        return loadEventInfoForPermsCheck(Optional.of(user), event, Optional.empty(), Optional.of(userLayersSharing));
    }

    @Override
    public EventInfoForPermsCheck loadEventInfoForPermsCheck(EventWithRelations event, UserLayersSharing userLayersSharing) {
        return loadEventInfoForPermsCheck(Optional.empty(), event, Optional.empty(), Optional.of(userLayersSharing));
    }

    @Override
    public EventInfoForPermsCheck loadInfoForPermsCheckByEvents(Optional<UserInfo> userInfo, Event event) {
        return loadInfoForPermsCheckByEvents(userInfo, singletonList(event)).get(0);
    }

    @Override
    public List<EventInfoForPermsCheck> loadInfoForPermsCheckByEvents(Optional<UserInfo> userInfo, List<Event> events) {
        var eventFields = StreamEx.of(events).map(EventFieldsForPermsCheck::fromEvent).toImmutableList();
        return loadInfoForPermsCheck(userInfo, eventFields);
    }

    @Override
    public EventInfoForPermsCheck loadEventInfoForPermsCheck(Optional<UserInfo> user, EventWithRelations event,
                                                             Optional<Layer> detachedLayer,
                                                             Optional<UserLayersSharing> userLayersSharing) {
        UserLayersSharing layerSharing = userLayersSharing.orElseGet(() -> getLayersSharing(user));
        return loadEventInfoForPermsCheck(user, event, detachedLayer, layerSharing);
    }

    private EventInfoForPermsCheck loadEventInfoForPermsCheck(Optional<UserInfo> user, EventWithRelations event,
                                                             Optional<Layer> detachedLayer,
                                                             UserLayersSharing layerSharing) {
        val uid = user.map(UserInfo::getUid);

        val userIsAttendee = uid.stream()
            .anyMatch(event::userIsAttendee);

        val userHasEventLayer = uid.stream()
            .anyMatch(event::existsEventLayerForUser);

        val userHasEventOnLayerSharedWithViewPerms = uid.stream()
            .anyMatch(userUid -> {
                return StreamEx.of(event.getLayers())
                    .append(detachedLayer.stream())
                    .remove(layer -> userUid.equals(layer.getCreatorUid()))
                    .anyMatch(layer -> layerSharing.containsAction(layer, LayerAction.VIEW_EVENT));
            });

        return new EventInfoForPermsCheck(
                event.getId(), event.getEvent().getType(), event.getEvent().getCreatorUid(),
                event.getEvent().getPermAll(), event.getEvent().getPermParticipants(),
                event.getEvent().getParticipantsInvite(),
                event.getParticipants().getOrganizerIdSafe(),
                event.organizerLetToEditAnyMeeting(),
                event.getResourcesForPermsCheck(),
                Option.x(event.getPrimaryLayer().map(LayerInfoForPermsCheck::fromLayer)),
                layerSharing,
                userIsAttendee,
                userHasEventLayer,
                userHasEventOnLayerSharedWithViewPerms,
                hasSuperSecret(event),
                superSecretsManager.hasSuperSecretAndUserIsNotKeeper(user, event),
                user.map(u -> superSecretsManager.hasSuperSecretUserIsKeep(u, event)).orElse(false));
    }

    private static boolean isOwner(UserInfo userInfo, EventInfoForPermsCheck event) {
        return event.getOrganizer().isPresent()
                ? event.getOrganizer().isSome(ParticipantId.yandexUid(userInfo.getUid()))
                : event.getCreatorUid().sameAs(userInfo.getUid()) && !event.getPrimaryLayer()
                .exists(l -> !event.getUserLayersSharing().containsAction(l.getId(), LayerAction.CREATE_EVENT));
    }

    @Override
    public void ensurePermittedOrganizerSetting(UserInfo user, Optional<Event> event, ParticipantsOrInvitationsData invitationsData,
                                                ActionSource actionSource) {
        val organizerEmail = invitationsData.getOrganizerEmail();
        if (!organizerEmail.isPresent()) {
            return;
        }

        val organizerUid = userManager.getUidByEmail(organizerEmail.get());

        if (!isAllowedToSetOrganizer(user, event, invitationsData, actionSource, organizerUid.toOptional())) {
            throw new PermissionDeniedUserException("User is not allowed to set organizer");
        }
        if (!organizerUid.isPresent()) {
            throw new PermissionDeniedUserException("Expected yandex user to be an organizer");
        }
    }

    private boolean isAllowedToSetOrganizer(UserInfo user, Optional<Event> event, ParticipantsOrInvitationsData invitationsData,
                                            ActionSource actionSource, Optional<PassportUid> organizerUid) {
        Supplier<Boolean> containsSpecialResourcesCanNotAdmin = () -> {
            val isYandexUser = user.getUid().isYandexTeamRu();
            val resourcesByEmails = resourceRoutines.findResourcesByEmails(invitationsData.getParticipantEmails());
            val unorganized = StreamEx.of(resourcesByEmails)
                    .map(Tuple2::get2)
                    .flatMap(Option::stream)
                    .filter(r -> !isYandexUser || !ResourceType.organizeableWithoutGroups().containsTs(r.getType()))
                    .toImmutableList();

            return !unorganized.isEmpty() && !canAdminAllResources(user, unorganized);
        };

        if (event.isPresent()) {
            return canChangeOrganizer(user, event.get(), actionSource)
                    && !containsSpecialResourcesCanNotAdmin.get();
        } else {
            if (user.isSuperUser() || organizerUid.equals(Optional.of(user.getUid()))) {
                return true;

            } else if (user.isExternalYt()) {
                return false;

            } else {
                return user.getUid().isYandexTeamRu() && !containsSpecialResourcesCanNotAdmin.get();
            }
        }
    }

    @Override
    public List<Long> getNonBookableResourceIds(UserInfo user, List<Resource> resources) {
        return StreamEx.of(resources)
                .filter(not(user::canBookResource))
                .map(Resource::getId)
                .toImmutableList();
    }

    private void checkUserCanViewEventsForParticipant(PassportUid uid) {
        if (!userManager.getUserInfo(uid).isSuperUser()) {
            throw new PermissionDeniedUserException("ParticipantUid must be empty or user with uid must be a superuser.");
        }
    }

    @Override
    public void ensureMoveAction(UserInfo user, long oldLayerId, long newLayerId, LayerAction oldAction, ActionSource source) {
        val oldLayer = layerRoutines.getLayerById(oldLayerId);
        val newLayer = layerRoutines.getLayerById(newLayerId);

        val oldLayerPermInfo = LayerInfoForPermsCheck.fromLayer(oldLayer);
        val newLayerPermInfo = LayerInfoForPermsCheck.fromLayer(newLayer);

        ensureLayerType(oldLayerPermInfo, LayerType.USER);
        ensureLayerType(newLayerPermInfo, LayerType.USER);

        ensureCanPerformLayerAction(user, oldLayerPermInfo, Optional.empty(), oldAction, source);
        ensureCanPerformLayerAction(user, newLayerPermInfo, Optional.empty(), LayerAction.CREATE_EVENT, source);
    }

    @Override
    public EnumSet<EventAction> getEventPermissions(UserInfo user, EventInfoForPermsCheck event, ActionSource actionSource) {
        val set = EnumSet.noneOf(EventAction.class);

        if (canDeleteEvent(user, event, actionSource)) {
            set.add(EventAction.DELETE);
        }

        if (canViewEvent(user, event, actionSource)) {
            set.add(EventAction.VIEW);
        }

        if (canEditEvent(user, event, actionSource)) {
            set.add(EventAction.EDIT);
        }

        if (canInviteToEvent(user, event, actionSource)) {
            set.add(EventAction.INVITE);
        }

        if (canChangeOrganizer(user, event, actionSource)) {
            set.add(EventAction.CHANGE_ORGANIZER);
        }

        return set;
    }

    @Override
    public EnumSet<EventAction> getEventPermissions(EventInfoForPermsCheck event, ActionSource actionSource) {
        val set = EnumSet.noneOf(EventAction.class);

        if (canDeleteEvent(event, actionSource)) {
            set.add(EventAction.DELETE);
        }

        if (canViewEvent(event, actionSource)) {
            set.add(EventAction.VIEW);
        }

        if (canEditEvent(event, actionSource)) {
            set.add(EventAction.EDIT);
        }

        return set;
    }

    private boolean canAccess(UserInfo user, Resource resource) {
        return canAccess(user, ResourceInfoForPermsCheck.fromResource(resource));
    }

    private boolean canAccess(UserInfo user, ResourceInfoForPermsCheck resource) {
        val domain = PassportDomain.cons(resource.getDomain());
        val clientDomain = resourceRoutines.getDomainByUidUnlessPublic(user.getUid());
        return clientDomain.isSome(domain) && user.getResourceTypesCanView().containsTs(resource.getType());
    }

    private UserLayersSharing getLayersSharing(Optional<UserInfo> user) {
        return user
            .map(UserInfo::getUid)
            .map(this::loadLayerSharing)
            .orElseGet(UserLayersSharing::empty);
    }

    @Override
    public UserLayersSharing loadLayerSharing(PassportUid uid) {
        return loadLayerSharing(singleton(uid)).get(uid);
    }

    @Override
    public Map<PassportUid, UserLayersSharing> loadLayerSharing(Set<PassportUid> uids) {
        val result = layerUserDao.findOwnAndInvitedLayerUserPerms(Cf.list(uids));
        return StreamEx.of(result)
            .mapToEntry(Tuple2::get1, Tuple2::get2)
            .mapValues(permSettings -> StreamEx.of(permSettings).toMap(UserLayersSharing.LayerPermissionSettings::getLayerId, UserLayersSharing.LayerPermissionSettings::getActions))
            .mapValues(UserLayersSharing::new)
            .toImmutableMap();
    }
}
