package ru.yandex.calendar.logic.sharing.perm;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import javax.inject.Inject;

import com.google.common.collect.Lists;
import lombok.Value;
import lombok.val;
import one.util.streamex.StreamEx;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectorsF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.event.EventWithRelations;
import ru.yandex.calendar.logic.sharing.participant.ParticipantId;
import ru.yandex.calendar.logic.sharing.participant.ParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.ParticipantKind;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;

public class SuperSecretsManager {
    @Value
    private static class SuperSecret {
        Set<PassportUid> users;
        Set<PassportUid> keepers;

        @Override
        public String toString() {
            return "SuperSecret{" +
                    "users=" + users +
                    ", keepers=" + keepers +
                    '}';
        }
    }

    private final DynamicProperty<ListF<String>> superSecretUsers =
            new DynamicProperty<>("superSecretUsers",
                    Cf.list("volozh;bunina|may;robot-calendar;bunina;tmakarova;tigran;yumorozova",
                            "daniilsh;luimin;nadyag;ysenderov;dmasyuk;vladimiris;krasil;nikslim;dazolot;romanmoroz;" +
                                    "ebikin;alexfedotov|korotich;allabo;sullyone;victoria-vs;robot-calendar")); //
    // CAL-6695

    @Inject
    private UserManager userManager;
    @Inject
    private PassportAuthDomainsHolder passportAuthDomainsHolder;

    private Set<PassportUid> getSuperSecretKeepers(Set<PassportUid> eventAttendees) {
        Set<PassportUid> result = new HashSet<>();
        final List<SuperSecret> superSecretList = extractSuperSecretList();
        for (SuperSecret superSecret : superSecretList) {
            for (PassportUid eventAttendee : eventAttendees) {
                if (superSecret.getUsers().contains(eventAttendee)) {
                    result.addAll(superSecret.getKeepers());
                }
            }
        }
        return result;
    }

    public List<SuperSecret> extractSuperSecretList() {
        List<SuperSecret> superSecretList = new ArrayList<>();
        for (String secretUsersKeepersStr : superSecretUsers.get()) {
            final List<String> secretUsersKeepersParts = Lists.newArrayList(secretUsersKeepersStr.split("\\|"));
            List<PassportUid> secretUsers = new ArrayList<>();
            List<PassportUid> keepers = new ArrayList<>();
            if (!secretUsersKeepersParts.isEmpty()) {
                secretUsers.addAll(extract(Collections.singletonList(secretUsersKeepersParts.get(0))));
                if (secretUsersKeepersParts.size() > 1) {
                    keepers.addAll(extract(Collections.singletonList(secretUsersKeepersParts.get(1))));
                }
            }
            superSecretList.add(new SuperSecret(new HashSet<>(secretUsers), new HashSet<>(keepers)));
        }
        return superSecretList;
    }

    private List<PassportUid> extract(List<String> secretUsersKeepersParts) {
        List<PassportUid> result = new ArrayList<>();
        for (String secretUsersKeepersPart : secretUsersKeepersParts) {
            for (String secretUserStr : secretUsersKeepersPart.split(";")) {
                final Optional<PassportUid> ytUserUidByLogin = userManager.getYtUserUidByLogin(secretUserStr);
                ytUserUidByLogin.ifPresent(result::add);
            }
        }
        return result;
    }

    public boolean hasSuperSecretUserIsKeep(UserInfo userInfo, Set<PassportUid> eventAttendees) {
        val eventSuperSecretsKeepers = getSuperSecretKeepers(eventAttendees);
        return eventSuperSecretsKeepers.contains(userInfo.getUid());
    }

    public boolean hasSuperSecretUserIsKeep(UserInfo userInfo, EventWithRelations event) {
        if (!passportAuthDomainsHolder.containsYandexTeamRu()) {
            return false;
        }

        val participants = event.getParticipants();
        val organizer = participants
                .getOrganizerIdWithInconsistent()
                .filter(user -> user.getKind() == ParticipantKind.YANDEX_USER)
                .map(ParticipantId::getUid);
        val participantUids = participants
                .getParticipantUidsSafeWithInconsistent()
                .plus(organizer);

        val eventUids = StreamEx.of(participantUids)
                .append(event.getPrimaryLayerCreatorUid().stream())
                .distinct()
                .collect(CollectorsF.toMutHashSet());

        return hasSuperSecretUserIsKeep(userInfo, eventUids);
    }

    public boolean hasSuperSecretAndUserIsNotKeeper(Optional<UserInfo> user, EventWithRelations event) {
        if (!passportAuthDomainsHolder.containsYandexTeamRu()) {
            return false;
        }

        ListF<PassportUid> participantUids = event.getParticipants()
                .getParticipantsSafeWithInconsistent().filterMap(ParticipantInfo.getUidF());

        SetF<PassportUid> eventUids = participantUids.plus(Option.x(event.getPrimaryLayerCreatorUid())).unique();

        final List<SuperSecret> superSecretList = extractSuperSecretList();
        Optional<PassportUid> uidOptional = user.map(UserInfo.getUidF());
        PassportUid uid = uidOptional.orElse(null);
        for (SuperSecret superSecret : superSecretList) {
            if (superSecret.getUsers().stream().anyMatch(eventUids::containsTs)
                    && (uid == null || !superSecret.getKeepers().contains(uid))) {
                return true;
            }
        }
        return false;
    }

    public Tuple2List<PassportUid, ListF<PassportUid>> getSuperSecretUsersAndSecretKeepers() {
        if (!passportAuthDomainsHolder.containsYandexTeamRu()) {
            return Tuple2List.tuple2List();
        }
        final List<SuperSecret> superSecretList = extractSuperSecretList();
        final List<Tuple2<PassportUid, ListF<PassportUid>>> tuple2s = new ArrayList<>();
        for (SuperSecret superSecret : superSecretList) {
            for (PassportUid superSecretUser : superSecret.getUsers()) {
                tuple2s.add(new Tuple2<>(superSecretUser, Cf.toList(superSecret.getKeepers())));
            }
        }
        return new Tuple2List<>(tuple2s);
    }
}
