package ru.yandex.calendar.logic.staff;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import com.amazonaws.util.json.JSONArray;
import com.amazonaws.util.json.JSONException;
import com.amazonaws.util.json.JSONObject;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.MapperFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.module.paramnames.ParameterNamesModule;
import lombok.extern.slf4j.Slf4j;
import one.util.streamex.StreamEx;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.micro.yt.entity.YtDepartment;
import ru.yandex.calendar.micro.yt.entity.YtOffice;
import ru.yandex.calendar.micro.yt.entity.YtRoom;
import ru.yandex.calendar.micro.yt.entity.YtUser;
import ru.yandex.calendar.micro.yt.entity.YtUserWithDepartmentIds;
import ru.yandex.mail.cerberus.LocationKey;
import ru.yandex.mail.cerberus.yt.mapper.YtDepartmentMapper;
import ru.yandex.mail.cerberus.yt.mapper.YtOfficeMapper;
import ru.yandex.mail.cerberus.yt.mapper.YtRoomMapper;
import ru.yandex.mail.cerberus.yt.mapper.YtUserMapper;
import ru.yandex.mail.cerberus.yt.staff.dto.StaffDepartmentGroup;
import ru.yandex.mail.cerberus.yt.staff.dto.StaffOffice;
import ru.yandex.mail.cerberus.yt.staff.dto.StaffRoom;
import ru.yandex.mail.cerberus.yt.staff.dto.StaffUser;
import ru.yandex.misc.lang.StringUtils;

@Slf4j
public class StaffV3 {

    private static final String ROOM_FIELDS = "id,is_deleted,is_bookable,type,name,floor.id,floor.number,floor.office.id,additional,capacity,phone,equipment.cork_board,equipment.desk,equipment.game_console,equipment.guest_wifi,equipment.projector,equipment.screen,equipment.seats,equipment.video_conferencing,equipment.voice_conferencing,equipment.marker_board,_meta.modified_at";
    private static final String USER_FIELDS = "id,uid,login,is_deleted,official.is_dismissed,official.is_robot,official.is_homeworker,official.affiliation,department_group.id,department_group.is_deleted,department_group.ancestors.id,department_group.ancestors.is_deleted,work_email,language.ui,groups.group.id,groups.group.url,groups.group.is_deleted,groups.group.ancestors.id,groups.group.ancestors.is_deleted,environment.timezone,location.office.id,location.table.floor.number,name.first.ru,name.last.ru,name.first.en,name.last.en,official.position,name.middle,personal.gender,work_phone,phones.type,phones.number,phones.is_main,cars,_meta.modified_at,work_mode";
    private static final String DEPARTMENT_FIELDS = "id,is_deleted,url,name,type,department.is_deleted,department.name.full,department.id,department.url,department.heads.person.uid,ancestors.is_deleted,ancestors.department.id,ancestors.department.level,ancestors.department.heads.person.uid,_meta.modified_at";
    private static final String OFFICE_FIELDS = "id,name.en,name.ru,code,city.name.en,city.name.ru,timezone,is_deleted,_meta.modified_at";


    private final String staffUrl;
    private final StaffHttpProvider provider;
    ObjectMapper objectMapper = new ObjectMapper()
            .registerModule(new ParameterNamesModule())
            .registerModule(new Jdk8Module())
            .registerModule(new JavaTimeModule())
            .enable(MapperFeature.ACCEPT_CASE_INSENSITIVE_ENUMS)
            .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);


    YtUserMapper ytUserMapper = new YtUserMapper();
    YtDepartmentMapper ytDepartmentMapper = new YtDepartmentMapper();
    YtRoomMapper ytRoomMapper = new YtRoomMapper();
    YtOfficeMapper ytOfficeMapper = new YtOfficeMapper();

    public StaffV3(String staffUrl, StaffHttpProvider provider) {
        this.staffUrl = staffUrl;
        this.provider = provider;
    }

    public int totalEntries(String requestString) throws IOException, JSONException {
        JSONObject json = new JSONObject(
                provider.doRequest(staffUrl + "/v3/" + requestString, 1, 1, "", "id"));
        return json.getInt("total");
    }

    private <T> List<T> parseHttpResponse(String response, Class<T> tClass) throws JSONException {
        JSONObject json = new JSONObject(response);
        JSONArray items = json.getJSONArray("result");
        List<T> result = new ArrayList<>();
        for (int i = 0; i < items.length(); i++) {
            try {
                result.add(objectMapper.readValue(items.getString(i), tClass));
            }catch (Exception e){
                log.warn("failed to parse json from staff", e);
            }
        }
        return result;
    }


    public List<YtUserWithDepartmentIds> persons(
            int limit, int page, String query, boolean isDeleted) throws IOException, JSONException {
        var response = provider.doRequest(
                staffUrl + "/v3/persons?_nopage=1&" + (isDeleted ? "is_deleted=true&" : ""),
                limit,
                page,
                query,
                USER_FIELDS);
        return StreamEx.of(parseHttpResponse(response, StaffUser.class))
                .map(this::convertUser).toList();
    }

    public List<YtDepartment> groups(
            int limit, int page, String query, boolean isDeleted) throws IOException, JSONException {
        var response = provider.doRequest(
                staffUrl + "/v3/groups?_nopage=1&" + (isDeleted ? "is_deleted=true&" : ""),
                limit,
                page,
                query,
                DEPARTMENT_FIELDS);
        return StreamEx.of(parseHttpResponse(response, StaffDepartmentGroup.class))
                .map(this::convertDepartment).toList();

    }

    public List<YtRoom> rooms(int limit, int page, String query, boolean isDeleted) throws IOException, JSONException {
        var response = provider.doRequest(
                staffUrl + "/v3/rooms?_nopage=1&" + (isDeleted ? "is_deleted=true&" : ""),
                limit,
                page,
                query,
                ROOM_FIELDS);
        return StreamEx.of(parseHttpResponse(response, StaffRoom.class)).map(this::convertRoom).toList();
    }

    public List<YtOffice> offices(int limit, int page, String query, boolean isDeleted) throws IOException, JSONException {
        var response = provider.doRequest(
                staffUrl + "/v3/offices?_nopage=1&" + (isDeleted ? "is_deleted=true&" : ""),
                limit,
                page,
                query,
                OFFICE_FIELDS);
        return StreamEx.of(parseHttpResponse(response, StaffOffice.class)).map(this::convertOffice).toList();
    }

    public Option<String> getTelegramLogin(String staffLogin) throws IOException {
        var response = provider.doRequest(
                staffUrl + "/v3/persons?_nopage=1",
                Cf.map("login", staffLogin, "_fields", "telegram_accounts"));

        try {
            String result = parseTelegramAccount(response);
            if (StringUtils.isBlank(result)) {
                return Option.empty();
            }
            return Option.of(result);
        } catch (JSONException e) {
            return Option.empty();
        }
    }

    private String parseTelegramAccount(String response) throws JSONException {
        JSONObject json = new JSONObject(response);
        JSONObject json1 = json.getJSONArray("result").getJSONObject(0);
        JSONArray json2 = json1.getJSONArray("telegram_accounts");
        return json2.getJSONObject(0).getString("value");

    }

    private YtUserWithDepartmentIds convertUser(StaffUser staffUser) {
        var user = ytUserMapper.mapToUser(staffUser);
        return new YtUserWithDepartmentIds(
                new YtUser(user.getUid(), user.getLogin(), user.getInfo().get()),
                staffUser.departmentIds().toImmutableSet());
    }

    private YtDepartment convertDepartment(StaffDepartmentGroup staffDepartmentGroup) {
        var group = ytDepartmentMapper.mapToGroup(staffDepartmentGroup);
        return new YtDepartment(group.getId(), group.getName(), group.isActive(), group.getInfo().get());
    }

    private YtRoom convertRoom(StaffRoom staffRoom) {
        var room = ytRoomMapper.mapToResource(staffRoom);
        return new YtRoom(
                room.getId(),
                room.getName(),
                room.getLocation().map(LocationKey::getId),
                room.isActive(),
                room.getInfo().get());
    }

    private YtOffice convertOffice(StaffOffice staffOffice) {
        var office = ytOfficeMapper.mapToLocation(staffOffice);
        return new YtOffice(office.getId(), office.getName(), office.getInfo().get());
    }
}
