package ru.yandex.calendar.logic.staff;

import java.time.Duration;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.PostConstruct;
import javax.inject.Inject;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.google.common.collect.Sets;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.calendar.logic.staff.dao.GroupsDao;
import ru.yandex.calendar.logic.staff.dao.UsersDao;
import ru.yandex.calendar.micro.yt.StaffCache;
import ru.yandex.calendar.micro.yt.entity.YtDepartment;
import ru.yandex.calendar.micro.yt.entity.YtUser;
import ru.yandex.calendar.micro.yt.entity.YtUserWithDepartmentIds;
import ru.yandex.mail.cerberus.GroupId;
import ru.yandex.mail.cerberus.Uid;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.function.Function.identity;
import static ru.yandex.mail.micronaut.common.CerberusUtils.mapToList;

@Slf4j
public class YtStaffCache implements StaffCache {

    @lombok.Value
    @AllArgsConstructor
    private static class CacheData {
        private static final CacheData EMPTY = new CacheData(emptyList(), emptyMap(), emptyMap(), emptyMap(), emptyMap(),
                emptyMap());

        List<YtUser> users;
        Map<Uid, Set<GroupId>> userDepartmentIds;
        Map<String, Set<YtUser>> userByEmail;
        Map<String, YtUser> userByLogin;
        Map<Uid, YtUser> userByUid;
        Map<GroupId, YtDepartment> departmentById;

        public CacheData(List<YtUserWithDepartmentIds> users, Collection<YtDepartment> departments) {
            this.users = mapToList(users, YtUserWithDepartmentIds::getUser);
            this.userDepartmentIds = StreamEx.of(users).toMap(
                    user -> user.getUser().getUid(),
                    YtUserWithDepartmentIds::getDepartmentIds,
                    Sets::union
            );
            this.userByEmail = StreamEx.of(this.users)
                    .mapToEntry(user -> user.getInfo().getWorkEmail(), identity())
                    .grouping(Collectors.toSet());
            this.userByLogin = new HashMap<>(); // GREG-899: Календарь не может подняться из-за двух uid у одного юзера
            this.users.forEach(u -> this.userByLogin.put(u.getLogin(), u));
            this.userByUid = new HashMap<>();
            this.users.forEach(u -> this.userByUid.put(u.getUid(), u));
            this.departmentById = StreamEx.of(departments).toMap(
                    YtDepartment::getId,
                    identity(),
                    (d1, d2) -> d2
            );
        }

        public static CacheData empty() {
            return EMPTY;
        }
    }

    @Autowired
    UsersDao usersDao;
    @Autowired
    GroupsDao groupsDao;

    private final ScheduledExecutorService executor;
    private volatile CacheData data;

    @Inject
    public YtStaffCache() {
        this.executor = new ScheduledThreadPoolExecutor(1);
        this.data = CacheData.empty();
    }

    private CacheData fetchData() throws JsonProcessingException {
        MasterSlaveContextHolder.PolicyHandle handle = MasterSlaveContextHolder.push(MasterSlavePolicy.R_SM);
        try {
            final long start = System.currentTimeMillis();
            log.info("Start to fetch data from db");

            var users = usersDao.getAll();

            var departments = groupsDao.getAll();

            log.debug("Data fetch complete; users={}; departments={}; took {}ms", users != null ? users.size() : null,
                    departments != null ? departments.size() : null,
                    System.currentTimeMillis() - start);
            return new CacheData(users, departments);
        } finally {
            handle.popSafely();
        }
    }


    // TODO: use @PostConstruct after https://github.com/micronaut-projects/micronaut-core/issues/3124 fix
    public void onStart() {
        try {
            executor.scheduleWithFixedDelay(() -> {
                try {
                    data = fetchData();
                } catch (Exception e) {
                    log.error("Failed to fetch data from db", e);
                }
            }, Duration.ZERO.getSeconds(), Duration.ofMinutes(10).getSeconds(), TimeUnit.SECONDS);
        } catch (Exception e) {
            log.error("Failed to fetch data from db", e);
            throw e;
        }
    }

    @PostConstruct
    public void initialize() {
        onStart();
    }

    @Override
    public List<YtUser> getUsers() {
        return data.getUsers();
    }

    @Override
    public Set<YtUser> getUsersByEmail(String email) {
        return data.getUserByEmail().getOrDefault(email, emptySet());
    }

    @Override
    public Optional<YtUser> getUserByLogin(String login) {
        return Optional.ofNullable(data.getUserByLogin().get(login));
    }

    @Override
    public Optional<YtUser> getUserByUid(Uid uid) {
        return Optional.ofNullable(data.getUserByUid().get(uid));
    }

    @Override
    public Optional<YtDepartment> getDepartmentById(GroupId id) {
        return Optional.ofNullable(data.getDepartmentById().get(id));
    }

    @Override
    public Set<GroupId> getUserDepartmentIds(Uid uid) {
        return data.getUserDepartmentIds().getOrDefault(uid, emptySet());
    }
}
