package ru.yandex.calendar.logic.staff.dao;

import java.sql.PreparedStatement;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import com.fasterxml.jackson.core.JsonProcessingException;
import lombok.SneakyThrows;
import lombok.Value;
import one.util.streamex.StreamEx;
import org.springframework.jdbc.core.BatchPreparedStatementSetter;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.calendar.micro.yt.entity.YtDepartment;
import ru.yandex.mail.cerberus.GroupId;
import ru.yandex.mail.cerberus.yt.data.YtDepartmentInfo;

public class GroupsDao extends StaffDao<YtDepartment> {
    @Override
    protected int[] insert(List<YtDepartment> groups) {
        return getJdbcTemplate().batchUpdate(
                "INSERT INTO staff_groups (id, name, active, info) VALUES (?, ?, ?, (?)::JSONB) " +
                        "ON CONFLICT (id) DO UPDATE SET " +
                        "name = excluded.name, active=excluded.active, info=excluded.info",
                new BatchPreparedStatementSetter() {

                    @SneakyThrows
                    public void setValues(PreparedStatement ps, int i) {
                        ps.setLong(1, groups.get(i).getId().getValue());
                        ps.setString(2, groups.get(i).getName());
                        ps.setBoolean(3, groups.get(i).isActive());
                        ps.setString(4, objectMapper.writeValueAsString(groups.get(i).getInfo()));
                    }

                    @Override
                    public int getBatchSize() {
                        return groups.size();
                    }
                });
    }

    @Override
    public int update(List<YtDepartment> entities) {
        var existing = existingIds(StreamEx.of(entities).map(ytDepartment -> ytDepartment.getId().getValue()).toImmutableList());
        insert(StreamEx.of(entities).filter(ytDepartment -> existing.contains(ytDepartment.getId().getValue())).toImmutableList());
        return Arrays.stream(insert(StreamEx.of(entities).filter(ytDepartment -> !existing.contains(ytDepartment.getId().getValue())).toImmutableList())).sum();
    }

    private List<Long> existingIds(List<Long> ids) {
        if(ids.isEmpty()){
            return Cf.list();
        }
        String inSql = String.join(",", Collections.nCopies(ids.size(), "?"));
        return getJdbcTemplate().query(
                String.format(
                        "SELECT id FROM staff_groups " +
                                "WHERE id IN (%s)", inSql),
                (rs, rowNum) -> {
                    return rs.getLong("id");
                },
                ids.toArray()
        );
    }


    @Override
    public List<YtDepartment> getAll(int limit, int offset) throws JsonProcessingException {
        @Value
        class GroupArgs {
            long id;
            String name;
            boolean active;
            String info;
        }
        var args = getJdbcTemplate().query(
                "SELECT * FROM staff_groups ORDER BY id LIMIT (?) OFFSET (?)",
                (rs, rowNum) -> {
                    long id = rs.getLong("id");
                    String name = rs.getString("name");
                    boolean active = rs.getBoolean("active");
                    String info = rs.getString("info");
                    return new GroupArgs(id, name, active, info);
                },
                limit, offset
        );
        List<YtDepartment> groups = new ArrayList<>();
        for (var arg : args) {
            YtDepartmentInfo info = objectMapper.readValue(arg.info, YtDepartmentInfo.class);
            groups.add(new YtDepartment(new GroupId(arg.id), arg.name, arg.active, info));
        }
        return groups;
    }

    @Override
    protected int count() {
        return getJdbcTemplate().query(
                "SELECT COUNT(*) FROM staff_groups",
                (rs, rowNum) -> {
                    return rs.getInt("count");
                }
        ).get(0);
    }
}
