package ru.yandex.calendar.logic.staff.dao;

import java.sql.PreparedStatement;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import com.fasterxml.jackson.core.JsonProcessingException;
import lombok.SneakyThrows;
import lombok.Value;
import one.util.streamex.StreamEx;
import org.springframework.jdbc.core.BatchPreparedStatementSetter;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.calendar.micro.yt.entity.YtOffice;
import ru.yandex.mail.cerberus.LocationId;
import ru.yandex.mail.cerberus.yt.data.YtOfficeInfo;

public class OfficesDao extends StaffDao<YtOffice> {
    @Override
    protected int[] insert(List<YtOffice> offices) {
        return getJdbcTemplate().batchUpdate(
                "INSERT INTO staff_locations (id, name, info) VALUES (?, ?, (?)::JSONB) " +
                        "ON CONFLICT (id) DO UPDATE SET " +
                        "name=excluded.name, info=excluded.info",
                new BatchPreparedStatementSetter() {

                    @SneakyThrows
                    public void setValues(PreparedStatement ps, int i) {
                        ps.setLong(1, offices.get(i).getId().getValue());
                        ps.setString(2, offices.get(i).getName());
                        ps.setString(3, objectMapper.writeValueAsString(offices.get(i).getInfo()));
                    }

                    @Override
                    public int getBatchSize() {
                        return offices.size();
                    }
                });
    }

    @Override
    public int update(List<YtOffice> entities) {
        var existing = existingIds(StreamEx.of(entities).map(ytOffice -> ytOffice.getId().getValue()).toImmutableList());
        insert(StreamEx.of(entities).filter(ytOffice -> existing.contains(ytOffice.getId().getValue())).toImmutableList());
        return Arrays.stream(insert(StreamEx.of(entities).filter(ytOffice -> !existing.contains(ytOffice.getId().getValue())).toImmutableList())).sum();
    }

    private List<Long> existingIds(List<Long> ids) {
        if(ids.isEmpty()){
            return Cf.list();
        }
        String inSql = String.join(",", Collections.nCopies(ids.size(), "?"));
        return getJdbcTemplate().query(
                String.format(
                        "SELECT id FROM staff_locations " +
                                "WHERE id IN (%s)", inSql),
                (rs, rowNum) -> {
                    return rs.getLong("id");
                },
                ids.toArray()
        );
    }

    @Override
    public List<YtOffice> getAll(int limit, int offset) throws JsonProcessingException {
        @Value
        class OfficeArgs {
            long id;
            String name;
            String info;
        }
        var args = getJdbcTemplate().query(
                "SELECT * FROM staff_locations ORDER BY id LIMIT (?) OFFSET (?)",
                (rs, rowNum) -> {
                    long id = rs.getLong("id");
                    String name = rs.getString("name");
                    String info = rs.getString("info");
                    return new OfficeArgs(id, name, info);
                },
                limit, offset
        );
        List<YtOffice> offices = new ArrayList<>();
        for (var arg : args) {
            YtOfficeInfo info = objectMapper.readValue(arg.info, YtOfficeInfo.class);
            offices.add(new YtOffice(new LocationId(arg.id), arg.name, info));
        }
        return offices;
    }

    @Override
    protected int count() {
        return getJdbcTemplate().query(
                "SELECT COUNT(*) FROM staff_locations",
                (rs, rowNum) -> {
                    return rs.getInt("count");
                }
        ).get(0);
    }
}
