package ru.yandex.calendar.logic.staff.dao;

import java.sql.PreparedStatement;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

import com.fasterxml.jackson.core.JsonProcessingException;
import lombok.SneakyThrows;
import lombok.Value;
import one.util.streamex.StreamEx;
import org.springframework.jdbc.core.BatchPreparedStatementSetter;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.calendar.micro.yt.entity.YtRoom;
import ru.yandex.mail.cerberus.LocationId;
import ru.yandex.mail.cerberus.ResourceId;
import ru.yandex.mail.cerberus.yt.data.YtRoomInfo;

public class RoomsDao extends StaffDao<YtRoom> {
    @Override
    public int[] insert(List<YtRoom> rooms) {
        return getJdbcTemplate().batchUpdate(
                "INSERT INTO staff_resource (id, name, active, location_id, info) " +
                        "SELECT val.id, val.name, val.active, val.location_id, val.info FROM " +
                        "(VALUES (?, ?, ?, ?, (?)::JSONB)) " +
                        "val (id, name, active, location_id, info) " +
                        "JOIN staff_locations ON val.location_id=staff_locations.id " +
                        "ON CONFLICT (id) DO UPDATE SET " +
                        "name=excluded.name, active=excluded.active, location_id=excluded.location_id, info=excluded.info",

                new BatchPreparedStatementSetter() {

                    @SneakyThrows
                    public void setValues(PreparedStatement ps, int i) {
                        ps.setLong(1, rooms.get(i).getId().getValue());
                        ps.setString(2, rooms.get(i).getName());
                        ps.setBoolean(3, rooms.get(i).isActive());
                        ps.setLong(4, rooms.get(i).getLocationId().get().getValue());
                        ps.setString(5, objectMapper.writeValueAsString(rooms.get(i).getInfo()));
                    }

                    @Override
                    public int getBatchSize() {
                        return rooms.size();
                    }
                });
    }

    @Override
    public int update(List<YtRoom> entities) {
        var existing = existingIds(StreamEx.of(entities).map(ytRoom -> ytRoom.getId().getValue()).toImmutableList());
        insert(StreamEx.of(entities).filter(ytRoom -> existing.contains(ytRoom.getId().getValue())).toImmutableList());
        return Arrays.stream(insert(StreamEx.of(entities).filter(ytRoom -> !existing.contains(ytRoom.getId().getValue())).toImmutableList())).sum();
    }

    private List<Long> existingIds(List<Long> ids) {
        if(ids.isEmpty()){
            return Cf.list();
        }
        String inSql = String.join(",", Collections.nCopies(ids.size(), "?"));
        return getJdbcTemplate().query(
                String.format(
                        "SELECT id FROM staff_resource " +
                                "WHERE id IN (%s)", inSql),
                (rs, rowNum) -> {
                    return rs.getLong("id");
                },
                ids.toArray()
        );
    }

    @Override
    public List<YtRoom> getAll(int limit, int offset) throws JsonProcessingException {
        @Value
        class RoomArgs {
            long id;
            String name;
            boolean active;
            long locationId;
            String info;
        }
        var args = getJdbcTemplate().query(
                "SELECT * FROM staff_resource ORDER BY id LIMIT (?) OFFSET (?)",
                (rs, rowNum) -> {
                    long id = rs.getLong("id");
                    String name = rs.getString("name");
                    boolean active = rs.getBoolean("active");
                    long locationId = rs.getLong("location_id");
                    String info = rs.getString("info");
                    return new RoomArgs(id, name, active, locationId, info);
                },
                limit,
                offset
        );

        List<YtRoom> rooms = new ArrayList<>();
        for (var arg : args) {
            YtRoomInfo info = objectMapper.readValue(arg.info, YtRoomInfo.class);
            rooms.add(
                    new YtRoom(
                            new ResourceId(arg.id),
                            arg.name,
                            Optional.of(new LocationId(arg.locationId)),
                            arg.active,
                            info));
        }
        return rooms;
    }

    @Override
    protected int count() {
        return getJdbcTemplate().query(
                "SELECT COUNT(*) FROM staff_resource",
                (rs, rowNum) -> rs.getInt("count")).get(0);
    }
}
