package ru.yandex.calendar.logic.stat;

import java.io.FileNotFoundException;
import java.io.PrintWriter;
import java.util.Locale;
import java.util.Map.Entry;

import ru.yandex.bolts.collection.Tuple3;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.misc.lang.StringUtils;

/**
 * Collects count and total/maximal run times of an abstract processes (by name).
 * During summarization, dumps this info, including average run times, to a
 * log file (human-readable format) and/or re-writable filename (zabbix format).
 * Usage example: method calls.
 * @author ssytnik
 */
public class CountAndTimeStats extends PeriodicStats<String, Tuple3<Integer, Long, Long>, Long> {
    private String optStatsFileName;
    private boolean dumpToLog;

    public CountAndTimeStats(long dumpIntervalMs, String optStatsFileName, boolean dumpToLog) {
        super(dumpIntervalMs, true);
        this.optStatsFileName = optStatsFileName;
        this.dumpToLog = dumpToLog;
        if (StringUtils.isEmpty(optStatsFileName) && !dumpToLog) {
            logger.warn("CountAndTimeStats.ctor(): neither statis file is specified, nor log dumping is on");
        }
    }

    @Override
    protected Tuple3<Integer, Long, Long> defaultValue(Long time) {
        return Tuple3.tuple(1, time, time);
    }

    @Override
    protected Tuple3<Integer, Long, Long> combine(Tuple3<Integer, Long, Long> oldValue, Long time) {
        return Tuple3.tuple(oldValue.get1() + 1, oldValue.get2() + time, Math.max(oldValue.get3(), time));
    }

    private void outToLog(String s) { if (dumpToLog) { logger.debug(s); } }
    private void outToPw(PrintWriter pw, String s) { if (pw != null) { pw.println(s); } }

    @Override
    public void summarizeStats() {
        boolean dumpToFile = StringUtils.isNotEmpty(optStatsFileName);
        PrintWriter optStatsFileWriter = null;
        try {
            try {
                optStatsFileWriter = (dumpToFile ? new PrintWriter(optStatsFileName) : null);
            } catch (FileNotFoundException e) {
                logger.error("CountAndTimeStats.summarizeStats(): could not open file '" + optStatsFileName + "'");
            }
            outToLog(String.format("=== CountAndTimeStats dump for latest %s ===",
                    AuxDateTime.formatDuration(dumpIntervalMs)));
            outToPw(optStatsFileWriter, String.format("dump.time.utc=%s", AuxDateTime.NOWTS().toString()));
            outToPw(optStatsFileWriter, String.format("dump.interval.ms=%d.00", dumpIntervalMs));
            int totalCallCount = 0;
            long totalSumTime = 0;
            for (Entry<String, Tuple3<Integer, Long, Long>> e : stats.entrySet()) {
                String method = e.getKey();
                Tuple3<Integer, Long, Long> t = e.getValue();
                int callCount = t.get1(); // positive, not zero!
                long sumTime = t.get2();
                long maxTime = t.get3();
                outToLog(String.format(" * [%s] %d x %s (max: %s) = %s",
                        method,
                        callCount,
                        AuxDateTime.formatDuration(sumTime / callCount),
                        AuxDateTime.formatDuration(maxTime),
                        AuxDateTime.formatDuration(sumTime)
                ));
                outToPw(optStatsFileWriter, String.format("%s.count=%d.00", method, callCount));
                outToPw(optStatsFileWriter, String.format( // need US / UK locale to show '.', not ','
                        Locale.US, "%s.avg=%.2f", method, (sumTime / (double) callCount)
                ));
                outToPw(optStatsFileWriter, String.format("%s.max=%d.00", method, maxTime));
                //outToPw(optStatsFileWriter, String.format("%s.sum=%d.00", method, sumTime));
                totalCallCount += callCount;
                totalSumTime += sumTime;
            }
            outToLog(String.format(
                    " = %d occurrence(s), total time = %s",
                    totalCallCount, AuxDateTime.formatDuration(totalSumTime)
            ));
            //outToLog("===");
            stats.clear();
        } finally {
            if (optStatsFileWriter != null) {
                optStatsFileWriter.close();
            }
        }
    }
}
