package ru.yandex.calendar.logic.stat;

import java.util.Map;

import javax.mail.Message;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.calendar.util.base.AuxColl;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.exception.ExceptionUtils;
import ru.yandex.commune.mail.MailMessage;
import ru.yandex.commune.mail.sendmail.Sendmail;
import ru.yandex.commune.monitoring.ExceptionCounterLogAppender;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.net.HostnameUtils;

/**
 * Similar to {@link ExceptionCounterLogAppender}
 * @author ssytnik
 *
 */
public class ExceptionCountStats extends PeriodicStats<String, Integer, Void> {
    private static final Email from = new Email(
        "ExceptionCountStats@" + HostnameUtils.localHostname() // better than SvcRoutines.getSelfUrlHost
    );
    private static final String FROM_PERSONAL = "Calendar exceptions monitoring";
    private static final int TOP_COUNT = 20;
    private Sendmail sendmail;
    private Email[] recipients;

    public ExceptionCountStats(long dumpIntervalMs, Sendmail sendmail, Email[] recipients) {
        super(dumpIntervalMs, true);
        this.sendmail = sendmail;
        this.recipients = recipients;
    }

    @Override
    protected Integer defaultValue(Void value) {
        return 1;
    }

    @Override
    protected Integer combine(Integer oldValue, Void value) {
        return oldValue + 1;
    }

    @Override
    protected void summarizeStats() {
        String intStr = AuxDateTime.formatDuration(dumpIntervalMs);
        if (stats.isEmpty()) {
            logger.debug("*** No exceptions occurred during latest " + intStr + " ***");
        } else {
            // NOTE: statsCopy does not need to be synchronized
            MapF<String, Integer> statsCopy = Cf.x(stats);
            String text = "*** Top " + TOP_COUNT + " exception(s) ***\n" +
                    statsCopy.entrySet()
                            .takeSortedByDesc(Map.Entry::getValue, TOP_COUNT)
                            .map(e -> String.format("[%6d] x %s", e.getValue(), e.getKey()))
                            .mkString("\n");
            logger.info(text);
            if (AuxColl.isSet(recipients)) {
                try {
                    MailMessage msg = MailMessage.empty();
                    msg = msg.withFrom(from, FROM_PERSONAL);
                    for (Email recipient : recipients) {
                        msg = msg.addRecipient(Message.RecipientType.TO, recipient, null);
                    }
                    msg = msg.withSubject("Exceptions statistics for latest " + intStr);
                    msg = msg.withContent(text, "text/plain");
                    msg = msg.withDate(new Instant());
                    sendmail.send(msg);
                } catch (Exception e) {
                    logger.error("Could not send email: " + e, e);
                    ExceptionUtils.rethrowIfTlt(e);
                }
            }
            stats.clear();
        }
    }
}
