package ru.yandex.calendar.logic.stat;

import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ConcurrentSkipListMap;

import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;


/**
 * Aggregates statistics and, time to time, performs its summarization.
 * Thread safe. Once stopped, cannot be restarted.
 * @author ssytnik
 */
public abstract class PeriodicStats<K, V, V0> {
    private Timer timer;

    protected static final Logger logger = LoggerFactory.getLogger(PeriodicStats.class);
    protected ConcurrentMap<K, V> stats;
    protected long dumpIntervalMs;

    protected PeriodicStats(long dumpIntervalMs, boolean sorted) {
        this.stats = sorted ? new ConcurrentSkipListMap<K, V>() : new ConcurrentHashMap<K, V>();
        this.dumpIntervalMs = dumpIntervalMs;
        this.timer = new Timer();
        TimerTask task = new TimerTask() { @Override public void run() { summarizeStats(); } };
        this.timer.schedule(task, dumpIntervalMs, dumpIntervalMs);
    }

    protected abstract V defaultValue(V0 value);

    protected abstract V combine(V oldValue, V0 value);

    public void add(K key, V0 value) {
        boolean success;
        do {
            V old = stats.get(key);
            if (old == null) { // ssytnik: in this situation, 'if' is more safe than (?:)
                success = (null == stats.putIfAbsent(key, defaultValue(value)));
            } else {
                success = stats.replace(key, old, combine(old, value));
            }
        } while (!success);
    }

    protected abstract void summarizeStats();

    public void stop() {
        timer.cancel();
    }
}
