package ru.yandex.calendar.logic.stat;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.calendar.logic.beans.generated.TodoItemFields;
import ru.yandex.calendar.util.db.CalendarJdbcDaoSupport;
import ru.yandex.calendar.util.db.SqlFunction;
import ru.yandex.commune.mapObject.MapField;
import ru.yandex.commune.test.random.RunWithRandomTest;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlExpression;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author Daniel Brylev
 */
public class TodoStatisticsDao extends CalendarJdbcDaoSupport {

    @RunWithRandomTest
    public long countUsersHaveNumberInRangeOfTodoItemsCreatedInInterval(
            int minNumber, int maxNumber, InstantInterval interval)
    {
        SqlCondition todoItemCondition = todoItemCreationTsIn(interval);
        SqlCondition havingCondition = todoItemIdColumnCount().ge(minNumber).and(todoItemIdColumnCount().le(maxNumber));
        return countUsersHaveTodoItems(todoItemCondition, havingCondition);
    }

    @RunWithRandomTest
    public long countUsersHaveMoreThanNumberOfTodoItemsCreatedInInterval(int number, InstantInterval interval) {
        SqlCondition todoItemCondition = todoItemCreationTsIn(interval);
        SqlCondition havingCondition = todoItemIdColumnCount().gt(number);
        return countUsersHaveTodoItems(todoItemCondition, havingCondition);
    }

    @RunWithRandomTest
    public long countUsersHaveTodoItemCreatedInInterval(InstantInterval interval) {
        SqlCondition todoItemCondition = todoItemCreationTsIn(interval);
        return countUsersHaveTodoItems(todoItemCondition);
    }

    @RunWithRandomTest
    public long countUsersHaveTodoItemWithDueTsCreatedInInterval(InstantInterval interval) {
        SqlCondition todoItemCondition = todoItemCreationTsIn(interval).and(todoItemDueTsIsNotNull());
        return countUsersHaveTodoItems(todoItemCondition);
    }

    @RunWithRandomTest
    public long countUsersHaveTodoItemCreatedOrModifiedInInterval(InstantInterval interval) {
        SqlCondition todoItemCondition = todoItemCreationTsIn(interval).or(todoItemLastUpdateTsIn(interval));
        return countUsersHaveTodoItems(todoItemCondition);
    }

    @RunWithRandomTest
    public long countTodoItemsCreatedInInterval(InstantInterval interval) {
        SqlCondition todoItemCondition = todoItemCreationTsIn(interval);
        return countTodoItems(todoItemCondition);
    }

    @RunWithRandomTest
    public long countTodoItemsWithDueTsCreatedInInterval(InstantInterval interval) {
        SqlCondition todoItemCondition = todoItemCreationTsIn(interval).and(todoItemDueTsIsNotNull());
        return countTodoItems(todoItemCondition);
    }

    @RunWithRandomTest
    public long countUsersCreatedTodoListBefore(Instant before) {
        String q = "SELECT COUNT(DISTINCT creator_uid) FROM todo_list WHERE creation_ts < ?";
        return getJdbcTemplate().queryForLong(q, before);
    }

    @RunWithRandomTest
    public long countUsersCreatedMoreThanNumberOfTodoListsBefore(int number, Instant before) {
        String nested =
                "SELECT creator_uid FROM todo_list" +
                " WHERE creation_ts < ?" +
                " GROUP BY creator_uid HAVING COUNT(todo_list.id) > ?";

        String q = "SELECT COUNT(*) FROM (" + nested + ") AS nested";
        return getJdbcTemplate().queryForLong(q, before, number);
    }

    @RunWithRandomTest
    public long countUsersWhoSentTodoListBefore(Instant before) {
        String q = "SELECT COUNT(DISTINCT uid) FROM todo_list_email WHERE sent_ts < ?";
        return getJdbcTemplate().queryForLong(q, before);
    }

    private long countUsersHaveTodoItems(SqlCondition todoItemCondition, SqlCondition havingCondition) {
        String nested =
                "SELECT list.creator_uid FROM todo_item item" +
                " INNER JOIN todo_list list ON list.id = item.todo_list_id" +
                " WHERE " + todoItemCondition.sqlForTable("item") +
                " GROUP BY list.creator_uid HAVING " + havingCondition.sqlForTable("item");

        String q = "SELECT COUNT(*) FROM (" + nested + ") AS nested";
        ListF<?> args = todoItemCondition.args().plus(havingCondition.args());
        return getJdbcTemplate().queryForLong(q, args);
    }

    private long countUsersHaveTodoItems(SqlCondition todoItemCondition) {
        String q = "SELECT COUNT(DISTINCT list.creator_uid) FROM todo_item item" +
                " INNER JOIN todo_list list ON list.id = item.todo_list_id" +
                " WHERE " + todoItemCondition.sqlForTable("item");
        return getJdbcTemplate().queryForLong(q, todoItemCondition.args());
    }

    private long countTodoItems(SqlCondition todoItemCondition) {
        String q = "SELECT COUNT(*) FROM todo_item WHERE " + todoItemCondition.sql();
        return getJdbcTemplate().queryForLong(q, todoItemCondition.args());
    }

    private static SqlCondition todoItemCreationTsIn(InstantInterval interval) {
        return fieldValueInInterval(TodoItemFields.CREATION_TS, interval);
    }

    private static SqlCondition todoItemLastUpdateTsIn(InstantInterval interval) {
        return fieldValueInInterval(TodoItemFields.LAST_UPDATE_TS, interval);
    }

    private static SqlCondition fieldValueInInterval(MapField<Instant> field, InstantInterval interval) {
        return field.ge(interval.getStart()).and(field.lt(interval.getEnd()));
    }

    private static SqlCondition todoItemDueTsIsNotNull() {
        return TodoItemFields.DUE_TS.column().isNotNull();
    }

    private static SqlExpression todoItemIdColumnCount() {
        return SqlFunction.count(TodoItemFields.ID.column());
    }
}
