package ru.yandex.calendar.logic.suggest;

import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.ReadableInstant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author dbrylev
 */
public class FreeInterval extends DefaultObject {
    private final InstantInterval interval;
    private final Option<LocalDate> dueDate;

    public FreeInterval(ReadableInstant start, ReadableInstant end, LocalDate dueDate) {
        this(new InstantInterval(start, end), Option.of(dueDate));
    }

    public FreeInterval(ReadableInstant start, ReadableInstant end, Option<LocalDate> dueDate) {
        this(new InstantInterval(start, end), dueDate);
    }

    public FreeInterval(InstantInterval interval, Option<LocalDate> dueDate) {
        this.interval = interval;
        this.dueDate = dueDate;
    }

    public Instant getStart() {
        return interval.getStart();
    }

    public Instant getEnd() {
        return interval.getEnd();
    }

    public InstantInterval getInterval() {
        return interval;
    }

    public Option<LocalDate> getDueDate() {
        return dueDate;
    }

    public static Function<FreeInterval, InstantInterval> getIntervalF() {
        return new Function<FreeInterval, InstantInterval>() {
            public InstantInterval apply(FreeInterval i) {
                return i.getInterval();
            }
        };
    }

    public FreeInterval withInterval(InstantInterval interval) {
        return new FreeInterval(interval, dueDate);
    }

    public static Function<FreeInterval, Instant> getStartF() {
        return i -> i.getInterval().getStart();
    }

    public static Function<FreeInterval, Instant> getEndF() {
        return i -> i.getInterval().getEnd();
    }

    public static Function<InstantInterval, FreeInterval> fromIntervalWithNoDueDateF() {
        return new Function<InstantInterval, FreeInterval>() {
            public FreeInterval apply(InstantInterval i) {
                return new FreeInterval(i, Option.<LocalDate>empty());
            }
        };
    }

    public static Function<LocalTimeOverlap, FreeInterval> fromLocalTimeOverlapForDateF(
            final LocalDate date, final DateTimeZone tz)
    {
        return o -> new FreeInterval(
                o.getStartDateTime(date, tz),
                o.getEndDateTime(date, tz),
                EventRoutines.convertDueTsToUntilDate(o.getStartDateTime(o.getSinceDate(), tz).toInstant(), tz));
    }

    public static Function<FreeInterval, Option<LocalDate>> getDueDateF() {
        return new Function<FreeInterval, Option<LocalDate>>() {
            public Option<LocalDate> apply(FreeInterval i) {
                return i.getDueDate();
            }
        };
    }

    public static Function1B<FreeInterval> overlapsF(InstantInterval interval) {
        return i -> i.getInterval().overlaps(interval);
    }
}
