package ru.yandex.calendar.logic.suggest;

import org.joda.time.DateTimeZone;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author dbrylev
 */
public class FreeIntervalSet {
    private final ListF<FreeInterval> intervals;

    FreeIntervalSet(ListF<FreeInterval> intervals) {
        if (intervals.isNotEmpty()) {
            SuggestUtils.validateSuccessive(intervals.map(FreeInterval.getIntervalF()));
        }
        this.intervals = intervals;
    }

    public static FreeIntervalSet empty() {
        return new FreeIntervalSet(Cf.<FreeInterval>list());
    }

    public static FreeIntervalSet fromIntervalsWithNoDueDate(ListF<InstantInterval> intervals) {
        ListF<InstantInterval> merged = IntervalSet.cons(intervals).getIntervals();

        return new FreeIntervalSet(merged.map(FreeInterval.fromIntervalWithNoDueDateF()));
    }

    public static FreeIntervalSet fromLocalTimeOverlapsForDate(
            LocalTimeOverlaps overlaps, LocalDate date, DateTimeZone tz)
    {
        return new FreeIntervalSet(overlaps.getOverlaps().map(FreeInterval.fromLocalTimeOverlapForDateF(date, tz)));
    }

    public static FreeIntervalSet fromSuccessiveIntervals(ListF<FreeInterval> intervals) {
        return new FreeIntervalSet(intervals);
    }

    public boolean isFreeIn(InstantInterval interval) {
        ListF<FreeInterval> overlaps = getOverlappingIntervals(interval);

        return overlaps.isNotEmpty()
                && SuggestUtils.intervalsContinuous(overlaps.map(FreeInterval.getIntervalF()))
                && !overlaps.first().getInterval().getStart().isAfter(interval.getStart())
                && !overlaps.last().getEnd().isBefore(interval.getEnd());
    }

    public Option<LocalDate> getMinDueDateIn(InstantInterval interval) {
        return getOverlappingIntervals(interval).filterMap(FreeInterval.getDueDateF()).minO();
    }

    public ListF<FreeInterval> getOverlappingIntervals(InstantInterval interval) {
        FreeInterval freeInt = new FreeInterval(interval, Option.<LocalDate>empty());

        return SuggestUtils.findOverlappingForSuccessiveIntervals(intervals, freeInt, FreeInterval::getInterval);
    }

    public FreeIntervalSet cutDuesBefore(final LocalDate due) {
        return new FreeIntervalSet(intervals.filter(new Function1B<FreeInterval>() {
            public boolean apply(FreeInterval i) {
                return !i.getDueDate().isPresent() || !i.getDueDate().get().isBefore(due);
            }
        }));
    }

    public ListF<FreeInterval> cutAndFill(InstantInterval interval) {
        ListF<FreeInterval> intervals = this.intervals.filter(i -> i.getInterval().overlaps(interval));

        if (intervals.isEmpty()) return Cf.list(new FreeInterval(interval, Option.<LocalDate>empty()));

        ListF<FreeInterval> set = Cf.arrayList();

        if (intervals.first().getStart().isAfter(interval.getStart())) {
            set.add(new FreeInterval(interval.getStart(), intervals.first().getStart(), Option.<LocalDate>empty()));
        }
        set.add(intervals.first().withInterval(intervals.first().getInterval().overlap(interval)));

        for (int i = 1; i < intervals.size(); ++i) {
            FreeInterval previous = intervals.get(i - 1);
            FreeInterval current = intervals.get(i);

            if (previous.getEnd().isBefore(current.getStart())) {
                set.add(new FreeInterval(previous.getEnd(), current.getStart(), Option.<LocalDate>empty()));
            }
            set.add(current.withInterval(current.getInterval().overlap(interval)));
        }
        if (intervals.last().getEnd().isBefore(interval.getEnd())) {
            set.add(new FreeInterval(intervals.last().getEnd(), interval.getEnd(), Option.<LocalDate>empty()));
        }
        return set;
    }

    public ListF<FreeInterval> getIntervals() {
        return intervals;
    }

    public static Function<ListF<InstantInterval>, FreeIntervalSet> fromIntervalsWithNoDueDateF() {
        return new Function<ListF<InstantInterval>, FreeIntervalSet>() {
            public FreeIntervalSet apply(ListF<InstantInterval> is) {
                return fromIntervalsWithNoDueDate(is);
            }
        };
    }
}
