package ru.yandex.calendar.logic.suggest;

import java.util.Collections;
import java.util.Comparator;

import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.function.Function2;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author gutman
 */
public class IntervalSet {

    private final ListF<InstantInterval> intervals;

    private IntervalSet(ListF<InstantInterval> intervals) {
        ListF<InstantInterval> filtered = Cf.arrayList();
        for (int i = 0; i < intervals.size() - 1; i++) {
            if (intervals.get(i).getEnd().isAfter(intervals.get(i + 1).getStart())) {
                Validate.fail("Not successive");
            }
            if (!intervals.get(i).getStart().isEqual(intervals.get(i + 1).getStart())) {
                filtered.add(intervals.get(i));
            }
        }
        filtered.addAll(intervals.lastO());
        this.intervals = filtered;
    }

    public static IntervalSet cons(ListF<InstantInterval> intervals) {
        return cons(intervals, Duration.millis(1));
    }

    public static IntervalSet cons(ListF<InstantInterval> intervals, Duration minGapDuration) {
        final long minGapMs = minGapDuration.getMillis();

        ListF<InstantInterval> sorted = intervals.sortedBy(InstantInterval::getStart);
        ListF<InstantInterval> merged = Cf2.merge(sorted,
                new Function2<InstantInterval, InstantInterval, ListF<InstantInterval>>()
        {
            public ListF<InstantInterval> apply(InstantInterval one, InstantInterval two) {

                return one.getEndMillis() + minGapMs > two.getStartMillis()
                        ? Cf.list(one.withEnd(ObjectUtils.max(one.getEnd(), two.getEnd())))
                        : Cf.list(one, two);
            }
        });
        return new IntervalSet(merged);
    }

    public static IntervalSet fromSuccessiveIntervals(ListF<InstantInterval> intervals) {
        return new IntervalSet(intervals);
    }

    public IntervalSet cut(InstantInterval cut) {
        return new IntervalSet(SuggestUtils.cut(intervals, cut));
    }

    public IntervalSet crop(InstantInterval crop) {
        return new IntervalSet(SuggestUtils.crop(intervals, crop));
    }

    public ListF<IntervalSet> splitByDaysSequences(int maxDaysInSequence, DateTimeZone tz) {
        return splitByDaysSequences(maxDaysInSequence, tz, true);
    }

    public ListF<IntervalSet> splitByDaysSequences(int maxDaysInSequence, DateTimeZone tz, boolean forward) {
        Validate.gt(maxDaysInSequence, 0);

        ListF<ListF<InstantInterval>> days = forward ? splitByDays(tz) : splitByDays(tz).reverse();
        ListF<IntervalSet> result = Cf.arrayList();

        while (days.isNotEmpty()) {
            ListF<ListF<InstantInterval>> took = days.take(maxDaysInSequence).takeWhile(ListF::isNotEmpty);
            result.add(cons(took.<InstantInterval>flatten()));
            days = days.drop(took.size()).dropWhile(ListF::isEmpty);
        }
        return result;
    }

    private ListF<ListF<InstantInterval>> splitByDays(DateTimeZone tz) {
        if (intervals.isEmpty()) return Cf.list();

        ListF<InstantInterval> intervals = this.intervals;
        ListF<ListF<InstantInterval>> result = Cf.arrayList();

        for (InstantInterval d : AuxDateTime.splitByDays(new InstantInterval(getStart(), getEnd()), tz)) {

            ListF<InstantInterval> overlaps = intervals.takeWhile(i -> i.overlaps(d));
            result.add(SuggestUtils.crop(overlaps, d.getStart(), d.getEnd()));

            intervals = intervals.dropWhile(i -> !i.getEnd().isAfter(d.getEnd()));
        }
        return result;
    }

    public boolean isEmpty() {
        return intervals.isEmpty();
    }

    public Instant getStart() {
        return intervals.first().getStart();
    }

    public Instant getEnd() {
        return intervals.last().getEnd();
    }

    public InstantInterval getBounds() {
        return new InstantInterval(getStart(), getEnd());
    }

    public boolean contains(Instant instant) {
        return contains(new InstantInterval(instant, instant));
    }

    public boolean contains(InstantInterval interval) {
        // since intervals are successive only rightmost starting not after given starts can contain given
        int pos = Collections.binarySearch(intervals, interval, Comparator.comparing(InstantInterval::getStart));

        if (pos >= 0) {
            return intervals.get(pos).contains(interval);

        } else if (pos < -1) {
            return intervals.get(-pos - 2).contains(interval);
        }
        return false;
    }

    public boolean overlaps(InstantInterval interval) {
        int start = Collections.binarySearch(intervals, interval, Comparator.comparing(InstantInterval::getStart));

        if (start < 0) {
            int prev = -start - 2;
            int next = prev + 1;

            return prev >= 0 && interval.overlaps(intervals.get(prev))
                    || next < intervals.size() && interval.overlaps(intervals.get(next));
        } else {
            return true;
        }
    }

    public ListF<InstantInterval> getOverlappingIntervals(InstantInterval interval) {
        return SuggestUtils.findOverlappingForSuccessiveIntervals(intervals, interval, i -> i);
    }

    public int countEndingBefore(Instant before) {
        InstantInterval interval = new InstantInterval(before, before);

        int pos = Collections.binarySearch(intervals, interval, Comparator.comparing(InstantInterval::getEnd));

        return pos >= 0 ? pos : -pos - 1;
    }

    public ListF<InstantInterval> getIntervals() {
        return intervals;
    }

    public IntervalSet invert(InstantInterval interval) {
        return invert(interval.getStart(), interval.getEnd());
    }

    public IntervalSet invert(Instant start, Instant end) {
        return new IntervalSet(SuggestUtils.invert(intervals, start, end));
    }

    public static IntervalSet empty() {
        return new IntervalSet(Cf.<InstantInterval>list());
    }

}
