package ru.yandex.calendar.logic.suggest;

import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.LocalTime;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionUtils;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author dbrylev
 */
public class RepetitionDays {
    private final LocalDate startDate;
    private final DayTimeInterval interval;

    private final ListF<DayTimeInterval> checkingIntervals;
    private final ListF<DayTimeInterval> muteIntervals;

    private final Repetition repetition;
    private final DateTimeZone tz;

    public RepetitionDays(
            Option<InstantInterval> usersDay, InstantInterval eventInterval,
            Repetition repetition, DateTimeZone tz)
    {
        ListF<InstantInterval> intervals = usersDay.plus(eventInterval);
        Validate.hasSize(1, intervals.map(i -> new LocalDate(i.getStart(), tz)).unique());
        Validate.isTrue(intervals.forAll(i -> !i.getDuration().isLongerThan(Duration.standardDays(1))));

        this.startDate = new LocalDate(eventInterval.getStart(), tz);

        this.interval = new DayTimeInterval(new InstantInterval(
                intervals.map(InstantInterval::getStart).min(),
                intervals.map(InstantInterval::getEnd).max()), tz);

        IntervalSet set = IntervalSet.cons(usersDay.plus(eventInterval));
        this.checkingIntervals = set.getIntervals().map(i -> new DayTimeInterval(i, tz));

        ListF<InstantInterval> is = Option.when(!eventInterval.isEmpty(),
                () -> eventInterval.withStart(eventInterval.getStart().plus(1)));
        this.muteIntervals = usersDay.flatMap(day -> SuggestUtils.cut(is, day).map(i -> new DayTimeInterval(i, tz)));

        this.repetition = repetition;
        this.tz = tz;
    }

    public boolean isDoubleDay() {
        return interval.getEnd().getDay() > 0 && interval.getEnd().getTime().isAfter(LocalTime.MIDNIGHT);
    }

    public InstantInterval getInterval(LocalDate base) {
        return interval.toInstantInterval(base, tz);
    }

    public ListF<InstantInterval> getCheckingIntervals(LocalDate base) {
        return checkingIntervals.map(i -> i.toInstantInterval(base, tz));
    }

    public IntervalSet getMuteIntervalSet(Instant from, Instant to) {
        return IntervalSet.cons(getDaysOverlappingInterval(from, to)
                .flatMap(date -> muteIntervals.map(i -> i.toInstantInterval(date, tz))));
    }

    public DateTimeZone getTz() {
        return tz;
    }

    public Instant getStart() {
        return interval.getStart().toLocalDateTime(startDate).toDateTime(tz).toInstant();
    }

    public LocalDate getStartDate() {
        return startDate;
    }

    public LocalTime getStartTime() {
        return interval.getStart().getTime();
    }

    public Option<Instant> getDue() {
        return repetition.getDueTs();
    }

    public ListF<LocalDate> getDays(int count) {
        return getDaysFromDate(getStartDate(), count);
    }

    public ListF<LocalDate> getDaysFromDate(LocalDate start, int count) {
        ListF<InstantInterval> is = RepetitionUtils.getIntervals(
                toRepetitionInfo(), getInterval(start).getStart(), Option.<Instant>empty(), true, count);

        return is.map(AuxDateTime.instantDateF(tz).compose(InstantInterval::getStart));
    }

    public Instant getNthIntervalEnd(int count) {
        LocalDate base = getDaysFromDate(startDate, count).lastO().getOrElse(startDate);
        return interval.getEnd().toLocalDateTime(base).toDateTime(tz).toInstant();
    }

    public ListF<LocalDate> getDaysOverlappingInterval(Instant from, Instant to) {
        ListF<InstantInterval> is = RepetitionUtils.getIntervals(toRepetitionInfo(), from, Option.of(to), true, 0);
        return is.map(AuxDateTime.instantDateF(tz).compose(InstantInterval::getStart));
    }

    private RepetitionInstanceInfo toRepetitionInfo() {
        return new RepetitionInstanceInfo(interval.toInstantInterval(startDate, tz), tz, Option.of(repetition),
                Cf.list(), Cf.list(), Cf.list());
    }
}
