package ru.yandex.calendar.logic.suggest;

import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.LocalTime;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.bolts.function.Function2;
import ru.yandex.calendar.frontend.webNew.dto.in.SuggestData;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.beans.generated.SettingsYt;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.InaccessibleResourcesRequest;
import ru.yandex.calendar.logic.event.avail.AvailRoutines;
import ru.yandex.calendar.logic.event.avail.AvailabilityIntervals;
import ru.yandex.calendar.logic.event.avail.AvailabilityRequest;
import ru.yandex.calendar.logic.event.avail.UserOrResourceAvailabilityIntervals;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceSet;
import ru.yandex.calendar.logic.resource.OfficeFilter;
import ru.yandex.calendar.logic.resource.OfficeManager;
import ru.yandex.calendar.logic.resource.ResourceFilter;
import ru.yandex.calendar.logic.resource.ResourceInaccessibility;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.resource.ResourceType;
import ru.yandex.calendar.logic.resource.SpecialResources;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.resource.schedule.ResourceDaySchedule;
import ru.yandex.calendar.logic.resource.schedule.ResourceEventsAndReservations;
import ru.yandex.calendar.logic.resource.schedule.ResourceScheduleManager;
import ru.yandex.calendar.logic.user.SettingsInfo;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.commune.holidays.HolidayRoutines;
import ru.yandex.inside.geobase.GeobaseIds;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author gutman
 * @author shinderuk
 */
public class SuggestManager {

    @Autowired
    private AvailRoutines availRoutines;
    @Autowired
    private ResourceScheduleManager resourceScheduleManager;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private DateTimeManager dateTimeManager;
    @Autowired
    private OfficeManager officeManager;
    @Autowired
    private UserManager userManager;
    @Autowired
    private EventRoutines eventRoutines;


    public static final LocalTime PREFERRED_TIME_START = new LocalTime(12, 0);
    public static final LocalTime PREFERRED_TIME = new LocalTime(15, 0);
    public static final LocalTime PREFERRED_TIME_END = new LocalTime(18, 0);

    static final LocalTime MORNING = new LocalTime(10, 0);
    static final LocalTime EVENING = new LocalTime(21, 0);

    private static final int RESOURCES_SEARCH_DAYS = 3;

    private static final int REPEATING_REQUIRED_INSTANCES = 8;
    private static final int REPEATING_LOOKUP_INSTANCES = 5;

    public static final int SUGGEST_SEARCH_DAYS = 60;


    public Suggest singleMeetingSuggest(
            PassportUid uid, ListF<PassportUid> users,
            SuggestData suggestData, SearchInterval searchInterval,
            Option<Long> exceptEventId, DateTimeZone tz, ActionInfo actionInfo)
    {
        Resources resources = findResourcesOkForSuggest(uid, suggestData,
                RepetitionInstanceInfo.noRepetition(suggestData.getEventInterval(tz), tz));

        if (resources.isEmpty()) return Suggest.empty();

        UsersFreeIntervalSet usersFreeIntervals = findAllUsersFreeIntervals(
                uid, users, searchInterval.getInterval(), suggestData.getEventInterval(tz),
                exceptEventId, suggestData.isIgnoreUsersEvents(), Option.empty(), actionInfo);

        boolean forward = searchInterval.isForward();
        ListF<IntervalSet> splits = usersFreeIntervals.getFreeIntervalSet()
                .splitByDaysSequences(RESOURCES_SEARCH_DAYS, tz, forward);

        Function2<IntervalSet, UsersAndResourcesAvailability, AvailableResourcesFinder> finderF = (split, avail) ->
                    new AvailableResourcesFinder(
                            avail, suggestData.getEventStart(tz), suggestData.getDuration(),
                            split.getBounds(), usersFreeIntervals.getMuteIntervalSet());

        if (suggestData.isSameRoomMode()) {
            ListF<AvailableResourcesInOffices> result = Cf.arrayList();
            int requiredOptions = suggestData.getNumberOfOptions().get();

            for (IntervalSet split: splits) {
                UsersAndResourcesAvailability availability = getUsersAndResourcesAvailability(
                        uid, split, suggestData, resources.getFound(), tz, exceptEventId, actionInfo);

                availability = availability.withSelected(resources.getSelected());

                result.addAll(finderF.apply(split, availability).findAvailableResourcesInOffices());

                if (result.size() >= requiredOptions) break;
            }
            if (!forward) {
                Collections.sort(result, Cf2.f(AvailableResourcesInOffices::getStart).andThenNaturalComparator());
            }
            if (result.isNotEmpty()) {
                int overflow = result.size() - (result.size() / requiredOptions) * requiredOptions;
                if (overflow < result.size()) {
                    result = forward ? result.take(result.size() - overflow) : result.rtake(result.size() - overflow);
                }
                Option<Instant> prevStart = Option.of(forward ? searchInterval.getStart() : result.first().getStart());
                Option<Instant> nextStart = Option.of(!forward ? searchInterval.getEnd() : result.last().getEnd());

                return new Suggest(result, prevStart, nextStart);
            }

        } else {
            for (IntervalSet split : splits) {
                UsersAndResourcesAvailability availability = getUsersAndResourcesAvailability(
                        uid, split, suggestData, resources.getFound(), tz, exceptEventId, actionInfo);

                ListF<AvailableResourcesInOffices> found =
                        finderF.apply(split, availability).findAvailableResourcesInOffices();

                if (found.isNotEmpty()) {
                    Option<Instant> prevStart = Option.of(forward ? searchInterval.getStart() : split.getStart());
                    Option<Instant> nextStart = Option.of(!forward ? searchInterval.getEnd() : split.getEnd());

                    return new Suggest(found, prevStart, nextStart);
                }
            }
        }
        Option<Instant> prevStart = Option.when(forward, searchInterval.getStart());
        Option<Instant> nextStart = Option.when(!forward, searchInterval.getEnd());
        return new Suggest(Cf.<AvailableResourcesInOffices>list(), prevStart, nextStart);
    }

    public Suggest repeatingMeetingSuggest(
            PassportUid uid, ListF<PassportUid> users,
            SuggestData suggestData, Repetition repetition, DateTimeZone repetitionTz,
            Option<Long> exceptEventId, DateTimeZone tz, ActionInfo actionInfo)
    {
        Resources resources = findResourcesOkForSuggest(uid, suggestData,
                RepetitionInstanceInfo.create(suggestData.getEventInterval(tz), repetitionTz, repetition));

        if (resources.isEmpty()) return Suggest.empty();

        DateTime eventStart = AuxDateTime.toDateTimeIgnoreGap(suggestData.getEventStart(), repetitionTz);
        DateTime eventEnd = AuxDateTime.toDateTimeIgnoreGap(suggestData.getEventEnd(), repetitionTz);

        if (suggestData.getDuration().isLongerThan(Duration.standardDays(1))) return Suggest.empty();

        Option<InstantInterval> commonUsersDay = findCommonDayIntervalForUsers(users, eventStart);
        InstantInterval eventInterval = new InstantInterval(eventStart, eventEnd);

        RepetitionDays daysRepetition = new RepetitionDays(commonUsersDay, eventInterval, repetition, repetitionTz);

        Tuple2List<InstantInterval, ListF<ResourceAvailability>> availabilities = findResourcesAvailabilityForSomeDays(
                uid, resources.getFound(), daysRepetition, exceptEventId, actionInfo);

        Option<IntervalSet> usersFreeIntervals = !suggestData.isIgnoreUsersEvents()
                ? Option.of(findAllUsersFreeIntervalsForSomeDays(
                        uid, users, daysRepetition, suggestData.getEventInterval(tz), exceptEventId, actionInfo))
                : Option.empty();

        IntervalSet muteIntervals = daysRepetition.getMuteIntervalSet(
                daysRepetition.getStart(),
                daysRepetition.getNthIntervalEnd(REPEATING_LOOKUP_INSTANCES + REPEATING_REQUIRED_INSTANCES));

        ListF<AvailableResourcesInOffices> found = Cf.arrayList();

        for (Tuple2<InstantInterval, ListF<ResourceAvailability>> dayAvailability : availabilities) {
            ListF<OfficeAvailability> officesAvailability = ResourceAvailability.groupByOffices(dayAvailability._2);
            MapF<Long, Integer> numbersOfResources = suggestData.getNumbersOfResources();

            UsersAndResourcesAvailability availability = usersFreeIntervals.isPresent()
                    ? UsersAndResourcesAvailability.onlyUsers(usersFreeIntervals.get())
                    : UsersAndResourcesAvailability.ignoreUsers();

            availability = availability.plusOffices(officesAvailability, numbersOfResources);

            if (suggestData.isSameRoomMode()) {
                availability = availability.withSelected(resources.getSelected());
            }

            AvailableResourcesFinder finder = new AvailableResourcesFinder(
                    availability, suggestData.getEventStart(tz), suggestData.getDuration(),
                    dayAvailability._1, muteIntervals);

            found.addAll(finder.findAvailableResourcesInOffices());
        }
        return new Suggest(found, Option.<Instant>empty(), Option.<Instant>empty());
    }

    public SuggestedDates suggestDates(
            PassportUid uid, ListF<PassportUid> users, SuggestData suggestData,
            DateTimeZone tz, Option<Long> exceptEventId, ActionInfo actionInfo)
    {
        Resources resources = findResourcesOkForSuggest(
                uid, suggestData, RepetitionInstanceInfo.noRepetition(suggestData.getEventInterval(tz), tz));

        if (resources.isEmpty()) return SuggestedDates.empty();

        LocalDate date = suggestData.getDate().get();
        Instant dateStart = date.toDateTimeAtStartOfDay(tz).toInstant();

        Option<InstantInterval> forwardInterval = Option.empty();
        Option<InstantInterval> backwardInterval = Option.empty();

        if (suggestData.getDirection().containsForward()) {
            Instant start = ObjectUtils.max(date.plusDays(1).toDateTimeAtStartOfDay(tz).toInstant(), actionInfo.getNow());
            forwardInterval = Option.of(new InstantInterval(
                    start, new LocalDate(start, tz).plusDays(SUGGEST_SEARCH_DAYS).toDateTimeAtStartOfDay(tz)));
        }
        if (suggestData.getDirection().containsBackward()) {
            Instant max = ObjectUtils.max(
                    date.minusDays(SUGGEST_SEARCH_DAYS).toDateTimeAtStartOfDay(tz).toInstant(), actionInfo.getNow());
            backwardInterval = Option.when(max.isBefore(dateStart), () -> new InstantInterval(max, dateStart));
        }

        InstantInterval searchInterval = new InstantInterval(
                backwardInterval.plus(forwardInterval).map(InstantInterval::getStart).min(),
                backwardInterval.plus(forwardInterval).map(InstantInterval::getEnd).max());

        UsersFreeIntervalSet usersFreeIntervals = findAllUsersFreeIntervals(
                uid, users, searchInterval, suggestData.getEventInterval(tz),
                exceptEventId, suggestData.isIgnoreUsersEvents(), Option.of(tz), actionInfo);

        ListF<LocalDate> dates = Cf.arrayList();

        Option<LocalDate> nextDate = Option.empty();
        Option<LocalDate> prevDate = Option.empty();

        for (InstantInterval interval: backwardInterval.plus(forwardInterval)) {
            boolean isForward = forwardInterval.exists(i -> interval.getStart().isEqual(i.getStart()));

            for (IntervalSet split: usersFreeIntervals.getFreeIntervalSet().crop(interval)
                    .splitByDaysSequences(RESOURCES_SEARCH_DAYS, tz, isForward))
            {
                UsersAndResourcesAvailability availability = getUsersAndResourcesAvailability(
                        uid, split, suggestData, resources.getFound(), tz, exceptEventId, actionInfo);

                AvailableResourcesFinder finder = new AvailableResourcesFinder(
                        availability, suggestData.getEventStart(tz), suggestData.getDuration(),
                        split.getBounds(), usersFreeIntervals.getMuteIntervalSet());

                ListF<LocalDate> found = finder.findDatesWithAvailableResourcesInOffices(tz);

                if (found.isNotEmpty() && isForward) {
                    DateTime end = split.getEnd().toDateTime(tz);
                    nextDate = Option.of(end.toLocalDate().minusDays(end.isEqual(end.withTimeAtStartOfDay()) ? 1 : 0));
                }
                if (found.isNotEmpty() && !isForward) {
                    prevDate = Option.of(new LocalDate(split.getStart(), tz))
                            .filter(new LocalDate(actionInfo.getNow(), tz)::isBefore);
                }
                if (found.isNotEmpty()) {
                    dates.addAll(found);
                    break;
                }
            }
        }
        return new SuggestedDates(dates, prevDate, nextDate);
    }

    public SuggestedResources suggestResources(
            PassportUid uid, ListF<PassportUid> users,
            SuggestData suggestData, DateTimeZone tz, Option<Long> exceptEventId,
            MapF<Long, ? extends Comparator<ResourceInfo>> comparatorByOffice, ActionInfo actionInfo)
    {
        Validate.forAll(suggestData.getOfficeIds(), comparatorByOffice::containsKeyTs);

        Resources resources = findResourcesOkForSuggest(
                uid, suggestData, RepetitionInstanceInfo.noRepetition(suggestData.getEventInterval(tz), tz));

        if (resources.isEmpty()) return SuggestedResources.empty();

        LocalDate date = suggestData.getDate().get();

        InstantInterval suggestInterval = consDayInterval(date, tz);

        ResourcesTimeline timeline = getResourcesTimeline(uid, suggestInterval, resources, tz, exceptEventId, actionInfo);

        ListF<InstantInterval> intervalsToLookup = SuggestUtils
                .getIntervalsToLookup(suggestInterval, suggestData.getSelectedOrEventStart(tz), suggestData.getDuration())
                .filter(i -> timeline.getOffices().forAll(o -> o.get2().exists(r -> r.isFreeIn(i))));

        if (intervalsToLookup.isEmpty()) return SuggestedResources.empty();

        InstantInterval mostPreferredInterval = consPreferredTimeInterval(date, tz);

        Instant preferredTime = date.toDateTime(date.isEqual(suggestData.getEventStart().toLocalDate())
                ? suggestData.getEventStart().toLocalTime() : PREFERRED_TIME, tz).toInstant();

        ListF<InstantInterval> selectedCommonIntervals = intervalsToLookup
                .filter(i -> timeline.getSelected().forAll(r -> r.isFreeIn(i)));

        ListF<IntervalSet> usersTimeline = !suggestData.isIgnoreUsersEvents()
                ? getUsersTimeline(uid, users, suggestInterval, exceptEventId, actionInfo)
                : Cf.list();

        int numberOfOptions = suggestData.getNumberOfOptions().getOrElse(5);

        MapF<Long, ListF<ResourceTimeline>> takenByOfficeId = Cf.x(new HashMap<Long, ListF<ResourceTimeline>>() {{
            timeline.getOffices().forEach((id, rs) -> put(id, Cf.arrayListWithCapacity(numberOfOptions)));
            timeline.getSelected().forEach(r -> get(r.getOfficeId()).add(r));
        }});
        MapF<Long, SetF<ResourceTimeline>> remainByOfficeId = Cf.x(new HashMap<Long, SetF<ResourceTimeline>>() {{
            timeline.getOffices().forEach((id, rs) -> put(id, Cf.toHashSet(rs)));
            timeline.getSelected().forEach(r -> get(r.getOfficeId()).removeTs(r));
        }});

        Option<Instant> selectedStart = suggestData.getSelectedStart(tz).orElse(Option.when(
                !usersTimeline.exists(t -> t.overlaps(suggestData.getEventInterval(tz))), suggestData.getEventStart(tz)));

        ListF<ListF<InstantInterval>> groups = Cf2.splitSorted(
                intervalsToLookup, Comparator.<InstantInterval>comparingInt(i -> 0)
                        .thenComparing(i -> selectedStart.isSome(i.getStart()))
                        .thenComparing(i -> i.getEnd().isAfter(actionInfo.getNow()))
                        .thenComparing(i -> mostPreferredInterval.contains(i.getStart()))
                        .thenComparing(i -> -usersTimeline.count(is -> is.overlaps(i)))
                        .reversed());

        for (ListF<InstantInterval> intervals: groups.map(is -> is.sorted(
                Comparator.comparing(selectedCommonIntervals::containsTs).reversed()
                        .thenComparing(i -> Math.abs(i.getStartMillis() - preferredTime.getMillis())))))
        {
            for (long officeId: timeline.getOffices().get1()) {
                ListF<ResourceTimeline> taken = takenByOfficeId.getTs(officeId);
                SetF<ResourceTimeline> remain = remainByOfficeId.getTs(officeId);
                MapF<ResourceTimeline, Integer> current = Cf.hashMapWithExpectedSize(remain.size());

                remain.iterator().filterMap(r -> Option.of(r.countFree(intervals))
                        .filterMap(s -> Option.when(s > 0, Tuple2.tuple(r, s)))).forEachRemaining(current::put);

                Function<Map.Entry<ResourceTimeline, Integer>, ResourceInfo> resourceF = e -> e.getKey().getResource();
                Comparator<ResourceInfo> comparator = comparatorByOffice.getOrThrow(officeId);

                for (InstantInterval interval: intervals) {
                    if (current.isEmpty() || taken.size() >= numberOfOptions) break;

                    ListF<Map.Entry<ResourceTimeline, Integer>> found;

                    if (selectedStart.isSome(interval.getStart())) {
                        found = current.entrySet().iterator()
                                .filter(e -> e.getKey().isFreeIn(interval))
                                .takeSorted(resourceF.andThen(comparator), numberOfOptions - taken.size());
                    } else {
                        found = current.entrySet().iterator()
                                .filter(e -> e.getKey().isFreeIn(interval))
                                .maxO(Comparator.<Map.Entry<ResourceTimeline, Integer>>comparingInt(Map.Entry::getValue)
                                        .thenComparing(resourceF.andThen(comparator.reversed())));
                    }
                    found.forEach(f -> {
                        taken.add(f.getKey());
                        remain.removeTs(f.getKey());
                        current.removeTs(f.getKey());
                    });
                }
            }
        }
        return new SuggestedResources(takenByOfficeId.values().map(t -> new OfficeTimeline(t.first().getOffice(), t)));
    }

    private ResourcesTimeline getResourcesTimeline(
            PassportUid uid, InstantInterval suggestInterval,
            Resources resources, DateTimeZone tz, Option<Long> exceptEventId, ActionInfo actionInfo)
    {
        InstantInterval timelineInterval = AuxDateTime.expandToDaysInterval(suggestInterval, tz);

        ListF<ResourceTimeline> resourcesTimeline = resourceScheduleManager
                .getResourceScheduleDataForInterval(
                        Option.of(uid), resources.getFound(), timelineInterval, tz, exceptEventId, actionInfo)
                .map((resource, events) -> {
                    ListF<InstantInterval> busyIntervals = mergeBookingRestrictionIntervals(
                            resource, events.getMergedIntervals(), suggestInterval, actionInfo.getNow());

                    ListF<InstantInterval> freeIntervals = SuggestUtils.invert(
                            busyIntervals, suggestInterval.getStart(), suggestInterval.getEnd());

                    return new ResourceTimeline(resource, IntervalSet.fromSuccessiveIntervals(freeIntervals), events);
                });

        Tuple2List<Long, ListF<ResourceTimeline>> officesTimeline = resourcesTimeline
                .groupBy(ResourceTimeline::getOfficeId).entries();

        ListF<ResourceTimeline> selectedResourcesTimeline = resources.getSelected().map(Cf2.f(ResourceInfo::getResourceId)
                .andThen(Cf2.f(resourcesTimeline.toMapMappingToKey(ResourceTimeline::getResourceId)::getOrThrow)));

        return new ResourcesTimeline(resourcesTimeline, selectedResourcesTimeline, officesTimeline);
    }

    private ListF<IntervalSet> getUsersTimeline(
            PassportUid uid, ListF<PassportUid> users,
            InstantInterval suggestInterval, Option<Long> exceptEventId, ActionInfo actionInfo)
    {
        ListF<UserOrResourceAvailabilityIntervals> intervals = availRoutines.getAvailabilityIntervalss(
                uid, users.map(UidOrResourceId.userF()),
                AvailabilityRequest.interval(suggestInterval).excludeEventId(exceptEventId), actionInfo);

        return intervals.map(is -> IntervalSet.cons(is.getIntervalsO().flatMap(AvailabilityIntervals::getIntervals)));
    }

    private UsersAndResourcesAvailability getUsersAndResourcesAvailability(
            PassportUid uid, IntervalSet usersFreeIntervals, SuggestData suggestData,
            ListF<ResourceInfo> resources, DateTimeZone tz, Option<Long> exceptEventId, ActionInfo actionInfo)
    {
        InstantInterval suggestInterval = new InstantInterval(usersFreeIntervals.getStart(), usersFreeIntervals.getEnd());

        Tuple2List<ResourceInfo, ResourceEventsAndReservations> schedule = resourceScheduleManager
                .getResourceScheduleDataForInterval(Option.of(uid), resources, suggestInterval, tz, exceptEventId, actionInfo);

        Tuple2List<ResourceInfo, FreeIntervalSet> resourcesFreeIntervals = schedule.toTuple2List((resource, events) -> {
            ListF<InstantInterval> busyIntervals = mergeBookingRestrictionIntervals(
                    resource, events.getMergedIntervals(), suggestInterval, actionInfo.getNow());

            ListF<InstantInterval> freeIntervals = SuggestUtils.invert(
                    busyIntervals, suggestInterval.getStart(), suggestInterval.getEnd());

            return Tuple2.tuple(resource, FreeIntervalSet.fromSuccessiveIntervals(
                    freeIntervals.map(i -> new FreeInterval(i, Option.empty()))));
        });

        ListF<ResourceAvailability> resourcesAvailability = resourcesFreeIntervals.map(ResourceAvailability.consF());

        ListF<OfficeAvailability> officesAvailability = ResourceAvailability.groupByOffices(resourcesAvailability);
        MapF<Long, Integer> numbersOfResources = suggestData.getNumbersOfResources();

        return UsersAndResourcesAvailability.onlyUsers(usersFreeIntervals).plusOffices(officesAvailability, numbersOfResources);
    }

    private Resources findResourcesOkForSuggest(
            PassportUid uid, SuggestData suggestRequest, RepetitionInstanceInfo repetitionInfo)
    {
        Tuple2List<Long, ResourceFilter> officeFilters = suggestRequest.getOfficeIds()
                .zipWith(suggestRequest.getResourceFilters()::getOrThrow);

        ListF<ResourceInfo> found = resourceRoutines.getDomainResourcesCanBookWithLayersAndOffices(uid, officeFilters);

        ListF<Email> selected = suggestRequest.getOffices().flatMap(SuggestData.Office.getSelectedResourceEmailsF());
        ListF<ResourceInfo> selectedRes = resourceRoutines.getDomainResourcesCanViewWithLayersAndOffices(
                uid, OfficeFilter.any(), ResourceFilter.any().withEmailFilter(selected));

        ListF<ResourceInaccessibility> inaccessible = eventRoutines.findInaccessibleResources(
                Option.of(userManager.getUserInfo(uid)),
                InaccessibleResourcesRequest.noEventsLookupAndDistance(found, repetitionInfo)
        );

        Function1B<ResourceInfo> isInaccessibleF =
                Function1B.wrap(inaccessible.map(ResourceInaccessibility::getResourceId)::containsTs)
                        .compose(ResourceInfo::getResourceId);

        SetF<String> noSuggestInRooms = SpecialResources.noSuggestInRooms.get().unique();

        if (!userManager.isYamoneyUser(uid)) {
            noSuggestInRooms = noSuggestInRooms.plus(found
                    .filter(ResourceInfo.isTypeF(ResourceType.YAMONEY_ROOM))
                    .map(ResourceInfo.exchangeNameF().andThen(Cf2.f(us -> us.getOrElse("")))));
        }
        noSuggestInRooms = noSuggestInRooms.minus(selected.map(e -> e.getLocalPart().toLowerCase()));

        ListF<ResourceInfo> resources = selectedRes.plus(found)
                .stableUniqueBy(ResourceInfo.resourceIdF())
                .filterNot(ResourceInfo.exchangeNameF().andThen(Cf2.isSomeOfF(noSuggestInRooms)))
                .filterNot(isInaccessibleF);

        MapF<Long, ListF<ResourceInfo>> foundByOfficeId = resources.groupBy(ResourceInfo.officeIdF());
        MapF<Long, ListF<ResourceInfo>> selectedByOfficeId = selectedRes.groupBy(ResourceInfo.officeIdF());

        Function<Long, Integer> foundByOfficeIdF = id -> foundByOfficeId.getOrElse(id, Cf.list()).size();
        Function<Long, Integer> selectedByOfficeIdF = id -> selectedByOfficeId.getOrElse(id, Cf.list()).size();

        ListF<Long> unfilledOfficeIds = Cf.arrayList();

        for (Tuple2<Long, Integer> t : suggestRequest.getNumbersOfResources().entries()) {
            if (foundByOfficeIdF.apply(t._1) < t._2) return Resources.empty();
            if (selectedByOfficeIdF.apply(t._1) < t._2) unfilledOfficeIds.add(t._1);
        }
        ListF<ResourceInfo> lookupResources;

        if (suggestRequest.isSameRoomMode() && selectedRes.exists(isInaccessibleF)) {
            lookupResources = Cf.list();

        } else if (suggestRequest.isSameRoomMode()) {
            lookupResources = selectedRes.filter(ResourceInfo.officeIdF().andThen(unfilledOfficeIds.containsF().notF()))
                    .plus(resources.filter(ResourceInfo.officeIdF().andThen(unfilledOfficeIds.containsF())));
        } else {
            lookupResources = resources;
        }
        return new Resources(lookupResources, selectedRes);
    }

    private UsersFreeIntervalSet findAllUsersFreeIntervals(
            PassportUid uid, ListF<PassportUid> users, InstantInterval interval,
            InstantInterval eventInterval, Option<Long> exceptEventId,
            boolean ignoreUsersEvents, Option<DateTimeZone> tzForPreferredTime, ActionInfo actionInfo)
    {
        ListF<InstantInterval> nightsAndHolidays = getAllUsersNightsAndHolidaysIntervals(users, interval, tzForPreferredTime);

        ListF<InstantInterval> is = Option.when(!eventInterval.isEmpty(),
                () -> eventInterval.withStart(eventInterval.getStart().plus(1)));
        ListF<InstantInterval> muteIntervals = nightsAndHolidays.flatMap(i -> SuggestUtils.crop(is, i)).stableUnique();

        ListF<InstantInterval> busyIntervals = SuggestUtils.cut(nightsAndHolidays, eventInterval);

        if (!ignoreUsersEvents) {
            busyIntervals = busyIntervals.plus(
                    getAllUsersEventsBusyIntervals(uid, users, interval, exceptEventId, false, actionInfo));
        }
        busyIntervals = SuggestUtils.crop(busyIntervals, interval.getStart(), interval.getEnd());

        IntervalSet freeIntervalsSet = IntervalSet.cons(busyIntervals, eventInterval.getDuration())
                .invert(interval.getStart(), interval.getEnd());
        IntervalSet muteIntervalSet = IntervalSet.cons(muteIntervals);

        return new UsersFreeIntervalSet(freeIntervalsSet, muteIntervalSet);
    }

    private ListF<InstantInterval> getAllUsersEventsBusyIntervals(
            PassportUid uid, ListF<PassportUid> users,
            InstantInterval suggestInterval, Option<Long> exceptEventId, boolean ignoreAbsences, ActionInfo actionInfo)
    {
        Validate.notEmpty(users);

        AvailabilityRequest request = AvailabilityRequest.interval(suggestInterval).excludeEventId(exceptEventId);

        if (ignoreAbsences) {
            request = request.excludeAbsencesEvents();
        }

        ListF<UserOrResourceAvailabilityIntervals> intervalss = availRoutines.getAvailabilityIntervalss(
                uid, users.map(UidOrResourceId.userF()), request, actionInfo);

        return intervalss.flatMap(
                is -> is.getIntervalsO().isPresent() ? is.getIntervalsO().get().getIntervals() : Cf.list());
    }

    private ListF<InstantInterval> getAllUsersNightsAndHolidaysIntervals(
            ListF<PassportUid> uids, InstantInterval suggestInterval, Option<DateTimeZone> tzForPreferredTime)
    {
        ListF<InstantInterval> nights = Cf.arrayList();
        ListF<InstantInterval> holidays = Cf.arrayList();

        MapF<PassportUid, SettingsInfo> settingsByUid = settingsRoutines.getSettingsByUidIfExistsBatch(uids);

        for (PassportUid user : uids) {
            Option<SettingsInfo> settings = settingsByUid.getO(user);
            DateTimeZone tz = settings.map(SettingsInfo.getTzF()).getOrElse(dateTimeManager.getDefaultTimezone());

            Option<Long> officeId = settings.filterMap(SettingsInfo.getYtF()).filterMap(SettingsYt.getActiveOfficeIdF());
            Option<Integer> officeCountryGeoId = officeId.map(officeManager.getCountryIdByOfficeIdF());

            LocalDate startDate = suggestInterval.getStart().toDateTime(tz).toLocalDate();
            LocalDate endDate = suggestInterval.getEnd().toDateTime(tz).toLocalDate();

            for (LocalDate date = startDate; !date.isAfter(endDate); date = date.plusDays(1)) {
                Instant dateStart = date.toDateTimeAtStartOfDay(tz).toInstant();
                Instant dateEnd = date.plusDays(1).toDateTimeAtStartOfDay(tz).toInstant();

                if (HolidayRoutines.isDayOff(date, officeCountryGeoId.getOrElse(GeobaseIds.RUSSIA))) {
                    holidays.add(new InstantInterval(dateStart, dateEnd));
                }
                nights.add(new InstantInterval(dateStart, AuxDateTime.toInstantIgnoreGap(date.toLocalDateTime(MORNING), tz)));
                nights.add(new InstantInterval(AuxDateTime.toInstantIgnoreGap(date.toLocalDateTime(EVENING), tz), dateEnd));

                if (tzForPreferredTime.isPresent()) {
                    nights.add(new InstantInterval(dateStart, AuxDateTime.toInstantIgnoreGap(
                            date.toLocalDateTime(PREFERRED_TIME_START), tzForPreferredTime.get())));
                    nights.add(new InstantInterval(AuxDateTime.toInstantIgnoreGap(
                            date.toLocalDateTime(PREFERRED_TIME_END), tzForPreferredTime.get()), dateEnd));
                }
            }
        }
        return nights.plus(holidays);
    }

    private Option<InstantInterval> findCommonDayIntervalForUsers(ListF<PassportUid> uids, final DateTime eventStart) {
        Validate.notEmpty(uids);

        ListF<DateTimeZone> tzs = dateTimeManager.getTimeZonesForUids(uids).get2().plus1(eventStart.getZone());

        ListF<InstantInterval> dayIntervals = tzs.unique().map(new Function<DateTimeZone, InstantInterval>() {
            public InstantInterval apply(DateTimeZone tz) {
                return new InstantInterval(
                        AuxDateTime.toInstantIgnoreGap(eventStart.toLocalDate().toLocalDateTime(MORNING), tz),
                        AuxDateTime.toInstantIgnoreGap(eventStart.toLocalDate().toLocalDateTime(EVENING), tz));
            }
        });
        Instant start = dayIntervals.map(InstantInterval::getStart).max();
        Instant end = dayIntervals.map(InstantInterval::getEnd).min();

        return start.isBefore(end) ? Option.of(new InstantInterval(start, end)) : Option.<InstantInterval>empty();
    }

    private Tuple2List<InstantInterval, ListF<ResourceAvailability>> findResourcesAvailabilityForSomeDays(
            PassportUid uid, ListF<ResourceInfo> resources, RepetitionDays daysRepetition,
            Option<Long> exceptEventId, ActionInfo actionInfo)
    {
        Tuple2List<LocalDate, LocalDate> daysAndMinDues = getLookupDaysAndMinRequiredDueDates(daysRepetition, actionInfo);

        if (daysAndMinDues.isEmpty()) return Tuple2List.tuple2List();

        Tuple2List<InstantInterval, ResourceAvailability> availabilities = Tuple2List.arrayList();
        for (Tuple2<ResourceInfo, ListF<RepetitionDaySchedule>> resourceSchedule
                : findResourceScheduleForRepetitionDaysInFuture(uid, resources, daysRepetition, exceptEventId, actionInfo))
        {
            Tuple2List<LocalDate, LocalTimeOverlaps> daysInFutureOverlaps =
                    findLocalTimeOverlapsForRepetitionDaySchedules(daysRepetition, resourceSchedule._2);

            for (Tuple2<LocalDate, LocalDate> dayAndMinDue : daysAndMinDues) {
                LocalDate day = dayAndMinDue._1;
                LocalDate minRequiredDueDate = dayAndMinDue._2;

                InstantInterval dayInterval = daysRepetition.getInterval(day);

                LocalTimeOverlaps timeOverlaps = daysInFutureOverlaps
                        .findBy1(AuxDateTime.localDateIsBeforeF(day).notF())
                        .map(Tuple2.<LocalDate, LocalTimeOverlaps>get2F()).getOrElse(LocalTimeOverlaps.empty());

                FreeIntervalSet overlapsFreeIntervals = FreeIntervalSet.fromLocalTimeOverlapsForDate(
                        timeOverlaps, day, daysRepetition.getTz());

                FreeIntervalSet freeIntervals = FreeIntervalSet.fromSuccessiveIntervals(
                        daysRepetition.getCheckingIntervals(day).flatMap(overlapsFreeIntervals::cutAndFill));

                freeIntervals = freeIntervals.cutDuesBefore(minRequiredDueDate);

                availabilities.add(dayInterval, new ResourceAvailability(resourceSchedule._1, freeIntervals));
            }
        }
        return availabilities.groupBy1().entries().sortedBy1(Comparator.comparing(InstantInterval::getStart));
    }

    private Tuple2List<ResourceInfo, ListF<RepetitionDaySchedule>> findResourceScheduleForRepetitionDaysInFuture(
            PassportUid uid, ListF<ResourceInfo> resources,
            RepetitionDays repetition, Option<Long> exceptEventId, ActionInfo actionInfo)
    {
        Instant searchStart = ObjectUtils.max(actionInfo.getNow(), repetition.getStart());
        Instant searchEnd = RepetitionInstanceSet.boundedByMaxCheckPeriod(repetition.getDue());

        ListF<LocalDate> baseDates = repetition.getDaysOverlappingInterval(searchStart, searchEnd);

        if (baseDates.isEmpty()) return resources.zipWith(Function.constF(Cf.list()));

        ListF<LocalDate> days = repetition.isDoubleDay()
                ? baseDates.plus(baseDates.map(d -> d.plusDays(1))).stableUnique()
                : baseDates;

        MapF<Long, ResourceInfo> resourceById = resources.toMapMappingToKey(ResourceInfo::getResourceId);

        ListF<ResourceDaySchedule> schedules = resourceScheduleManager.getResourceScheduleDataForDays(
                Option.of(uid), resources.map(ResourceInfo.resourceIdF()),
                days, repetition.getTz(), exceptEventId, actionInfo);

        MapF<Long, MapF<LocalDate, ListF<InstantInterval>>> scheduleByDateByResourceId = schedules
                .groupBy(ResourceDaySchedule::getResourceId)
                .mapValues(ss -> ss.toMap(ResourceDaySchedule::getDate, s -> {
                    ResourceInfo resource = resourceById.getTs(ss.first().getResourceId());

                    return mergeBookingRestrictionIntervals(resource, s.getSchedule().getInstantIntervals(),
                            AuxDateTime.getDayInterval(s.getDate(), repetition.getTz()), actionInfo.getNow());
                }));

        Function<Long, ListF<RepetitionDaySchedule>> resourceScheduleF = resourceId -> {
            MapF<LocalDate, ListF<InstantInterval>> byDate = scheduleByDateByResourceId.getOrThrow(resourceId);

            return baseDates.map(repetition.isDoubleDay()
                    ? day -> new RepetitionDaySchedule(day, byDate.getOrThrow(day).plus(byDate.getOrThrow(day.plusDays(1))))
                    : day -> new RepetitionDaySchedule(day, byDate.getOrThrow(day)));
        };
        return resources.zipWith(ResourceInfo.resourceIdF().andThen(resourceScheduleF));
    }

    private IntervalSet findAllUsersFreeIntervalsForSomeDays(
            PassportUid uid, ListF<PassportUid> users, RepetitionDays daysRepetition,
            InstantInterval eventInterval, Option<Long> exceptEventId, ActionInfo actionInfo)
    {
        ListF<RepetitionDaySchedule> schedules = findAllUsersScheduleForRepetitionDaysInFuture(
                uid, users, daysRepetition, eventInterval, exceptEventId, actionInfo);

        Tuple2List<LocalDate, LocalTimeOverlaps> daysInFutureOverlaps =
                findLocalTimeOverlapsForRepetitionDaySchedules(daysRepetition, schedules);

        ListF<InstantInterval> freeIntervals = Cf.arrayList();

        for (LocalDate day : getLookupDaysAndMinRequiredDueDates(daysRepetition, actionInfo).get1()) {
            LocalTimeOverlaps timeOverlaps = daysInFutureOverlaps
                    .findBy1(AuxDateTime.localDateIsBeforeF(day).notF())
                    .map(Tuple2::get2).getOrElse(LocalTimeOverlaps.empty());

            FreeIntervalSet overlapsFreeIntervals = FreeIntervalSet.fromLocalTimeOverlapsForDate(
                    timeOverlaps, day, daysRepetition.getTz());

            freeIntervals.addAll(daysRepetition.getCheckingIntervals(day)
                    .flatMap(overlapsFreeIntervals::cutAndFill)
                    .filterMap(i -> Option.when(!i.getDueDate().isPresent(), i.getInterval())));
        }
        return IntervalSet.fromSuccessiveIntervals(freeIntervals);
    }

    private ListF<RepetitionDaySchedule> findAllUsersScheduleForRepetitionDaysInFuture(
            PassportUid uid, ListF<PassportUid> users, RepetitionDays daysRepetition,
            InstantInterval eventInterval, Option<Long> exceptEventId, ActionInfo actionInfo)
    {
        Instant searchStart = ObjectUtils.max(actionInfo.getNow(), daysRepetition.getStart());
        Instant searchEnd = RepetitionInstanceSet.boundedForUsersAvailabilities(daysRepetition.getDue());

        ListF<LocalDate> baseDates = daysRepetition.getDaysOverlappingInterval(searchStart, searchEnd);

        if (baseDates.isEmpty()) return Cf.list();

        InstantInterval interval = new InstantInterval(searchStart, searchEnd);

        IntervalSet busyIntervals = IntervalSet.cons(
                getAllUsersEventsBusyIntervals(uid, users, interval, exceptEventId, true, actionInfo),
                eventInterval.getDuration());

        return baseDates.map(day -> new RepetitionDaySchedule(
                day, busyIntervals.getOverlappingIntervals(daysRepetition.getInterval(day))));
    }

    private Tuple2List<LocalDate, LocalTimeOverlaps> findLocalTimeOverlapsForRepetitionDaySchedules(
            RepetitionDays repetition, ListF<RepetitionDaySchedule> schedules)
    {
        Tuple2List<LocalDate, LocalTimeOverlaps> result = Tuple2List.arrayList();
        LocalTimeOverlaps currentOverlaps = LocalTimeOverlaps.empty();

        for (RepetitionDaySchedule schedule : schedules.sortedByDesc(RepetitionDaySchedule::getDate)) {
            LocalDate baseDate = schedule.getDate();
            InstantInterval dayInterval = repetition.getInterval(baseDate);

            ListF<LocalTimeOverlap> overlaps = Cf.arrayList();
            for (InstantInterval interval : schedule.getEventsIntervals()) {
                if (!interval.overlaps(dayInterval)) continue;

                InstantInterval overlap = interval.overlap(dayInterval);

                overlaps.add(new LocalTimeOverlap(
                        new DayTime(baseDate, overlap.getStart(), repetition.getTz()),
                        new DayTime(baseDate, overlap.getEnd(), repetition.getTz()),
                        new LocalDate(baseDate)));
            }
            currentOverlaps = currentOverlaps.union(overlaps);
            result.add(schedule.getDate(), currentOverlaps);
        }
        return result.reverse();
    }

    private Tuple2List<LocalDate, LocalDate> getLookupDaysAndMinRequiredDueDates(
            RepetitionDays repetition, ActionInfo actionInfo)
    {
        ListF<LocalDate> lookupDays = repetition.getDays(REPEATING_LOOKUP_INSTANCES);

        ListF<LocalDate> futureRepetitionDays = repetition.getDaysFromDate(
                new LocalDate(actionInfo.getNow(), repetition.getTz()),
                REPEATING_LOOKUP_INSTANCES + REPEATING_REQUIRED_INSTANCES);

        Tuple2List<LocalDate, LocalDate> result = Tuple2List.arrayList();
        for (LocalDate day: lookupDays) {

            Option<LocalDate> requiredDueDate = futureRepetitionDays
                    .dropWhile(AuxDateTime.localDateIsBeforeF(day)).take(REPEATING_REQUIRED_INSTANCES).lastO();

            result.add(day, requiredDueDate.getOrElse(lookupDays.last()));
        }
        return result;

    }

    private static ListF<InstantInterval> mergeBookingRestrictionIntervals(
            ResourceInfo resource, ListF<InstantInterval> busyIntervals, InstantInterval scheduleInterval, Instant now)
    {
        Option<DateTime> maxStart = SpecialResources.getEventMaxStart(resource, now);

        Option<InstantInterval> distance = maxStart.filterMap(dt -> Option.when(
                dt.isBefore(scheduleInterval.getEnd()),
                () -> new InstantInterval(ObjectUtils.max(dt, scheduleInterval.getStart()), scheduleInterval.getEnd())));

        ListF<InstantInterval> dates = SuggestUtils.crop(
                SpecialResources.getRestrictionIntervals(resource), scheduleInterval);

        return distance.isPresent() || dates.isNotEmpty()
                ? IntervalSet.cons(busyIntervals.plus(distance.plus(dates))).getIntervals()
                : busyIntervals;
    }

    private static InstantInterval consDayInterval(LocalDate date, DateTimeZone tz) {
        return new InstantInterval(date.toDateTime(MORNING, tz), date.toDateTime(EVENING, tz));
    }

    private static InstantInterval consPreferredTimeInterval(LocalDate date, DateTimeZone tz) {
        return new InstantInterval(date.toDateTime(PREFERRED_TIME_START, tz), date.toDateTime(PREFERRED_TIME_END, tz));
    }

    private static class Resources {
        private final ListF<ResourceInfo> found;
        private final ListF<ResourceInfo> selected;

        private Resources(ListF<ResourceInfo> found, ListF<ResourceInfo> selected) {
            this.found = found;
            this.selected = selected;
        }

        public boolean isEmpty() {
            return found.isEmpty();
        }

        public ListF<ResourceInfo> getFound() {
            return found;
        }

        public ListF<ResourceInfo> getSelected() {
            return selected;
        }

        public static Resources empty() {
            return new Resources(Cf.list(), Cf.list());
        }
    }

    private static class ResourcesTimeline {
        private final ListF<ResourceTimeline> resources;
        private final ListF<ResourceTimeline> selected;
        private final Tuple2List<Long, ListF<ResourceTimeline>> offices;

        public ResourcesTimeline(
                ListF<ResourceTimeline> resources,
                ListF<ResourceTimeline> selected,
                Tuple2List<Long, ListF<ResourceTimeline>> offices)
        {
            this.resources = resources;
            this.selected = selected;
            this.offices = offices;
        }

        public ListF<ResourceTimeline> getResources() {
            return resources;
        }

        public ListF<ResourceTimeline> getSelected() {
            return selected;
        }

        public Tuple2List<Long, ListF<ResourceTimeline>> getOffices() {
            return offices;
        }
    }
}
