package ru.yandex.calendar.logic.suggest;

import java.util.Collections;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author gutman
 */
public class SuggestUtils {

    static boolean intervalsContinuous(ListF<InstantInterval> is) {
        return is.zip(is.drop(1)).forAll(endStart -> endStart._1.getEnd().equals(endStart._2.getStart()));
    }

    static boolean intervalsSuccessive(ListF<InstantInterval> is) {
        return is.zip(is.drop(1)).forAll(endStart -> endStart._1.getEndMillis() <= endStart._2.getStartMillis());
    }

    public static ListF<InstantInterval> crop(ListF<InstantInterval> intervals, final Instant start, final Instant end) {
        return intervals.filterMap(instantInterval -> {
            {
                if (instantInterval.getEnd().getMillis() <= start.getMillis()) return Option.empty();
                if (instantInterval.getStart().getMillis() >= end.getMillis()) return Option.empty();

                if (instantInterval.getStart().isBefore(start)) instantInterval = instantInterval.withStart(start);
                if (instantInterval.getEnd().isAfter(end)) instantInterval = instantInterval.withEnd(end);

                return Option.of(instantInterval);
            }
        });
    }

    public static ListF<InstantInterval> crop(ListF<InstantInterval> intervals, InstantInterval crop) {
        return crop(intervals, crop.getStart(), crop.getEnd());
    }

    public static ListF<InstantInterval> cut(ListF<InstantInterval> intervals, InstantInterval cut) {
        return intervals.flatMap(i -> {
            if (i.overlaps(cut)) {
                ListF<InstantInterval> result = Cf.arrayListWithCapacity(2);

                if (i.getEnd().isAfter(cut.getStart()) && i.getStart().isBefore(cut.getStart())) {
                    result.add(i.withEnd(cut.getStart()));
                }
                if (i.getEnd().isAfter(cut.getEnd()) && i.getStart().isBefore(cut.getEnd())) {
                    result.add(i.withStart(cut.getEnd()));
                }
                return result;
            } else {
                return Cf.list(i);
            }
        });
    }

    public static ListF<InstantInterval> invert(ListF<InstantInterval> intervals, Instant start, Instant end) {
        Validate.isTrue(start.getMillis() <= end.getMillis());

        if (intervals.isEmpty()) {
            return Cf.list(new InstantInterval(start, end));
        }

        validateSuccessive(intervals);

        ListF<InstantInterval> inverted = intervals.zip(intervals.drop(1)).map(
                (x, y) -> new InstantInterval(x.getEnd(), y.getStart()));

        if (start.isBefore(intervals.first().getStart())) {
            inverted = Cf.list(new InstantInterval(start, intervals.first().getStart())).plus(inverted);
        }
        if (end.isAfter(intervals.last().getEnd())) {
            inverted = inverted.plus(new InstantInterval(intervals.last().getEnd(), end));
        }
        return inverted;
    }

    public static <T> ListF<T> findOverlappingForSuccessiveIntervals(
            ListF<T> successiveIntervals, T overlappingInterval, Function<T, InstantInterval> intervalF)
    {
        ListF<T> is = successiveIntervals;
        InstantInterval interval = intervalF.apply(overlappingInterval);

        Instant since = interval.getStart();
        Instant till = interval.getEnd();

        int start = Collections.binarySearch(is, overlappingInterval,
                intervalF.andThen(InstantInterval::getStart).andThenNaturalComparator());
        if (start < 0) {
            start = -start - 2;
            start = start >= 0 && since.isBefore(intervalF.apply(is.get(start)).getEnd()) ? start : start + 1;
        }

        int end = Collections.binarySearch(is, overlappingInterval,
                intervalF.andThen(InstantInterval::getEnd).andThenNaturalComparator());
        if (end < 0) {
            end = -end - 1;
            end = end < is.size() && intervalF.apply(is.get(end)).getStart().isBefore(till) ? end + 1 : end;
        } else {
            end = end + 1;
        }
        return successiveIntervals.subList(start, end);
    }

    static void validateSuccessive(ListF<InstantInterval> intervals) {
        Validate.isTrue(intervalsSuccessive(intervals));
    }

    static ListF<Instant> getContainedHalfHoursStarts(InstantInterval interval) {
        DateTime start = interval.getStart().toDateTime(DateTimeZone.UTC);
        start = start.withMinuteOfHour(start.getMinuteOfHour() / 30 * 30).withSecondOfMinute(0).withMillisOfSecond(0);

        if (start.isBefore(interval.getStart())) start = start.plusMinutes(30);

        ListF<Instant> result = Cf.arrayList();
        for (; start.isBefore(interval.getEnd()); start = start.plusMinutes(30)) {
            result.add(start.toInstant());
        }
        return result;
    }

    static ListF<InstantInterval> getIntervalsToLookup(
            InstantInterval interval, Instant eventStart, Duration eventDuration)
    {
        ListF<Instant> halfHourStarts = SuggestUtils.getContainedHalfHoursStarts(interval);

        if (interval.contains(eventStart) && !halfHourStarts.containsTs(eventStart)) {
            halfHourStarts = halfHourStarts.plus1(eventStart).sorted();
        }
        return halfHourStarts.map(start -> new InstantInterval(start.toInstant(), eventDuration));
    }

}
