package ru.yandex.calendar.logic.suggest;

import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.calendar.logic.beans.generated.Office;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author gutman
 */
public class UsersAndResourcesAvailability {

    private final Option<IntervalSet> allUsersFreeTimes;
    private final ListF<OfficeAvailability> officesAvailability;
    private final MapF<Long, Integer> requiredNumberOfResourcesByOfficeId;
    private final MapF<Long, ListF<Long>> selectedResourceIdsByOfficeId;

    private UsersAndResourcesAvailability(
            Option<IntervalSet> allUsersFreeTimes,
            ListF<OfficeAvailability> officesAvailability,
            MapF<Long, Integer> requiredNumberOfResourcesByOfficeId,
            MapF<Long, ListF<Long>> selectedResourceIdsByOfficeId)
    {
        Validate.forAll(officesAvailability,
                OfficeAvailability.getOfficeIdF().andThen(requiredNumberOfResourcesByOfficeId::containsKeyTs));

        this.allUsersFreeTimes = allUsersFreeTimes;
        this.officesAvailability = officesAvailability;
        this.requiredNumberOfResourcesByOfficeId = requiredNumberOfResourcesByOfficeId;
        this.selectedResourceIdsByOfficeId = selectedResourceIdsByOfficeId;
    }

    public UsersAndResourcesAvailability plusOffice(Office office,
            Tuple2List<ResourceInfo, ListF<InstantInterval>> freeIntervals, int requiredNumberOfResources)
    {
        Validate.forAll(officesAvailability, OfficeAvailability.getOfficeIdF().andThenEquals(office.getId()).notF());
        Validate.isTrue(requiredNumberOfResources > 0);

        ListF<ResourceAvailability> resourceAvailabilities = freeIntervals
                .map2(FreeIntervalSet.fromIntervalsWithNoDueDateF())
                .map(ResourceAvailability.consF());

        OfficeAvailability officeAvailability = new OfficeAvailability(office, resourceAvailabilities);

        return new UsersAndResourcesAvailability(allUsersFreeTimes,
                officesAvailability.plus(officeAvailability),
                requiredNumberOfResourcesByOfficeId.plus1(office.getId(), requiredNumberOfResources),
                selectedResourceIdsByOfficeId);
    }

    public UsersAndResourcesAvailability plusOffices(
            ListF<OfficeAvailability> officesAvailability, MapF<Long, Integer> requiredNumberOfResourcesByOfficeId)
    {
        return new UsersAndResourcesAvailability(
                this.allUsersFreeTimes,
                this.officesAvailability.plus(officesAvailability),
                this.requiredNumberOfResourcesByOfficeId.plus(requiredNumberOfResourcesByOfficeId),
                selectedResourceIdsByOfficeId);
    }

    public UsersAndResourcesAvailability withSelected(ListF<ResourceInfo> resources) {
        MapF<Long, ListF<Long>> selectedResourceIdsByOfficeId = resources
                .toTuple2List(ResourceInfo.officeIdF(), ResourceInfo.resourceIdF()).groupBy1();

        return new UsersAndResourcesAvailability(
                allUsersFreeTimes, officesAvailability,
                requiredNumberOfResourcesByOfficeId, selectedResourceIdsByOfficeId);
    }

    Option<AvailableResourcesInOffices> getAvailableResourcesIfAllOfficesAreAvailable(InstantInterval interval) {
        if (allUsersFreeTimes.isPresent() && !allUsersFreeTimes.get().contains(interval)) {
            return Option.empty();
        }

        if (officesAvailability.isEmpty()) {
            return Option.empty();
        }

        AvailableResourcesInOffices availableResourcesInOffices = AvailableResourcesInOffices.empty(interval);
        for (OfficeAvailability officeAvailability : officesAvailability) {
            Tuple2List<ResourceInfo, Option<LocalDate>> availableInOffice = officeAvailability.getFreeResources(interval);
            int requiredNumberOfResources = requiredNumberOfResourcesByOfficeId.getOrThrow(officeAvailability.getOfficeId());

            ListF<Long> availableInOfficeResourceIds = availableInOffice.get1().map(ResourceInfo.resourceIdF());
            ListF<Long> selectedResourceIds = selectedResourceIdsByOfficeId.getOrElse(
                    officeAvailability.getOfficeId(), Cf.list());

            if (requiredNumberOfResources > availableInOffice.size()) {
                return Option.empty();
            }
            if (selectedResourceIds.exists(availableInOfficeResourceIds.containsF().notF())) {
                return Option.empty();
            }
            availableResourcesInOffices = availableResourcesInOffices
                    .plusOne(officeAvailability.getOffice(), availableInOffice);
        }

        return Option.of(availableResourcesInOffices);
    }

    public static UsersAndResourcesAvailability onlyUsers(IntervalSet allUsersFreeTimes) {
        return new UsersAndResourcesAvailability(Option.of(allUsersFreeTimes), Cf.list(), Cf.map(), Cf.map());
    }

    public static UsersAndResourcesAvailability ignoreUsers() {
        return new UsersAndResourcesAvailability(Option.empty(), Cf.list(), Cf.map(), Cf.map());
    }

}
