package ru.yandex.calendar.logic.svc;

import org.jdom.Element;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.calendar.logic.layer.LayerDao;
import ru.yandex.calendar.util.base.AuxBase;
import ru.yandex.calendar.util.data.DataProvider;
import ru.yandex.calendar.util.db.CalendarJdbcDaoSupport;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.inside.passport.PassportSid;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.q.SqlCondition;

/**
 * Routines related to services that are added to calendar database.
 * For these services, it is allowed to add/keep events to calendar.
 * Initialized once, on-demand; initialization check does not block.
 * @author ssytnik
 */
public class DbSvcRoutines extends CalendarJdbcDaoSupport {

    @Autowired
    private LayerDao layerDao;


    public static final Service TV;
    private static final ListF<Service> services; // sorted by service name alphabetically
    private static final MapF<PassportSid, Service> servicesBySid;
    private static final ListF<PassportSid> sids;

    private static Service service(
            PassportSid sid, String domainName, String name, String sName, String enName,
            String layerCssClass, int layerHeadBgColor, int layerHeadFgColor,
            int layerBodyBgColor, boolean showWhenPossible)
    {
        return new Service(
                sid, domainName, name, sName, enName,
                layerCssClass, layerHeadBgColor, layerHeadFgColor,
                layerBodyBgColor, showWhenPossible);
    }

    static {
        services = Cf.list(
            service(PassportSid.AFISHA,   "afisha.yandex.ru",   "Афиша",         "Афиша",       "afisha",   "afisha",   0xE33830, 0xFFFFFF, 0xFCEBEA, false),
            service(PassportSid.BALANCE,  "balance.yandex.ru",  "Баланс",        "Баланс",      "balance",  "balance",  0x615770, 0xFFFFFF, 0xEDECEF, false),
            service(PassportSid.CALENDAR, "calendar.yandex.ru", "Мои события",   "Мои события", "calendar", "calendar", 0x6482A1, 0xFFFFFF, 0xFBFBED, false),
            service(PassportSid.TV,       "tv.yandex.ru",       "Телепрограмма", "ТВ",          "tv",       "tv",       0x58585A, 0xFFFFFF, 0xE8E8E8, false)
        );

        servicesBySid = services.toMapMappingToKey(Service.getSidF());
        sids = services.map(Service.getSidF());

        TV = servicesBySid.getOrThrow(PassportSid.TV);
    }

    public ListF<Service> getServices() {
        return services;
    }

    public static MapF<PassportSid, Service> getServicesBySid() {
        return servicesBySid;
    }

    public ListF<PassportSid> getSids() {
        return sids;
    }

    public static Integer getSid(DataProvider gDp) {
        // NOTE: difference from 'getSidByDomainName' is that there is no
        // 'sid' existence check. However, it can be added since needed.
        return AuxBase.toNullableIntUnsafe(gDp.getText("sid", true));
    }

    public static Service getService(PassportSid sid) {
        return getServicesBySid().getO(sid).getOrThrow("service by sid not found: ", sid);
    }

    public boolean exists(PassportSid sid) {
        return getServicesBySid().containsKeyTs(sid);
    }

    public static void appendSidElm(Element eGeneral, PassportSid sid) {
        CalendarXmlizer.appendElm(eGeneral, "sid", sid.getValue());
    }

    public long getLayerId(PassportSid sid, PassportUid uid) {
        SqlCondition condition = SqlCondition.condition("sid = ? AND creator_uid = ?", sid, uid);
        return layerDao.findLayers(condition).single().getId();
    }


    // This method serves as a replacement for uiGetLayers code, without database access.
    // sids - specify services (for which layers exist) that should be excluded from search.
    public void addPossibleLayers(CollectionF<PassportSid> sids, Element rootElement) {
        Element ePossibleLayers = null; // may become <possible-layers>
        for (Service svc : getServices()) { // svcList is sorted properly (alphabetically)
            PassportSid sid = svc.getSid();
            // NOTE: '!AuxBase.longObjEqual(SvcRoutines.CALENDAR_SID, sid)' is now
            // unnecessary due to calendar db value of 'show_when_possible' == 0.
            if (svc.isShowWhenPossible() && !sids.containsTs(sid)) {
                Element ePossibleLayer = new Element("possible-layer");
                CalendarXmlizer.appendElm(ePossibleLayer, "sid", sid.getValue());
                CalendarXmlizer.appendElm(ePossibleLayer, "domain-name", svc.getDomainName());
                CalendarXmlizer.appendElm(ePossibleLayer, "name", svc.getName());
                if (ePossibleLayers == null) {
                    ePossibleLayers = new Element("possible-layers");
                }
                ePossibleLayers.addContent(ePossibleLayer);
            } // if
        } // for
        if (ePossibleLayers != null) {
            rootElement.addContent(ePossibleLayers);
        }
    }
}
