package ru.yandex.calendar.logic.svc;

import java.net.IDN;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.PropName;
import ru.yandex.calendar.logic.beans.generated.Settings;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.util.Environment;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.PassportUidGroup;
import ru.yandex.inside.passport.YandexPassport;
import ru.yandex.inside.passport.blackbox.PassportAuthDomain;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder.PolicyHandle;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.ip.InternetDomainName;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class SvcRoutines {
    private static final Logger logger = LoggerFactory.getLogger(SvcRoutines.class);

    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private YandexPassport yandexPassport;
    @Value("${calendar.url.prefix}")
    private String calendarUrlPrefix;

    private boolean addSidToPassport() {
        return Environment.isProductionOrPre();
    }

    // Calendar
    // just a literal (service name), used at property names and as a service type at db (contacts)
    public static final String CALENDAR_NAME = "calendar"; // just a literal (service name)
    // Email/narod/domains
    public static final String AT_YANDEX_RU = "@yandex.ru";
    public static final String AT_YANDEX_TEAM_RU = "@yandex-team.ru";

    public String getSelfUrlHost() {
        return getSelfUrlHost(PassportAuthDomainsHolder.H.getDefaultDomain().getOrElse(PassportAuthDomain.PUBLIC));
    }

    public String getSelfUrlHost(PassportAuthDomain domain, Language language) {
        if (language.getForSettings() == Language.ENGLISH) {
            if (domain == PassportAuthDomain.PUBLIC && Environment.isProductionOrPre()) {
                return "calendar.yandex.com";
            }
        }
        return getSelfUrlHost(domain);
    }

    public String getSelfUrlHost(PassportAuthDomain domain) {
        return String.format("%s.%s", calendarUrlPrefix, domain.getDomain());
    }

    public Email getCalendarInfoEmail(PassportAuthDomain domain) {
        return new Email("info@" + getSelfUrlHost(domain));
    }

    public Email getCalendarInfoEmail(Email email) {
        return new Email("info@" + getSelfUrlHost(PassportAuthDomain.byEmail(email)));
    }

    public Email getCalendarInfoEmail(PassportUid uid) {
        return getCalendarInfoEmail(PassportAuthDomain.byUid(uid));
    }

    public String getCalendarUrlForDomain(PassportAuthDomain domain, Language language) {
        return (Environment.isProductionOrPre() ? "https" : "http") + "://" + getSelfUrlHost(domain, language);
    }

    public String getCalendarUrlForDomain(PassportAuthDomain domain) {
        return getCalendarUrlForDomain(domain, Language.RUSSIAN);
    }

    public String getCalendarUrlForDomain(InternetDomainName domain) {
        return getCalendarUrlForDomain(PassportAuthDomain.byDomain(domain));
    }

    public String getCalendarUrlBySettings(Settings s) {
        PassportUidGroup yandexUidGroup = PassportUidGroup.byUid(s.getUid().getUid());
        Option<String> pddPrefix = Option.empty();
        if (yandexUidGroup == PassportUidGroup.DOMAINS) {
            pddPrefix = Option.of("/for/" + UrlUtils.urlEncode(IDN.toUnicode(s.getDomain().get())));
        }
        return getCalendarUrlForDomain(PassportAuthDomain.byUid(s.getUid()), s.getLanguage()) + pddPrefix.getOrElse("");
    }

    public String getCalendarUrlForUid(PassportUid uid) {
        return getCalendarUrlBySettings(settingsRoutines.getSettingsByUid(uid).getCommon());
    }

    public String getEventPageUrlPrefixForUid(PassportUid uid) {
        return getEventPageUrlPrefixBySettings(settingsRoutines.getSettingsByUid(uid).getCommon());
    }

    public String getEventPageUrlPrefixBySettings(Settings settings) {
        return getCalendarUrlBySettings(settings) + "/event?event_id=";
    }

    public String getCalendarUrlForEmail(Email email) {
        return getCalendarUrlForDomain(email.getDomain());
    }

    // Tries to add calendar sid for user, then tries to update settings
    public void addCalendarSidToPassport(PassportUid uid) {
        if (!addSidToPassport()) {
            logger.debug(PropName.ADD_SID_TO_PASSPORT.get() + " is off, quitting");
            return;
        }
        if (uid.getGroup() == PassportUidGroup.YANDEX_TEAM_RU) {
            return;
        }

        if (!settingsRoutines.isSidAdded(uid)) {
            logger.debug("Attempting to add calendar sid for uid " + uid);
            PolicyHandle h = MasterSlaveContextHolder.push(MasterSlavePolicy.RW_M);
            try {
                yandexPassport.admSubscribe(uid.getUid(), "calendar");
                Settings settingsData = new Settings();
                settingsData.setIsSidAdded(true);
                settingsRoutines.updateOrCreateSettingsByUid(settingsData, uid);
            } catch (Exception e) {
                final String msg = "could not add sid / update settings for uid = " + uid + ": " + e.getMessage();
                logger.error(msg); // keep silence, this is non-critical error
            } finally {
                h.popSafely();
            }
        }
    }
}
