package ru.yandex.calendar.logic.telemost;

import java.util.regex.Pattern;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import io.micrometer.core.instrument.MeterRegistry;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.apache.http.client.HttpClient;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpRequestBase;

import ru.yandex.calendar.tvm.TvmClient;
import ru.yandex.calendar.tvm.TvmHeaders;
import ru.yandex.calendar.util.HttpClientConfiguration;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.utils.JacksonResponseHandlerFactory;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;

/**
 * @link https://wiki.yandex-team.ru/rtc/group-calls/telemost-api/mvp1/
 * @link https://wiki.yandex-team.ru/disk/mpfs/PlatformAPI/Auth/
 * @link https://a.yandex-team.ru/arc_vcs/disk/telemost/src/main/java/ru/yandex/chemodan/app/telemost/web/v2/actions/ConferencesActionsV2.java
 */
@Slf4j
public class TelemostClient {
    private static final String HEADER_UID = "X-Uid";
    private static final Pattern METRICS_URI = Pattern.compile("(conferences(?=/$)|link-calendar-event$|broadcast$)");

    private final String conferencesUrl;

    private final HttpClient httpClient;
    private final TvmClient tvmClient;
    private final int telemostTvmId;

    private final ResponseHandler<ObjectNode> jsonResponseHandler;

    public TelemostClient(
            String baseUrl,
            HttpClientConfiguration httpConf,
            MeterRegistry registry,
            TvmClient tvmClient,
            int telemostTvmId
    ) {
        this.conferencesUrl = baseUrl + "/v2/telemost/conferences/";
        this.tvmClient = tvmClient;
        this.telemostTvmId = telemostTvmId;
        this.httpClient = httpConf.consClient(registry, "telemost", METRICS_URI);
        this.jsonResponseHandler = new JacksonResponseHandlerFactory(new ObjectMapper()).cons(ObjectNode.class);
    }

    public String createConference(PassportUid creatorUid) {
        val request = new HttpPost(conferencesUrl);
        addHeaders(request, creatorUid);

        val response = ApacheHttpClientUtils.execute(request, httpClient, jsonResponseHandler);
        return response.get("uri").textValue();
    }

    public String createBroadcast(PassportUid creatorUid, String conferenceUri) {
        String url = conferencesUrl + UrlUtils.urlEncode(conferenceUri) + "/broadcast";

        val request = new HttpPut(url);
        addHeaders(request, creatorUid);

        val response = ApacheHttpClientUtils.execute(request, httpClient, jsonResponseHandler);
        return response.get("broadcast_uri").textValue();
    }

    public void linkCalendarEvent(PassportUid creatorUid, String conferenceUri, long eventId) {
        String url = conferencesUrl + UrlUtils.urlEncode(conferenceUri)
                + "/link-calendar-event?calendar_event_id=" + eventId;

        val request = new HttpPost(url);
        addHeaders(request, creatorUid);

        ApacheHttpClientUtils.execute(request, httpClient, jsonResponseHandler);
    }

    private void addHeaders(HttpRequestBase httpRequestBase, PassportUid uid) {
        httpRequestBase.setHeader(TvmHeaders.SERVICE_TICKET, tvmClient.getServiceTicketFor(telemostTvmId));
        httpRequestBase.setHeader(HEADER_UID, uid.toString());
    }
}
