package ru.yandex.calendar.logic.telemost;

import java.util.function.Supplier;

import lombok.AllArgsConstructor;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Lazy;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Unit;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.frontend.worker.task.TelemostGenerateTask;
import ru.yandex.calendar.logic.XivaNotificationManager;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActorId;
import ru.yandex.calendar.logic.event.EventChangesInfo;
import ru.yandex.calendar.logic.event.EventDbManager;
import ru.yandex.calendar.logic.event.EventInstanceForUpdate;
import ru.yandex.calendar.logic.event.EventInvitationManager;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.SequenceAndDtStamp;
import ru.yandex.calendar.logic.event.UpdateInfo;
import ru.yandex.calendar.logic.event.dao.EventDao;
import ru.yandex.calendar.logic.event.meeting.UpdateMode;
import ru.yandex.calendar.logic.event.model.EventData;
import ru.yandex.calendar.logic.event.repetition.EventAndRepetition;
import ru.yandex.calendar.logic.event.repetition.RepetitionRoutines;
import ru.yandex.calendar.logic.ics.EventInstanceStatusInfo;
import ru.yandex.calendar.logic.sharing.EventParticipantsChangesInfo;
import ru.yandex.calendar.logic.sharing.participant.ParticipantId;
import ru.yandex.calendar.logic.telemost.TelemostManager.PatchResult.ConferenceCreated;
import ru.yandex.calendar.logic.update.LockTransactionManager;
import ru.yandex.calendar.logic.update.UpdateLock2;
import ru.yandex.calendar.util.PgOnetimeUtils;
import ru.yandex.commune.bazinga.impl.OnetimeJob;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.ThreadLocalX;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.time.InstantInterval;

import static ru.yandex.calendar.logic.telemost.TelemostEventDataUtils.containsAnyLink;
import static ru.yandex.calendar.logic.telemost.TelemostEventDataUtils.containsConferenceLink;
import static ru.yandex.calendar.logic.telemost.TelemostEventDataUtils.needGenerate;
import static ru.yandex.calendar.logic.telemost.TelemostEventDataUtils.needGenerateAndNotContainsLink;
import static ru.yandex.calendar.logic.telemost.TelemostEventDataUtils.needGenerateBroadcast;
import static ru.yandex.calendar.logic.telemost.TelemostEventDataUtils.needGenerateConference;

@Slf4j
public class TelemostManager implements TelemostConstants {

    @Autowired
    private TelemostClient telemostClient;
    @Autowired
    private TelemostPatcher patcher;
    @Autowired
    private TelemostJobStorage jobStorage;
    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private EventDao eventDao;
    @Autowired
    private EventInvitationManager eventInvitationManager;
    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private UpdateLock2 updateLock2;
    @Autowired
    private LockTransactionManager lockTransactionManager;
    @Autowired
    private XivaNotificationManager xivaNotificationManager;

    private final ThreadLocalX<Unit> selfUpdate = new ThreadLocalX<>();

    public PatchResult patchCreateData(EventData data, ParticipantId organizerId) {
        Check.isFalse(selfUpdate.isSet());

        Event event = data.getEvent();

        if (!needGenerate(event)) {
            return new PatchResult.NothingToDoFurther();
        }
        if (!organizerId.isYandexUser()) {
            log.debug("Ignoring since organizer is not yandex-user {}", organizerId);

            return new PatchResult.NothingToDoFurther();
        }
        if (RepetitionRoutines.isNoneRepetition(data.getRepetition())) {
            if (containsAnyLink(event)) {
                log.debug("Already have link for {}", getDebugInfo(event));

                return new PatchResult.NothingToDoFurther();
            } else {
                return createConferenceAndInsertLink(event, event, organizerId.getUid());
            }
        } else {
            return promiseConferenceCreationAndCleanLink(event, event, organizerId.getUid());
        }
    }

    public PatchResult patchUpdateData(
            Event changes,
            EventInstanceForUpdate eventInstance,
            EventParticipantsChangesInfo participantsChanges,
            UpdateMode updateMode
    ) {
        if (selfUpdate.isSet()) {
            return new PatchResult.NothingToDoFurther();
        }
        Event updatedEvent = eventInstance.getEvent().copy();
        updatedEvent.setFields(changes);

        boolean generatedTelemost = needGenerate(eventInstance.getEvent());
        boolean willGenerateTelemost = needGenerate(updatedEvent);

        boolean wasRepeating = eventInstance.getEvent().getRepetitionId().isPresent()
                && !updatedEvent.getRecurrenceId().isPresent();
        boolean willBeRepeating = updatedEvent.getRepetitionId().isPresent();

        if (generatedTelemost && !willGenerateTelemost) {
            patcher.cleanDescriptionAndLinks(changes, updatedEvent);

            log.debug("Telemost cleared for {}", getDebugInfo(updatedEvent));
            return new PatchResult.ConferenceCleared(updatedEvent.getId());
        }
        val creatorUid = Lazy.withSupplier(() -> participantsChanges.getNewOrganizer()
                .orElse(eventInstance.getEventWithRelations().getOrganizerIdSafe())
                .flatMapO(ParticipantId::getUidIfYandexUser)
                .orElse(eventInstance.getEvent().getCreatorUid()));

        if (willGenerateTelemost && (!generatedTelemost || wasRepeating != willBeRepeating)) {
            if (!willBeRepeating && containsConferenceLink(updatedEvent)) {
                log.debug("Already have link for {}", getDebugInfo(updatedEvent));
                return new PatchResult.NothingToDoFurther();

            } else if (!willBeRepeating && updateMode != UpdateMode.RECURRENCE_FOR_MASTER) {
                return createConferenceAndInsertLink(changes, updatedEvent, creatorUid.get());

            } else {
                return promiseConferenceCreationAndCleanLink(changes, updatedEvent, creatorUid.get());
            }
        }
        return new PatchResult.NothingToDoFurther();
    }

    public void patchJsonDataForRecurrenceUpdateWithMaster(
            Event recurrenceChanges,
            EventInstanceForUpdate recurrenceInstance,
            EventChangesInfo masterChanges,
            EventInstanceForUpdate masterInstance
    ) {
        Check.isFalse(selfUpdate.isSet());

        Event updatedMaster = masterInstance.getEvent().copy();
        updatedMaster.setFields(masterChanges.getEventChanges());

        Event updatedRecurrence = recurrenceInstance.getEvent().copy();
        updatedRecurrence.setFields(recurrenceChanges);

        Option<Boolean> conferenceChange = Option.of(needGenerateConference(updatedMaster))
                .filterNot(Function1B.equalsF(needGenerateConference(masterInstance.getEvent())))
                .filterNot(Function1B.equalsF(needGenerateConference(updatedRecurrence)));

        Option<Boolean> broadcastChange = Option.of(needGenerateBroadcast(updatedMaster))
                .filterNot(Function1B.equalsF(needGenerateBroadcast(masterInstance.getEvent())))
                .filterNot(Function1B.equalsF(needGenerateBroadcast(updatedRecurrence)));

        if (conferenceChange.isPresent() || broadcastChange.isPresent()) {
            log.debug("Switching needs conference to {} and broadcast to {} for {}",
                    conferenceChange, broadcastChange, getDebugInfo(updatedRecurrence));

            patcher.setNeedGenerate(
                    recurrenceChanges, updatedRecurrence, conferenceChange, broadcastChange);
        }
    }

    public void onEventUpdated(PatchResult result, long eventId, ActionInfo actionInfo) {
        onEventUpdated(result, () -> eventId, () -> eventDbManager.getEventAndRepetitionById(eventId), actionInfo);
    }

    public void onEventUpdated(PatchResult result, EventAndRepetition event, ActionInfo actionInfo) {
        onEventUpdated(result, event::getEventId, () -> event, actionInfo);
    }

    public void rescheduleGeneration(EventAndRepetition event, ActionInfo actionInfo) {
        if (selfUpdate.isSet()) return;

        if (needGenerateAndNotContainsLink(event.getEvent())) {
            rescheduleGenerationForNextInstance(event, actionInfo);
        }
    }

    public void cancelScheduledGeneration(Event event) {
        if (selfUpdate.isSet()) return;

        if (needGenerateAndNotContainsLink(event)) {
            jobStorage.deleteJob(event.getId());
        }
    }

    public void generateLinkOrReschedule(long eventId, ActionInfo actionInfo) {
        Check.isFalse(selfUpdate.isSet());

        Option<UpdateInfo> updateInfo = lockTransactionManager.lockAndDoInTransaction(
                () -> updateLock2.lockEvent(eventId),
                () -> doGenerateLinkOrReschedule(eventId, actionInfo.withNow(Instant.now()))
        );
        updateInfo.ifPresent(info ->
                xivaNotificationManager.notifyLayersUsersAboutEventsChange(info.getAffectedLayerIds(), actionInfo));
    }

    private void onEventUpdated(
            PatchResult result,
            Supplier<Long> eventIdSupplier,
            Supplier<EventAndRepetition> eventSupplier,
            ActionInfo actionInfo
    ) {
        if (result instanceof ConferenceCreated) {
            val data = (ConferenceCreated) result;
            long eventId = eventIdSupplier.get();

            telemostClient.linkCalendarEvent(data.creatorUid, data.conferenceUri, eventId);
            log.debug("Conference linked to {}", eventId);

            jobStorage.deleteJob(eventId);

        } else if (result instanceof PatchResult.ConferencePromised) {
            rescheduleGenerationForNextInstance(eventSupplier.get(), actionInfo);

        } else if (result instanceof PatchResult.ConferenceCleared) {
            jobStorage.deleteJob(((PatchResult.ConferenceCleared) result).eventId);
        }
    }

    private ConferenceCreated createConferenceAndInsertLink(
            Event changes, Event source, PassportUid creatorUid
    ) {
        String conferenceUri = telemostClient.createConference(creatorUid);

        Option<String> broadcastUri = needGenerateBroadcast(source)
                ? Option.of(telemostClient.createBroadcast(creatorUid, conferenceUri))
                : Option.empty();

        patcher.patchLinks(changes, source, conferenceUri, broadcastUri, creatorUid);

        log.debug("Conference created for {}", getDebugInfo(source));

        return new ConferenceCreated(conferenceUri, creatorUid);
    }

    private PatchResult.ConferencePromised promiseConferenceCreationAndCleanLink(
            Event changes, Event source, PassportUid creatorUid
    ) {
        patcher.cleanDescriptionAndLinks(changes, source);
        patcher.patchPromise(changes, source, creatorUid);

        log.debug("Conference promised for {}", getDebugInfo(source));

        return new PatchResult.ConferencePromised();
    }

    private void rescheduleGenerationForNextInstance(EventAndRepetition event, ActionInfo actionInfo) {
        findSchedulingInstance(event, actionInfo).ifPresent(this::rescheduleGeneration);
    }

    private void rescheduleGeneration(SchedulingInstance instance) {
        val task = new TelemostGenerateTask(instance.eventId);
        val job = PgOnetimeUtils.makeJob(task, instance.getScheduleTime());

        val activeJob = jobStorage.findActiveJob(task);
        val activeScheduleTime = activeJob.map(OnetimeJob::getScheduleTime);

        if (!activeJob.isPresent() || !activeScheduleTime.isSome(job.getScheduleTime())) {
            jobStorage.saveOrMergeJob(job);
            log.debug("Generating for {} scheduled to {}", instance.eventId, job.getScheduleTime());
        }
    }

    private Option<UpdateInfo> doGenerateLinkOrReschedule(long eventId, ActionInfo actionInfo) {
        val eventO = eventDbManager.getEventByIdSafe(eventId);

        if (!eventO.exists(TelemostEventDataUtils::needGenerateAndNotContainsLink)) {
            log.debug("No suitable event found {}", eventO.map(TelemostManager::getDebugInfo));
            return Option.empty();
        }
        val event = eventDbManager.getEventAndRepetitionByEvent(eventO.get());
        Option<SchedulingInstance> instance = findSchedulingInstance(event, actionInfo);

        if (instance.isEmpty()) {
            log.debug("No suitable instance found for {}", getDebugInfo(eventO.get()));
            return Option.empty();
        }
        val suitable = instance.get().scheduleSuitableInterval;

        if (!suitable.contains(actionInfo.getNow())) {
            log.debug("Rescheduling since {} is not in {}", actionInfo.getNow(), suitable);

            rescheduleGeneration(instance.get());
            return Option.empty();
        }
        UpdateInfo updateInfo;

        selfUpdate.set(Unit.U);
        try {
            updateInfo = createConferenceAndUpdateEvent(eventId, instance.get().getStart(), actionInfo);
        } finally {
            selfUpdate.remove();
        }

        val excluded = new EventAndRepetition(
                event.getEvent(),
                event.getRepetitionInfo().plusExdates(Cf.list(instance.get().getStart())));

        rescheduleGenerationForNextInstance(excluded, actionInfo);
        return Option.of(updateInfo);
    }

    private UpdateInfo createConferenceAndUpdateEvent(long eventId, Instant instanceStart, ActionInfo actionInfo) {
        val instance = eventRoutines.getSingleEventInstanceForModifierOrCreateRecurrence(
                Option.empty(), eventId, instanceStart, Option.empty(), actionInfo
        );
        val creatorUid = instance.getEventWithRelations().getOrganizerUidIfMeeting()
                .orElse(instance.getEvent().getCreatorUid());

        val eventChanges = new Event();
        ConferenceCreated creation = createConferenceAndInsertLink(eventChanges, instance.getEvent(), creatorUid);

        val changesFactory = new EventChangesInfo.EventChangesInfoFactory();
        changesFactory.setEventChanges(eventChanges);

        val updateInfo = eventRoutines.updateEvent(
                ActorId.yaCalendar(), Option.empty(), instance, changesFactory.create(),
                EventInstanceStatusInfo.needToUpdate(instance.getEventId()),
                SequenceAndDtStamp.web(instance.getEvent(), actionInfo), Option.empty(), actionInfo);

        val messages = eventInvitationManager.createEventInvitationOrCancelMails(
                ActorId.yaCalendar(), updateInfo.getSendingInfos(), actionInfo
        );
        eventInvitationManager.sendEventMails(messages.plus(updateInfo.getLayerNotifyMails()), actionInfo);

        // Like EventWebUpdater does
        eventDao.updateEventIncrementSequenceById(eventId);

        telemostClient.linkCalendarEvent(creation.creatorUid, creation.conferenceUri, instance.getEventId());
        log.debug("Conference linked to {}", getDebugInfo(instance.getEvent()));

        return updateInfo;
    }

    private static Option<SchedulingInstance> findSchedulingInstance(EventAndRepetition event, ActionInfo actionInfo) {
        val instance = event.getFirstInstanceIntervalEndingAfter(actionInfo.getNow());

        if (instance.isPresent()) {
            val suitable = instance.get().withStart(
                    instance.get().getStart().minus(Duration.standardHours(GENERATE_HOURS_BEFORE_START)));

            return Option.of(new SchedulingInstance(event.getEventId(), instance.get(), suitable));
        } else {
            log.debug("Event {} has no instances ending after {}", getDebugInfo(event.getEvent()), actionInfo.getNow());

            return Option.empty();
        }
    }

    private static String getDebugInfo(Event event) {
        val id = event.getValueIfSet(EventFields.ID);
        val recurrenceId = event.getValueIfSet(EventFields.RECURRENCE_ID);

        return Option.empty().plus(id).plus(recurrenceId).mkString("{", " / ", "}");
    }

    public interface PatchResult {
        @Value
        class ConferenceCreated implements PatchResult {
            String conferenceUri;
            PassportUid creatorUid;
        }

        class ConferencePromised implements PatchResult {
        }

        @Value
        class ConferenceCleared implements PatchResult {
            long eventId;
        }

        class NothingToDoFurther implements PatchResult {
        }
    }

    @AllArgsConstructor
    private static class SchedulingInstance {
        final long eventId;
        final InstantInterval instanceInterval;
        final InstantInterval scheduleSuitableInterval;

        public Instant getStart() {
            return instanceInterval.getStart();
        }

        public Instant getScheduleTime() {
            return scheduleSuitableInterval.getStart();
        }
    }
}
