package ru.yandex.calendar.logic.todo;

import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.EmptyResultDataAccessException;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.calendar.logic.beans.GenericBeanDao;
import ru.yandex.calendar.logic.beans.generated.TodoItem;
import ru.yandex.calendar.logic.beans.generated.TodoItemFields;
import ru.yandex.calendar.logic.beans.generated.TodoItemHelper;
import ru.yandex.calendar.logic.beans.generated.TodoList;
import ru.yandex.calendar.logic.beans.generated.TodoListFields;
import ru.yandex.calendar.logic.beans.generated.TodoListHelper;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.todo.id.ListIdOrExternalId;
import ru.yandex.calendar.logic.todo.id.TodoIdOrExternalId;
import ru.yandex.calendar.util.db.CalendarJdbcDaoSupport;
import ru.yandex.calendar.util.db.FieldsAssignment;
import ru.yandex.commune.test.random.RunWithRandomTest;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.cache.Cache;
import ru.yandex.misc.cache.tl.TlCache;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.db.q.SqlOrder;
import ru.yandex.misc.db.q.SqlPart;
import ru.yandex.misc.db.q.SqlQuery;
import ru.yandex.misc.db.q.SqlQueryUtils;
import ru.yandex.misc.lang.Validate;

/**
 * @author Stepan Koltsov
 */
public class TodoDao extends CalendarJdbcDaoSupport {

    @Autowired
    private GenericBeanDao genericBeanDao;

    private final Cache<Long, Instant> listsLastUpdateTsCache = TlCache.asCache(
            TodoDao.class.getSimpleName() + ".listsLastUpdateTs");


    private ListF<TodoItem> findTodoItemsCreatedByUid(PassportUid creatorUid, SqlCondition todoItemCondition) {
        return genericBeanDao.loadBeans(TodoItemHelper.INSTANCE,
                TodoItemFields.CREATOR_UID.eq(creatorUid).and(todoItemCondition));
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public PassportUid findTodoListCreatorUidById(long todoListId) {
        String q = "SELECT creator_uid FROM todo_list WHERE id = ?";
        return new PassportUid(getJdbcTemplate().queryForObject(q, Long.class, todoListId));
    }

    public long findNotDeletedTodoListsCountByCreatorUidAndIds(
            PassportUid uid, ListF<ListIdOrExternalId> todoListIds)
    {
        SqlCondition idCondition = ListIdOrExternalId.idsInSetCondition(todoListIds);
        String q = "SELECT COUNT(*) FROM todo_list WHERE deleted = FALSE AND creator_uid = ? " + idCondition.andSql();
        return getJdbcTemplate().queryForLong(q, uid, idCondition.args());
    }

    @RunWithRandomTest
    public boolean existsAndNotDeletedTodoListWithIdCreatedByUid(long todoListId, PassportUid uid) {
        String q = "SELECT 1 FROM todo_list WHERE deleted = FALSE AND id = ? AND creator_uid = ?";
        return getJdbcTemplate().queryForOption(q, Integer.class, todoListId, uid).isPresent();
    }

    @RunWithRandomTest
    public void updateTodoListsSetDeletedByIds(ListF<Long> ids, ActionInfo actionInfo) {
        SqlPart info = deletionInfoAssignment(actionInfo);
        String q = "UPDATE todo_list SET deleted = TRUE, " + info.sql() + " WHERE id " + SqlQueryUtils.inSet(ids);

        if (skipQuery(ids, q, info.args())) return;

        getJdbcTemplate().update(q, info.args());

        listsLastUpdateTsCache.putInCacheBatch(ids.map(id -> Tuple2.tuple(id, actionInfo.getNow())));
    }

    @RunWithRandomTest(possible = EmptyResultDataAccessException.class)
    public void updateTodoListSetNotDeletedByIds(ListIdOrExternalId todoListId, ActionInfo actionInfo) {
        TodoList data = new TodoList();
        data.setDeleted(false);

        updateTodoListById(data, todoListId, actionInfo);
    }

    @RunWithRandomTest
    public int updateTodoListsStaleLastUpdateTs(ListF<Long> listIds, ActionInfo actionInfo) {
        SqlPart info = modificationInfoAssignment(actionInfo);

        listIds = listIds.filterNot(id -> Option.x(listsLastUpdateTsCache.getFromCache(id))
                .exists(ts -> !actionInfo.getNow().isAfter(ts)));

        SqlCondition c = TodoListFields.ID.column().inSet(listIds)
                .and(TodoListFields.LAST_UPDATE_TS.lt(actionInfo.getNow()));

        String q = "UPDATE todo_list SET " + info.sql() + c.whereSql();

        if (skipQuery(c, q, info.args(), c.args())) return 0;

        int updated = getJdbcTemplate().update(q, info.args(), c.args());

        listsLastUpdateTsCache.putInCacheBatch(listIds.map(id -> Tuple2.tuple(id, actionInfo.getNow())));

        return updated;
    }

    @RunWithRandomTest
    public ListF<TodoList> findNotDeletedUserTodoLists(PassportUid uid) {
        String q = "SELECT * FROM todo_list WHERE creator_uid = ? AND deleted = FALSE ORDER BY creation_ts DESC";
        return getJdbcTemplate().queryForList(q, TodoList.class, uid);
    }

    @RunWithRandomTest
    public Option<TodoList> findFirstCreatedNotDeletedUserTodoList(PassportUid uid) {
        String q = "SELECT * FROM todo_list WHERE creator_uid = ? AND deleted = FALSE ORDER BY creation_ts LIMIT 1";
        return getJdbcTemplate().queryForOption(q, TodoList.class, uid);
    }

    public TodoList findNotDeletedTodoListById(ListIdOrExternalId listId) {
        return findNotDeletedTodoListByIdO(listId).get();
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public Option<TodoList> findNotDeletedTodoListByIdO(ListIdOrExternalId listId) {
        String q = "SELECT * FROM todo_list WHERE deleted = FALSE " + listId.idEq().andSql();
        return getJdbcTemplate().queryForOption(q, TodoList.class, listId.idEq().args());
    }

    @RunWithRandomTest
    public Option<Long> findNotDeletedTodoListIdByCaldavCollId(PassportUid creatorUid, String collectionId) {
        String q = "SELECT id FROM todo_list WHERE deleted = FALSE AND creator_uid = ? AND caldav_coll_id = ?";
        return getJdbcTemplate().queryForOption(q, Long.class, creatorUid, collectionId);
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public TodoList findTodoListById(ListIdOrExternalId listId) {
        String q = "SELECT * FROM todo_list WHERE " + listId.idEq().sql();
        return getJdbcTemplate().queryForObject(q, TodoList.class, listId.idEq().args());
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public void updateTodoListTitleAndDescription(long todoListId, String title, String description, ActionInfo actionInfo) {
        SqlPart info = modificationInfoAssignment(actionInfo);
        String q = "UPDATE todo_list SET title = ?, description = ?, " + info.sql() + " WHERE id = ?";
        getJdbcTemplate().updateRow(q, title, description, info.args(), todoListId);

        listsLastUpdateTsCache.putInCache(todoListId, actionInfo.getNow());
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public void updateTodoItemTitleAndPos(long todoItemId, String title, int pos, ActionInfo actionInfo) {
        SqlPart info = modificationInfoAssignment(actionInfo);
        String q = "UPDATE todo_item SET title = ?, pos = ?, " + info.sql() + " WHERE id = ?";
        getJdbcTemplate().updateRow(q, title, pos, info.args(), todoItemId);
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public void updateTodoItemPos(long todoItemId, int pos, ActionInfo actionInfo) {
        SqlPart info = modificationInfoAssignment(actionInfo);
        String q = "UPDATE todo_item SET pos = ?, " + info.sql() + " WHERE id = ?";
        getJdbcTemplate().updateRow(q, pos, info.args(), todoItemId);
    }

    public long saveTodoList(TodoList todoListData, ActionInfo actionInfo) {
        Validate.V.isFalse(todoListData.getFieldValueO(TodoListFields.DELETED).getOrElse(false));

        TodoList tmp = todoListData.copy();
        tmp.setCreationTs(actionInfo.getNow());
        tmp.setCreationSource(actionInfo.getActionSource());
        tmp.setLastUpdateTs(actionInfo.getNow());

        long id = genericBeanDao.insertBeanGetGeneratedKey(tmp);

        listsLastUpdateTsCache.putInCache(id, actionInfo.getNow());

        return id;
    }

    public long saveTodoItem(TodoItem todoItemData, ActionInfo actionInfo) {
        Validate.V.isFalse(todoItemData.getFieldValueO(TodoItemFields.DELETED).getOrElse(false));

        TodoItem tmp = todoItemData.copy();
        tmp.setCreationTs(actionInfo.getNow());
        tmp.setCreationSource(actionInfo.getActionSource());
        tmp.setLastUpdateTs(actionInfo.getNow());

        return genericBeanDao.insertBeanGetGeneratedKey(tmp);
    }

    @RunWithRandomTest(possible = EmptyResultDataAccessException.class)
    public TodoItem findNotDeletedNotArchivedTodoItemById(TodoIdOrExternalId id, PassportUid uid) {
        return findNotDeletedNotArchivedTodoItemByIds(Cf.list(id), uid).singleO().getOrThrow(() ->
                new EmptyResultDataAccessException("Todo item not found by id " + id + " and uid " + uid, 1));
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public ListF<TodoItem> findNotDeletedNotArchivedTodoItemByIds(ListF<TodoIdOrExternalId> itemIds, PassportUid uid) {
        return findNotDeletedNotArchivedUserTodoItems(uid, TodoIdOrExternalId.idsInSetCondition(itemIds));
    }

    public ListF<TodoItem> findNotDeletedNotArchivedTodoItems(SqlCondition condition) {
        return findNotDeletedNotArchivedTodoItems(condition, SqlOrder.unordered());
    }

    public ListF<TodoItem> findNotDeletedNotArchivedTodoItems(SqlCondition condition, SqlOrder order) {
        return findNotDeletedNotArchivedTodoItems(condition, order, SqlLimits.all());
    }

    @RunWithRandomTest
    public ListF<TodoItem> findNotDeletedNotArchivedTodoItems(SqlCondition condition, SqlOrder order, SqlLimits limits) {
        condition = condition.and(TodoItemFields.DELETED.eq(false));
        condition = condition.and(TodoItemFields.ARCHIVED.eq(false));
        return genericBeanDao.loadBeans(TodoItemHelper.INSTANCE, condition, order, limits);
    }

    public ListF<TodoItem> findNotDeletedNotArchivedUserTodoItems(
            PassportUid uid, SqlCondition condition, SqlOrder order, SqlLimits limits)
    {
        return findNotDeletedNotArchivedTodoItems(TodoItemFields.CREATOR_UID.eq(uid).and(condition), order, limits);
    }

    @RunWithRandomTest
    public long findNotDeletedNotArchivedTodoItemsCount(PassportUid uid, SqlCondition condition) {
        condition = condition.and(TodoItemFields.CREATOR_UID.eq(uid));
        condition = condition.and(TodoItemFields.DELETED.eq(false));
        condition = condition.and(TodoItemFields.ARCHIVED.eq(false));

        return genericBeanDao.countBeans(TodoItemHelper.INSTANCE, condition);
    }

    @RunWithRandomTest
    public ListF<TodoItem> findNotDeletedNotArchivedUserTodoItems(PassportUid uid, SqlCondition condition) {
        return findNotDeletedNotArchivedUserTodoItems(uid, condition, SqlOrder.unordered(), SqlLimits.all());
    }

    public ListF<TodoItem> findNotDeletedUserTodoItems(PassportUid uid, SqlCondition condition) {
        return findNotDeletedUserTodoItems(uid, condition, SqlOrder.unordered(), SqlLimits.all());
    }

    public ListF<TodoItem> findNotDeletedUserTodoItems(
            PassportUid uid, SqlCondition condition, SqlOrder order, SqlLimits limits)
    {
        return genericBeanDao.loadBeans(TodoItemHelper.INSTANCE, condition
                .and(TodoItemFields.CREATOR_UID.eq(uid))
                .and(TodoItemFields.DELETED.eq(false)), order, limits);
    }

    @RunWithRandomTest(possible = EmptyResultDataAccessException.class)
    public TodoItem findNotDeletedTodoItemByIdAndCreatorUid(TodoIdOrExternalId id, PassportUid uid) {
        return findNotDeletedUserTodoItems(uid, id.idEq()).singleO().getOrThrow(() ->
                new EmptyResultDataAccessException("Todo item not found by id " + id + " and uid " + uid, 1));
    }

    public long updateTodoItemGetListId(TodoItem todoItem, long todoItemId, ActionInfo actionInfo) {
        Validate.V.isFalse(todoItem.getFieldValueO(TodoItemFields.DELETED).getOrElse(false));

        Validate.V.isFalse(todoItem.isFieldSet(TodoItemFields.ID));
        Validate.V.isFalse(todoItem.isFieldSet(TodoItemFields.EXTERNAL_ID));

        TodoItem tmp = todoItem.copy();
        tmp.setLastUpdateTs(actionInfo.getNow());
        tmp.setModificationSource(actionInfo.getActionSource());

        SqlQuery q = SqlQueryUtils.updateQuery("todo_item",
                tmp.getFieldValues().map1(f -> f.column().name()), TodoItemFields.ID.eq(todoItemId));

        return getJdbcTemplate().queryForLong(q.sql() + " RETURNING todo_list_id", q.args());
    }

    public void updateTodoListById(TodoList todoList, ListIdOrExternalId todoListId, ActionInfo actionInfo) {
        Validate.V.isFalse(todoList.isFieldSet(TodoListFields.ID));
        Validate.V.isFalse(todoList.isFieldSet(TodoListFields.EXTERNAL_ID));

        TodoList tmp = todoList.copy();
        tmp.setModificationSource(actionInfo.getActionSource());
        tmp.setLastUpdateTs(actionInfo.getNow());

        SqlQuery q = SqlQueryUtils.updateQuery("todo_list",
                tmp.getFieldValues().map1(f -> f.column().name()), todoListId.idEq());

        long id = getJdbcTemplate().queryForLong(q.sql() + " RETURNING id", q.args());

        listsLastUpdateTsCache.putInCache(id, actionInfo.getNow());
    }

    @RunWithRandomTest
    public Option<TodoItem> findNotDeletedNotArchivedTodoItemByExternalIdAndCreatorUid(String externalId, PassportUid uid) {
        return findNotDeletedNotArchivedTodoItemByIds(Cf.list(TodoIdOrExternalId.externalId(externalId)), uid).singleO();
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public ListF<TodoItem> findNotDeletedNotArchivedTodoItemsByExternalIdsAndCreatorUid(PassportUid uid, ListF<String> externalIds) {
        return findNotDeletedNotArchivedTodoItemByIds(externalIds.map(TodoIdOrExternalId::externalId), uid);
    }

    @RunWithRandomTest
    public ListF<TodoItem> findNotDeletedNotArchivedTodoItemsByExternalIdsAndTodoListId(ListF<String> externalIds, long todoListId) {
        SqlCondition condition = TodoItemFields.EXTERNAL_ID.column().inSet(externalIds);
        condition = condition.and(TodoItemFields.TODO_LIST_ID.eq(todoListId));
        return findNotDeletedNotArchivedTodoItems(condition, SqlOrder.unordered());
    }

    @RunWithRandomTest
    public ListF<TodoList> findNotDeletedTodoLists(SqlCondition c) {
        c = c.and(TodoListFields.DELETED.eq(false));
        return findTodoLists(c);
    }

    @RunWithRandomTest
    public ListF<TodoList> findTodoLists(SqlCondition c) {
        return genericBeanDao.loadBeans(TodoListHelper.INSTANCE, c);
    }

    @RunWithRandomTest
    public ListF<TodoList> findTodoLists(SqlCondition condition, SqlOrder order, SqlLimits limits) {
        return genericBeanDao.loadBeans(TodoListHelper.INSTANCE, condition, order, limits);
    }

    @RunWithRandomTest
    public ListF<TodoList> findNotDeletedTodoListsByUid(ListF<PassportUid> uid) {
        return findNotDeletedTodoLists(TodoListFields.CREATOR_UID.column().inSet(uid));
    }

    @RunWithRandomTest
    public ListF<TodoList> findTodoListsByUid(ListF<PassportUid> uid) {
        SqlCondition c = TodoListFields.CREATOR_UID.column().inSet(uid);
        return findTodoLists(c);
    }

    @RunWithRandomTest
    public ListF<TodoItem> findNotDeletedNotArchivedTodoItemsByTodoListIds(ListF<Long> todoListIds) {
        String q = "SELECT * FROM todo_item WHERE deleted = FALSE AND archived = FALSE AND todo_list_id " +
                SqlQueryUtils.inSet(todoListIds);

        if (skipQuery(todoListIds, q)) return Cf.list();

        return getJdbcTemplate().queryForList(q, TodoItem.class);
    }

    @RunWithRandomTest
    public ListF<TodoItem> findNotDeletedNotArchivedNotCompletedTodoItemsByTodoListIds(ListF<Long> todoListIds) {
        String q = "SELECT * FROM todo_item " +
                "WHERE deleted = FALSE AND archived = FALSE AND completion_ts IS NULL AND todo_list_id " +
                SqlQueryUtils.inSet(todoListIds);

        if (skipQuery(todoListIds, q)) return Cf.list();

        return getJdbcTemplate().queryForList(q, TodoItem.class);
    }

    @RunWithRandomTest
    public ListF<Long> findNotDeletedNotArchivedNotCompletedTodoItemIdsByTodoListId(long todoListId) {
        SqlCondition c = SqlCondition.all(
                TodoItemFields.DELETED.eq(false),
                TodoItemFields.ARCHIVED.eq(false),
                TodoItemFields.COMPLETION_TS.column().isNull(),
                TodoItemFields.TODO_LIST_ID.eq(todoListId));

        return getJdbcTemplate().queryForList("SELECT id FROM todo_item" + c.whereSql(), Long.class, c.args());
    }

    @RunWithRandomTest
    public long findNotDeletedNotArchivedNotCompletedTodoItemsCount(PassportUid uid) {
        return getUserNotDeletedNotArchivedTodoItemsCount(uid, TodoItemFields.COMPLETION_TS.column().isNull());
    }

    @RunWithRandomTest
    public ListF<TodoItem> findNotDeletedNotArchivedTodoItemsByUidAndMinLastUpdateTs(PassportUid uid, Instant since) {
        SqlCondition condition =
                TodoItemFields.DELETED.eq(false).and(
                TodoItemFields.ARCHIVED.eq(false)).and(
                TodoItemFields.LAST_UPDATE_TS.ge(since));
        return findTodoItemsCreatedByUid(uid, condition);
    }

    @RunWithRandomTest
    public ListF<TodoItem> findDeletedTodoItemsByUidAndMinDeletedTs(PassportUid uid, Instant since) {
        SqlCondition condition =
                TodoItemFields.DELETED.eq(true).and(
                TodoItemFields.DELETION_TS.ge(since));
        return findTodoItemsCreatedByUid(uid, condition);
    }

    @RunWithRandomTest
    public ListF<TodoItem> findNotDeletedNotArchivedTodoItemsByTodoListIdAndMinLastUpdateTs(long todoListId, Instant since) {
        String q = "SELECT * FROM todo_item WHERE todo_list_id = ? AND last_update_ts >= ?" +
                " AND deleted = FALSE AND archived = FALSE";
        return getJdbcTemplate().queryForList(q, TodoItem.class, todoListId, since);
    }

    @RunWithRandomTest
    public ListF<TodoItem> findDeletedTodoItemsByTodoListIdAndMinDeletedTs(long todoListId, Instant since) {
        String q = "SELECT * FROM todo_item WHERE todo_list_id = ? AND deleted = TRUE AND deletion_ts >= ?";
        return getJdbcTemplate().queryForList(q, TodoItem.class, todoListId, since);
    }

    @RunWithRandomTest
    public ListF<TodoItem> findNotDeletedNotArchivedTodoItemsByUid(PassportUid uid) {
        return findNotDeletedNotArchivedUserTodoItems(uid, SqlCondition.trueCondition());
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public void updateTodoItemsSetDeletedByIds(ListF<Long> todoItemIds, ActionInfo actionInfo) {
        SqlPart info = deletionInfoAssignment(actionInfo);
        String sql = "UPDATE todo_item SET deleted = TRUE, " + info.sql() +
                " WHERE id " + SqlQueryUtils.inSet(todoItemIds);

        if (skipQuery(todoItemIds, sql, info.args())) return;

        getJdbcTemplate().update(sql, info.args());
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public ListF<Long> updateTodoItemsSetArchivedByIdsGetListIds(ListF<TodoIdOrExternalId> todoItemIds, ActionInfo actionInfo) {
        SqlPart info = modificationInfoAssignment(actionInfo);
        SqlCondition idCondition = TodoIdOrExternalId.idsInSetCondition(todoItemIds);
        String sql = "UPDATE todo_item SET archived = TRUE, archived_ts = ?, " + info.sql() +
                " WHERE " + idCondition.sql() + " RETURNING todo_list_id";

        return getJdbcTemplate().queryForList(sql, Long.class, actionInfo.getNow(), info.args(), idCondition.args());
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public ListF<Long> updateTodoItemsSetNotArchivedByIdsGetListIds(ListF<TodoIdOrExternalId> todoItemIds, ActionInfo actionInfo) {
        SqlPart info = modificationInfoAssignment(actionInfo);
        SqlCondition idCondition = TodoIdOrExternalId.idsInSetCondition(todoItemIds);
        String sql = "UPDATE todo_item SET archived = FALSE, " + info.sql()
                + " WHERE " + idCondition.sql() + " RETURNING todo_list_id";
        return getJdbcTemplate().queryForList(sql, Long.class, info.args(), idCondition.args());
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public void updateTodoItemsSetNotArchivedAndNotCompletedByIds(ListF<TodoIdOrExternalId> todoItemIds, ActionInfo actionInfo) {
        SqlPart info = modificationInfoAssignment(actionInfo);
        SqlCondition idCondition = TodoIdOrExternalId.idsInSetCondition(todoItemIds);
        String sql = "UPDATE todo_item SET archived = FALSE, archived_ts = NULL, completion_ts = NULL, " + info.sql() +
                " WHERE " + idCondition.sql();
        getJdbcTemplate().update(sql, info.args(), idCondition.args());
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public void updateTodoItemsSetNotDeletedByIds(ListF<Long> todoItemIds, ActionInfo actionInfo) {
        SqlPart info = modificationInfoAssignment(actionInfo);
        String sql = "UPDATE todo_item SET deleted = FALSE, " + info.sql() +
                " WHERE id " + SqlQueryUtils.inSet(todoItemIds);

        if (skipQuery(todoItemIds, sql, info.args())) return;

        getJdbcTemplate().update(sql, info.args());
    }

    @RunWithRandomTest
    public void updateTodoItemsSetDeletedByTodoListIds(ListF<Long> todoListIds, ActionInfo actionInfo) {
        SqlPart info = deletionInfoAssignment(actionInfo);
        SqlCondition c = TodoItemFields.TODO_LIST_ID.column().inSet(todoListIds).and(TodoItemFields.DELETED.eq(false));

        String q = "UPDATE todo_item SET deleted = TRUE, " + info.sql() + c.whereSql();

        if (skipQuery(todoListIds, q, info.args(), c.args())) return;

        getJdbcTemplate().update(q, info.args(), c.args());
    }

    @RunWithRandomTest
    public int updateNotDeletedNotArchivedTodoItemsPosition(
            long todoListId, Tuple2List<Long, Integer> positions, ActionInfo actionInfo)
    {
        SqlPart info = modificationInfoAssignment(actionInfo);

        String q = "UPDATE todo_item SET pos = ?, " + info.sql()
                + " WHERE pos != ? AND id = ? AND todo_list_id = ? AND deleted = FALSE AND archived = FALSE";

        ListF<ListF<?>> args = positions.map(
                ip -> Cf.<Object>list(ip.get2()).plus(info.args()).plus(Cf.list(ip.get2(), ip.get1(), todoListId)));

        return skipQuery(args, q, args) ? 0 : Cf.intList(genericBeanDao.batchUpdate(q, args)).sum(Cf.Integer);
    }

    @RunWithRandomTest
    public int findLastTodoItemPositionWithinList(final long todoListId, final boolean isCompleted) {
        SqlCondition completionCond = isCompleted ?
                TodoItemFields.COMPLETION_TS.column().isNotNull() :
                TodoItemFields.COMPLETION_TS.column().isNull();
        String q = "SELECT MAX(pos) FROM todo_item WHERE todo_list_id = ? " + completionCond.andSql();
        return getJdbcTemplate().queryForInt(q, todoListId);
    }

    @RunWithRandomTest
    public int getUserNotDeletedListsCount(PassportUid uid, SqlCondition todoListCondition) {
        String q = "SELECT COUNT(*) FROM todo_list WHERE creator_uid = ? AND deleted = FALSE" + todoListCondition.andSql();
        return getJdbcTemplate().queryForInt(q, uid, todoListCondition.args());
    }

    @RunWithRandomTest
    public int getUserNotDeletedListsCount(PassportUid uid) {
        return getUserNotDeletedListsCount(uid, SqlCondition.trueCondition());
    }

    @RunWithRandomTest
    public int getUserCreatedListsCount(PassportUid uid) {
        String q = "SELECT COUNT(*) FROM todo_list WHERE creator_uid = ?";
        return getJdbcTemplate().queryForInt(q, uid);
    }

    @RunWithRandomTest
    public int getUserNotDeletedNotArchivedTodoItemsCount(PassportUid uid, SqlCondition todoItemCondition) {
        SqlCondition c = SqlCondition.trueCondition()
                .and(TodoItemFields.CREATOR_UID.eq(uid))
                .and(TodoItemFields.DELETED.eq(false))
                .and(TodoItemFields.ARCHIVED.eq(false))
                .and(todoItemCondition);

        return getJdbcTemplate().queryForInt("SELECT COUNT(1) FROM todo_item" + c.whereSql(), c.args());
    }

    @RunWithRandomTest
    public Tuple2List<Long, Long> findNotDeletedNotArchivedNotCompletedTodoItemsCountsByTodoListIds(
            ListF<Long> listIds)
    {
        String q = "SELECT todo_list_id, COUNT(*) FROM todo_item " +
                " WHERE todo_list_id " + SqlQueryUtils.inSet(listIds) +
                " AND deleted = FALSE AND archived = FALSE AND completion_ts IS NULL" +
                " GROUP BY todo_list_id";

        if (skipQuery(listIds, q)) return Tuple2List.tuple2List();

        return getJdbcTemplate().queryForList2(q, Long.class, Long.class);
    }

    @RunWithRandomTest
    public ListF<TodoItem> findNotDeletedNotArchivedTodoItemsByTodoListIdsAndMinLastUpdateTs(ListF<Long> listIds, Instant modifiedSince) {
        String q = "SELECT * FROM todo_item " +
                " WHERE todo_list_id " + SqlQueryUtils.inSet(listIds) + " AND last_update_ts >= ?" +
                " AND deleted = FALSE AND archived = FALSE";

        if (skipQuery(listIds, q, modifiedSince)) return Cf.list();

        return getJdbcTemplate().queryForList(q, TodoItem.class, modifiedSince);
    }

    @RunWithRandomTest
    public ListF<TodoItem> findDeletedTodoItemsByTodoListIdsAndMinDeletedTs(ListF<Long> listIds, Instant deletedSince) {
        String q = "SELECT * FROM todo_item WHERE todo_list_id " + SqlQueryUtils.inSet(listIds) + " AND deleted = TRUE AND deletion_ts >= ?";

        if (skipQuery(listIds, q, deletedSince)) return Cf.list();

        return getJdbcTemplate().queryForList(q, TodoItem.class, deletedSince);
    }

    public ListF<TodoList> findDeletedTodoListsByUidAndMinDeletedTs(PassportUid uid, Instant deletedSince) {
        String q = "SELECT * FROM todo_list WHERE creator_uid = ? AND deleted = TRUE AND deletion_ts >= ?";
        return getJdbcTemplate().queryForList(q, TodoList.class, uid, deletedSince);
    }

    @RunWithRandomTest
    public ListF<TodoList> findNotDeletedTodoListsByUidAndMinLastUpdateTs(PassportUid uid, Instant modifiedSince) {
        String q = "SELECT * FROM todo_list WHERE creator_uid = ? AND deleted = FALSE AND last_update_ts >= ?";
        return getJdbcTemplate().queryForList(q, TodoList.class, uid, modifiedSince);
    }

    @RunWithRandomTest
    public ListF<TodoItem> findNotDeletedCompletedTodosByCreatorUid(PassportUid uid, int offset, int limit) {
        return findNotDeletedUserTodoItems(
                uid, TodoItemFields.COMPLETION_TS.column().isNotNull(),
                SqlOrder.orderByColumnDesc("completion_ts"), SqlLimits.range(offset, limit));
    }

    @RunWithRandomTest
    public ListF<TodoItem> findNotDeletedArchivedTodosByCreatorUid(PassportUid uid) {
        return findNotDeletedUserTodoItems(uid, TodoItemFields.ARCHIVED.eq(true));
    }

    @RunWithRandomTest
    public ListF<TodoItem> findNotDeletedCompletedTodosByCreatorUid(PassportUid uid) {
        return findNotDeletedUserTodoItems(uid, TodoItemFields.COMPLETION_TS.column().isNotNull());
    }

    @RunWithRandomTest
    public int findNotDeletedCompletedTodosCount(PassportUid uid) {
        SqlCondition c = SqlCondition.trueCondition()
                .and(TodoItemFields.CREATOR_UID.eq(uid))
                .and(TodoItemFields.DELETED.eq(false))
                .and(TodoItemFields.COMPLETION_TS.column().isNotNull());

        return getJdbcTemplate().queryForInt("SELECT COUNT(1) FROM todo_item" + c.whereSql(), c.args());
    }

    @RunWithRandomTest
    public Option<Instant> findMaxTodoListLastUpdateTs(PassportUid uid) {
        String q = "SELECT MAX(last_update_ts) FROM todo_list WHERE creator_uid = ?";
        return Option.ofNullable(getJdbcTemplate().queryForObject(q, Instant.class, uid));
    }

    @RunWithRandomTest
    public Option<Instant> findTodoListLastUpdateTs(long todoListId) {
        return getJdbcTemplate().queryForOption(
                "SELECT last_update_ts FROM todo_list WHERE id = ?", Instant.class, todoListId);
    }

    @RunWithRandomTest
    public ListF<TodoItem> findNotDeletedNotArchivedTodoItemsByCreatorUid(
            PassportUid uid, int offset, int limit, SqlCondition todoItemCondition)
    {
        return findNotDeletedUserTodoItems(
                uid, TodoItemFields.ARCHIVED.eq(false).and(todoItemCondition),
                SqlOrder.orderByColumn("id"), SqlLimits.range(offset, limit));
    }

    @RunWithRandomTest
    public ListF<TodoList> findNotDeletedTodoListsByCreatorUid(
            PassportUid uid, int offset, int limit, SqlCondition todoListCondition)
    {
        String q =
            "SELECT * FROM todo_list " +
            "WHERE creator_uid = ? AND deleted = FALSE" + todoListCondition.andSql() + " " +
            "ORDER BY id DESC LIMIT ? OFFSET ?";
        return getJdbcTemplate().queryForList(q, TodoList.class, uid, todoListCondition.args(), limit, offset);
    }

    @RunWithRandomTest
    public ListF<TodoList> findNonDeletedTodoListsByCreatorUid(PassportUid uid) {
        String q = "SELECT * FROM todo_list WHERE creator_uid = ? AND deleted = FALSE";
        return getJdbcTemplate().queryForList(q, TodoList.class, uid);
    }

    @RunWithRandomTest
    public boolean findTodoListExistsByExternalIdAndUid(String externalId, PassportUid uid) {
        String q = "SELECT EXISTS (SELECT 1 FROM todo_list WHERE external_id = ? AND creator_uid = ?)";
        return getJdbcTemplate().queryForObject(q, Boolean.class, externalId, uid);
    }

    @RunWithRandomTest
    public boolean findNotDeletedTodoListExistsById(ListIdOrExternalId todoListId) {
        String q = "SELECT EXISTS (SELECT 1 FROM todo_list WHERE deleted = false " + todoListId.idEq().andSql() + ")";
        return getJdbcTemplate().queryForObject(q, Boolean.class, todoListId.idEq().args());
    }

    @RunWithRandomTest
    public Option<Long> findNotDeletedTodoListIdById(ListIdOrExternalId todoListId) {
        String q = "SELECT id FROM todo_list WHERE deleted = false " + todoListId.idEq().andSql();
        return getJdbcTemplate().queryForOption(q, Long.class, todoListId.idEq().args());
    }

    @RunWithRandomTest
    public ListF<TodoItem> findDeletedTodoItemsByTodoItemIds(ListF<TodoIdOrExternalId> todoItemIds, PassportUid uid) {
        return findTodoItemsCreatedByUid(uid, TodoItemFields.DELETED.eq(true)
                .and(TodoIdOrExternalId.idsInSetCondition(todoItemIds)));
    }

    @RunWithRandomTest
    public ListF<TodoItem> findNotDeletedCompletedTodosByTodoListsAndMinCompletionTs(
            ListF<Long> todoListIds, Instant completedSince)
    {
        return genericBeanDao.loadBeans(TodoItemHelper.INSTANCE, SqlCondition.trueCondition()
                .and(TodoItemFields.TODO_LIST_ID.column().inSet(todoListIds))
                .and(TodoItemFields.DELETED.eq(false))
                .and(TodoItemFields.COMPLETION_TS.ge(completedSince)));
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public long updateTodoItemNotificationTsClearNotificationSentTsAndNotificationShownInWebGetListId(
            long todoItemId, Instant notificationTs)
    {
        String q = "UPDATE todo_item" +
                " SET notification_ts = ?, notification_sent_ts = NULL, notification_shown_in_web = FALSE" +
                " WHERE id = ? RETURNING todo_list_id";

        return getJdbcTemplate().queryForLong(q, notificationTs, todoItemId);
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public long updateTodoItemClearNotificationTsGetListId(long todoItemId) {
        String q = "UPDATE todo_item" +
                " SET notification_ts = NULL, notification_sent_ts = NULL, notification_shown_in_web = FALSE" +
                " WHERE id = ? RETURNING todo_list_id";

       return getJdbcTemplate().queryForLong(q, todoItemId);
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public long updateTodoItemNotificationTsGetListId(long todoItemId, Instant notificationTs) {
        String q = "UPDATE todo_item SET notification_ts = ?" +
                " WHERE id = ? RETURNING todo_list_id";
        return getJdbcTemplate().queryForLong(q, notificationTs, todoItemId);
    }

    @RunWithRandomTest
    public void updateTodoItemsSetNotDeletedByTodoListIdAndDeletionTs(
            long todoListId, Instant deletionTs, ActionInfo actionInfo)
    {
        SqlPart info = modificationInfoAssignment(actionInfo);
        SqlCondition c = TodoItemFields.TODO_LIST_ID.eq(todoListId).and(TodoItemFields.DELETION_TS.eq(deletionTs));

        String sql = "UPDATE todo_item SET deleted = FALSE, " + info.sql() + c.whereSql();
        getJdbcTemplate().update(sql, info.args(), c.args());
    }

    @RunWithRandomTest
    public ListF<Long> updateNotDeletedTodoItemsWithStartOrDueTsSetLastUpdateTsGetListIds(PassportUid uid, Instant now) {
        SqlCondition c = SqlCondition.trueCondition()
                .and(TodoItemFields.CREATOR_UID.eq(uid))
                .and(TodoItemFields.DELETED.eq(false))
                .and(TodoItemFields.START_TS.column().isNotNull().or(TodoItemFields.DUE_TS.column().isNotNull()));

        return getJdbcTemplate().queryForList("UPDATE todo_item SET last_update_ts = ?"
                + c.whereSql() + " RETURNING todo_list_id", Long.class, now, c.args());
    }

    @RunWithRandomTest
    public ListF<TodoItem> getTodoItemsForNotification(Instant minNotificationTs, Instant maxNotificationTs) {
        String q = "SELECT * FROM todo_item" +
                " WHERE notification_sent_ts IS NULL AND notification_ts >= ? AND notification_ts < ?";
        return getJdbcTemplate().queryForList(q, TodoItem.class, minNotificationTs, maxNotificationTs);
    }

    @RunWithRandomTest
    public void deleteTodoItemsByTodoListIds(ListF<Long> todoListIds) {
        String q = "DELETE FROM todo_item WHERE todo_list_id " + SqlQueryUtils.inSet(todoListIds);

        if (skipQuery(todoListIds, q)) return;

        getJdbcTemplate().update(q);
    }

    @RunWithRandomTest
    public void deleteTodoListsByTodoListIds(ListF<Long> todoListIds) {
        String q = "DELETE FROM todo_list WHERE id " + SqlQueryUtils.inSet(todoListIds);

        if (skipQuery(todoListIds, q)) return;

        getJdbcTemplate().update(q);
    }

    private static SqlPart modificationInfoAssignment(ActionInfo actionInfo) {
        return new FieldsAssignment(Tuple2List.<String, Object>fromPairs(
                "modification_source", actionInfo.getActionSource(),
                "last_update_ts", actionInfo.getNow()));
    }

    private static SqlPart deletionInfoAssignment(ActionInfo actionInfo) {
        return new FieldsAssignment(Tuple2List.<String, Object>fromPairs(
                "deletion_source", actionInfo.getActionSource(),
                "deletion_ts", actionInfo.getNow(),
                "last_update_ts", actionInfo.getNow()));
    }
} //~
