package ru.yandex.calendar.logic.todo;

import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDateTime;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.frontend.web.cmd.run.Situation;
import ru.yandex.calendar.frontend.worker.task.SendTodoMailTask;
import ru.yandex.calendar.logic.beans.generated.TodoItem;
import ru.yandex.calendar.logic.beans.generated.TodoItemFields;
import ru.yandex.calendar.logic.beans.generated.TodoList;
import ru.yandex.calendar.logic.beans.generated.TodoListEmail;
import ru.yandex.calendar.logic.beans.generated.TodoListFields;
import ru.yandex.calendar.logic.contact.UnivContact;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActorId;
import ru.yandex.calendar.logic.event.EventInvitationManager;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsCalendar;
import ru.yandex.calendar.logic.sending.param.TodoListMessageParameters;
import ru.yandex.calendar.logic.sending.param.TodoMessageParameters;
import ru.yandex.calendar.logic.sending.real.MailSender;
import ru.yandex.calendar.logic.sharing.Nick;
import ru.yandex.calendar.logic.svc.SvcRoutines;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.logic.user.SettingsInfo;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.PgOnetimeUtils;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.calendar.util.dates.ExtendedDateTime;
import ru.yandex.commune.bazinga.impl.storage.BazingaStorage;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDuplicateBehavior;
import ru.yandex.commune.mail.MailAddress;
import ru.yandex.commune.mailer.MailUtil;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox.PassportAuthDomain;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.email.Email;

/**
 * @author gutman
 */
public class TodoListEmailManager {

    private static final int MAX_EMAILS_PER_24H = 100;

    @Autowired
    private TodoListEmailDao todoListEmailDao;
    @Autowired
    private SvcRoutines svcRoutines;
    @Autowired
    private UserManager userManager;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private DateTimeManager dateTimeManager;
    @Autowired
    private EventInvitationManager eventInvitationManager;
    @Autowired
    private BazingaStorage bazingaStorage;
    @Autowired
    private TodoDao todoDao;

    @Autowired
    private MailSender mailSender;

    public void sendTodoListEmails(PassportUid uid, ListF<Email> emails, TodoList todoList,
            ListF<TodoItem> todoItems, IcsCalendar icsCalendar, MailAddress sender,
            Option<Language> languageO, ActionInfo actionInfo)
    {
        ListF<TodoListMessageParameters> messageParameterss = Cf.arrayList();

        long alreadySentEmails = todoListEmailDao.findTodoListEmailsCountSentSince(
                uid, actionInfo.getNow().minus(Duration.standardHours(24)));

        if (alreadySentEmails + emails.size() > MAX_EMAILS_PER_24H) {
            throw CommandRunException.createSituation("Too many emails per 24 hours", Situation.TOO_MANY_EMAILS);
        }

        for (Email email : emails) {
            MailAddress recipient = new MailAddress(email);
            boolean toYandexUser = PassportAuthDomain.byEmail(email) == PassportAuthDomain.YANDEX_TEAM_RU
                    || MailUtil.isYandexEmail(recipient.toInternetAddress());

            String calendarUrl = svcRoutines.getCalendarUrlForEmail(email);

            Option<PassportUid> recipientUid = userManager.getUidByEmail(email);

            Language language;
            DateTimeZone tz;
            if (toYandexUser && recipientUid.isPresent()) {
                language = languageO.getOrElse(() -> settingsRoutines.getLanguage(uid));
                tz = dateTimeManager.getTimeZoneForUid(uid);
            } else {
                language = languageO.getOrElse(settingsRoutines.getDefaultLanguage());
                tz = dateTimeManager.getDefaultTimezone();
            }
            TodoListMessageParameters messageParameters = new TodoListMessageParameters(
                    language, icsCalendar, todoList.getTitle(), todoItems,
                    sender, recipient, toYandexUser, tz, calendarUrl);

            messageParameterss.add(messageParameters);

            saveTodoListEmail(uid, todoList, actionInfo.getNow(), email);
        }

        mailSender.sendEmailsViaTask(Cf.toList(messageParameterss), actionInfo);
    }

    public Option<TodoMessageParameters> prepareTodoMail(
            PassportUid uid, TodoMailType type, ActionInfo actionInfo)
    {
        SettingsInfo settings = settingsRoutines.getSettingsByUid(uid);
        DateTimeZone userTz = settings.getTz();

        if (!type.getTime(settings.getCommon()).isPresent()) {
            return Option.empty();
        }

        ExtendedDateTime day = new ExtendedDateTime(actionInfo.getNow(), userTz);

        SqlCondition cond;

        if (type == TodoMailType.PLANNED) {
            cond = SqlCondition.trueCondition()
                    .and(TodoItemFields.DUE_TS.ge(day.toInstantAtStartOfDay()))
                    .and(TodoItemFields.DUE_TS.lt(day.toInstantAtEndOfDay()))
                    .and(TodoItemFields.COMPLETION_TS.column().isNull());

        } else if (type == TodoMailType.EXPIRED) {
            cond = SqlCondition.trueCondition()
                    .and(TodoItemFields.DUE_TS.lt(day.toInstantAtStartOfDay()))
                    .and(TodoItemFields.COMPLETION_TS.column().isNull());
        } else {
            throw new IllegalArgumentException("Unexpected type: " + type);
        }

        ListF<TodoItem> items = todoDao.findNotDeletedNotArchivedUserTodoItems(uid, cond);
        ListF<TodoList> lists = todoDao.findTodoLists(TodoListFields.ID.column().inSet(items.map(TodoItem::getTodoListId)));

        if (items.isEmpty()) {
            return Option.empty();
        }

        Language lang = Language.RUSSIAN; // CAL-9651
        MailAddress sender = eventInvitationManager.getSender(ActorId.yaCalendar()).getMailAddress(lang);
        UnivContact recipient = new UnivContact(settings.getEmail(), Nick.getRecipient(settings.getCommon()), Option.of(uid));

        return Option.of(new TodoMessageParameters(
                new LocalDateTime(actionInfo.getNow(), userTz), lists, items,
                sender, recipient, lang, svcRoutines.getCalendarUrlBySettings(settings.getCommon()), type));
    }

    public void scheduleTodoMailTasks(SettingsInfo settings, Instant now) {
        scheduleTodoMailTasks(settings, now, ActiveUidDuplicateBehavior.MERGE_IF_READY);
    }

    public void scheduleTodoMailTasksIfNotYet(SettingsInfo settings, Instant now) {
        scheduleTodoMailTasks(settings, now, ActiveUidDuplicateBehavior.DO_NOTHING);
    }

    public void scheduleTodoMailTask(SettingsInfo settings, TodoMailType type, Instant now) {
        scheduleTodoMailTask(settings, type, now, ActiveUidDuplicateBehavior.MERGE_IF_READY);
    }

    private void scheduleTodoMailTasks(SettingsInfo settings, Instant now, ActiveUidDuplicateBehavior deduplication) {
        TodoMailType.types().forEach(type -> scheduleTodoMailTask(settings, type, now, deduplication));
    }

    private void scheduleTodoMailTask(
            SettingsInfo settings, TodoMailType type, Instant now, ActiveUidDuplicateBehavior deduplication)
    {
        type.getTime(settings.getCommon()).forEach(time -> {
            LocalDateTime localNow = new LocalDateTime(now, settings.getTz());

            LocalDateTime localScheduleTime = time.isAfter(localNow.toLocalTime())
                    ? localNow.withFields(time)
                    : localNow.withFields(time).plusDays(1);

            bazingaStorage.addOnetimeJob(PgOnetimeUtils.makeJob(
                    new SendTodoMailTask(settings.getUid(), type),
                    localScheduleTime.toDateTime(settings.getTz()).toInstant()), deduplication);
        });
    }

    private void saveTodoListEmail(PassportUid uid, TodoList todoList, Instant now, Email email) {
        TodoListEmail todoListEmail = new TodoListEmail();
        todoListEmail.setEmail(email);
        todoListEmail.setSentTs(now);
        todoListEmail.setUid(uid);
        todoListEmail.setTodoListId(todoList.getId());
        todoListEmailDao.saveTodoListEmail(todoListEmail);
    }

}
