package ru.yandex.calendar.logic.todo;

import org.jdom.Element;
import org.joda.time.DateTimeZone;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.logic.beans.Bean;
import ru.yandex.calendar.logic.beans.generated.TodoItem;
import ru.yandex.calendar.logic.beans.generated.TodoItemFields;
import ru.yandex.calendar.logic.beans.generated.TodoList;
import ru.yandex.calendar.logic.beans.generated.TodoListFields;
import ru.yandex.calendar.util.base.AuxColl;
import ru.yandex.calendar.util.base.MultiMap;
import ru.yandex.calendar.util.db.CalendarJdbcDaoSupport;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlOrder;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * Appends information about user todo lists and todo lists
 * content (todo items and attached events) to the output xml.
 * @author akirakozov
 */
public class TodoListsXmlAppender extends CalendarJdbcDaoSupport {
    private static final Logger logger = LoggerFactory.getLogger(TodoListsXmlAppender.class);

    public enum Mode {
        DEFAULT, ALL, FOR_SCHEDULE
    }

    @Autowired
    private TodoDao todoDao;

    private void appendXmlInner(Element rootEl, ListF<TodoList> todoLists,
            MapF<Long, ListF<TodoItem>> itemsByListIdMMap, Mode mode, DateTimeZone tz)
    {
        MapF<Long, TodoList> listsMap = todoLists.toMapMappingToKey(TodoListFields.ID.getF());
        ListF<Long> listIds = sortByCompletion(todoLists, itemsByListIdMMap);
        for (Long tlId : listIds) {
            TodoList tl = listsMap.getTs(tlId);
            if (Mode.ALL == mode || itemsByListIdMMap.keySet().containsTs(tlId)) {
                Element todoListEl = tl.addXmlElementTo(rootEl, tz);
                ListF<TodoItem> items = itemsByListIdMMap.getTs(tlId);
                // Add todo items
                if (AuxColl.isSet(items)) {
                    for (int i = 0; i < 2; i++) {
                        boolean open = i == 0;
                        final String collTagName = (open ? "open-todo-items" : "completed-todo-items");
                        Element itemsEl = new Element(collTagName);
                        todoListEl.addContent(itemsEl);
                        for (TodoItem item : items) {
                            if (open != item.getCompletionTs().isPresent()) {
                                Element todoItemEl = new Element("todo-item");
                                itemsEl.addContent(todoItemEl);
                                item.appendXmlTo(
                                    todoItemEl, tz,
                                    TodoItemFields.ID, TodoItemFields.TITLE, TodoItemFields.DUE_TS,
                                    TodoItemFields.POS, TodoItemFields.COLOR
                                );
                            }
                        } // for todo items
                    } // for open/completed
                }
            } // if list contains items
        } // for all lists
    }

    /** @return function that returns: 0 for empty lists (no items), 1 for lists with open items, and 2 for complete lists */
    private Function<TodoList, Integer> getTodoListTypeOrderF(final MapF<Long, ListF<TodoItem>> itemsByListIdMMap) {
        return new Function<TodoList, Integer>() {
            @Override public Integer apply(TodoList a) {
                long todoListId = a.getId();
                if (!itemsByListIdMMap.containsKeyTs(todoListId)) {
                    return 0;
                }
                ListF<TodoItem> items = itemsByListIdMMap.getTs(todoListId);
                for (TodoItem item : items) {
                    if (!item.getCompletionTs().isPresent()) {
                        return 1;
                    }
                }
                return 2;
            }
        };
    }

    /** Sorts list ids according to {@link #getTodoListTypeOrderF(MultiMap)} function (preserving order within same type) */
    private ListF<Long> sortByCompletion(final ListF<TodoList> todoLists, final MapF<Long, ListF<TodoItem>> itemsByListIdMMap) {
        return todoLists.sortedBy(getTodoListTypeOrderF(itemsByListIdMMap).memoize()).map(Bean::getId);
    }

    public void appendXml(Element rootEl, PassportUid uid, DateTimeZone tz, Mode mode, Option<Long> todoListId, SqlCondition todoItemCondition) {
        if (rootEl == null) { throw new IllegalArgumentException("root element is missing"); }
        ListF<TodoList> todoLists = todoDao.findNotDeletedUserTodoLists(uid);
        CalendarXmlizer.setAttr(rootEl, "count", todoLists.size());
        if (!todoLists.isEmpty()) {
            SetF<Long> requestedTodoListIds = todoListId.isPresent() ?
                    Cf.hashSet(todoListId.get()) : todoLists.map(TodoListFields.ID.getF()).unique();

            SqlCondition inCondition = TodoItemFields.TODO_LIST_ID.column().inSet(requestedTodoListIds);
            SqlCondition condition = inCondition.and(todoItemCondition);
            SqlOrder order = SqlOrder.orderByColumn("pos");
            ListF<TodoItem> todoItems = todoDao.findNotDeletedNotArchivedTodoItems(condition, order);

            MapF<Long, ListF<TodoItem>> itemsByListIdMMap = todoItems.groupBy(TodoItemFields.TODO_LIST_ID.getF());

            appendXmlInner(rootEl, todoLists, itemsByListIdMMap, mode, tz);
        }
    }
} //~
