package ru.yandex.calendar.logic.todo;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.EmptyResultDataAccessException;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.CalendarUtils;
import ru.yandex.calendar.frontend.web.cmd.RequestDataConverter;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.frontend.web.cmd.run.Situation;
import ru.yandex.calendar.logic.beans.generated.TodoItem;
import ru.yandex.calendar.logic.beans.generated.TodoItemFields;
import ru.yandex.calendar.logic.beans.generated.TodoList;
import ru.yandex.calendar.logic.beans.generated.TodoListFields;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.grid.ViewType;
import ru.yandex.calendar.logic.todo.id.ListIdOrExternalId;
import ru.yandex.calendar.logic.todo.id.TodoIdOrExternalId;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.util.base.UidGen;
import ru.yandex.calendar.util.color.Color;
import ru.yandex.calendar.util.data.DataProvider;
import ru.yandex.calendar.util.dates.DateTimeFormatter;
import ru.yandex.calendar.util.dates.DayOfWeek;
import ru.yandex.calendar.util.db.DbUtils;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.MasterSlaveUnitUnavailableException;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.lang.Validate;

/**
 * @author ssytnik
 */
public class TodoRoutines {
    public static final String DEFAULT_LIST_NAME = "Не забыть";
    public static final Color DEFAULT_LIST_COLOR = Color.parseRgba("#B9B9B9FF");

    @Autowired
    private TodoDao todoDao;
    @Autowired
    private TodoExportTokenDao todoExportTokenDao;
    @Autowired
    private YandexMailSettingsUpdater yandexMailSettingsUpdater;
    @Autowired
    private TodoListEmailManager todoListEmailManager;
    @Autowired
    private SettingsRoutines settingsRoutines;

    public long createTodoList(PassportUid uid, TodoList todoListData, ActionInfo actionInfo) {
        if (todoListData.isFieldSet(TodoListFields.EXTERNAL_ID)) {
            ListIdOrExternalId id = ListIdOrExternalId.externalId(todoListData.getExternalId(), uid);

            if (todoDao.findNotDeletedTodoListExistsById(id)) {
                throw CommandRunException.createSituation(
                        "List with external_id=" + todoListData.getExternalId() + ", uid=" + uid + " exists",
                        Situation.TODO_LIST_ALREADY_EXISTS);
            }
        } else if (todoDao.getUserCreatedListsCount(uid) == 0) {
            yandexMailSettingsUpdater.updateShowTodoOn(uid);
        }

        TodoList newTodoList = new TodoList();
        newTodoList.setCreatorUid(uid);
        newTodoList.setCreationTs(actionInfo.getNow());
        newTodoList.setCreationSource(actionInfo.getActionSource());
        newTodoList.setFields(todoListData);
        newTodoList.setFieldValueDefault(TodoListFields.EXTERNAL_ID, CalendarUtils.generateExternalId());
        return todoDao.saveTodoList(newTodoList, actionInfo);
    }

    public long createTodoListDefault(PassportUid uid, Option<String> listNameO, ActionInfo actionInfo) {
        TodoList todoListData = new TodoList();
        todoListData.setTitle(listNameO.isPresent() ? listNameO.get() : DEFAULT_LIST_NAME);
        todoListData.setColor(DEFAULT_LIST_COLOR.getRgb());
        return createTodoList(uid, todoListData, actionInfo);
    }

    public long updateTodoList(PassportUid uid, DataProvider dp, ActionInfo actionInfo) {
        // NOTE: there's no need to call 'ensureListBelongsTo', as we look for TodoList using creator_uid
        // XXX refactor to just: UPDATE todo_list SET title = ?, description = ? WHERE creator_uid = ? AND id = ?
        // XXX and then ensure that updated rows count is 1 (it should be 1 even if values remain the same)
        TodoList todoListData = RequestDataConverter.convertTodoList(dp);
        long todoListId = todoListData.getId();

        String newTitle = todoListData.getTitle();
        String newDescription = todoListData.getFieldValueO(TodoListFields.DESCRIPTION).getOrElse("");

        return updateTodoListTitleAndDescriptionById(uid, todoListId, newTitle, newDescription, actionInfo);
    }

    public long updateTodoListTitleAndDescriptionById(
            PassportUid uid, long todoListId, String newTitle, String newDescription, ActionInfo actionInfo)
    {
        TodoList todoList = todoDao.findNotDeletedTodoListById(ListIdOrExternalId.id(todoListId));
        if (!todoList.getCreatorUid().sameAs(uid)) {
            throw new IllegalStateException();
        }

        if (
            !ObjectUtils.equals(newTitle, todoList.getTitle()) ||
            !ObjectUtils.equals(newDescription, todoList.getDescription())
        ) {
            todoDao.updateTodoListTitleAndDescription(todoList.getId(), newTitle, newDescription, actionInfo);
        }
        return todoListId;
    }

    public void updateTodoListById(
            PassportUid uid, TodoList todoListData, ListIdOrExternalId idOrExternalId, ActionInfo actionInfo)
    {
        ensureTodoListExistsByUidAndListId(uid, idOrExternalId);
        Validate.V.isFalse(todoListData.isFieldSet(TodoListFields.ID));
        Validate.V.isFalse(todoListData.isFieldSet(TodoListFields.EXTERNAL_ID));

        TodoList existingList = todoDao.findNotDeletedTodoListById(idOrExternalId);

        Validate.V.isFalse(todoListData.isFieldSet(TodoListFields.CREATOR_UID));
        Validate.V.equals(uid, existingList.getCreatorUid());

        todoDao.updateTodoListById(todoListData, idOrExternalId, actionInfo);
    }

    public void deleteTodoListSafe(PassportUid uid, ListIdOrExternalId idOrExternalId, ActionInfo actionInfo) {
        try {
            deleteTodoList(uid, idOrExternalId, actionInfo);
        } catch (Exception ignored) {
        }
    }

    public void deleteTodoList(PassportUid uid, ListIdOrExternalId idOrExternalId, ActionInfo actionInfo) {
        ensureTodoListExistsByUidAndListId(uid, idOrExternalId);

        Long todoListId;
        if (idOrExternalId.isId()) {
            todoListId = idOrExternalId.getId();
        } else {
            todoListId = todoDao.findTodoListById(idOrExternalId).getId();
        }

        todoDao.updateTodoItemsSetDeletedByTodoListIds(Cf.list(todoListId), actionInfo);
        todoDao.updateTodoListsSetDeletedByIds(Cf.list(todoListId), actionInfo);
    }

    public void ensureTodoListExistsByUidAndListId(PassportUid uid, ListIdOrExternalId id) {
        ensureTodoListsExistByUidAndListIds(uid, Cf.list(id));
    }

    private void ensureTodoListsExistByUidAndListIds(PassportUid uid, ListF<ListIdOrExternalId> todoListIds) {
        todoListIds = todoListIds.stableUnique();
        long todoListsCount = todoDao.findNotDeletedTodoListsCountByCreatorUidAndIds(uid, todoListIds);
        if (todoListsCount != todoListIds.size()) {
            throw CommandRunException.createSituation(
                    "Not all todo lists found for uid = " + uid + ", list ids = " + todoListIds,
                    Situation.TODO_LIST_NOT_FOUND);
        }
    }

    public long createTodoItem(PassportUid uid, DataProvider dp, DateTimeZone tz, ActionInfo actionInfo) {
        TodoItem todoItemData = RequestDataConverter.convertTodoItem(Option.of(tz), dp);
        return checkPermissionsAndCreateTodoItem(uid, todoItemData, actionInfo);
    }

    public long checkPermissionsAndCreateTodoItem(PassportUid uid, TodoItem todoItemData, ActionInfo actionInfo) {
        ensureTodoListExistsByUidAndListId(uid, ListIdOrExternalId.id(todoItemData.getTodoListId()));
        return createTodoItem(uid, todoItemData, actionInfo);
    }

    public long createTodoItem(PassportUid uid, TodoItem todoItemData, ActionInfo actionInfo) {
        TodoItem todoItem = new TodoItem();
        todoItem.setFields(todoItemData);
        todoItem.setCreatorUid(uid);

        boolean isCompletedLng =
                todoItemData.isFieldSet(TodoItemFields.COMPLETION_TS) &&
                todoItemData.getCompletionTs().isPresent();
        if (!todoItem.isFieldSet(TodoItemFields.POS)) {
            int lastPosition = todoDao.findLastTodoItemPositionWithinList(todoItemData.getTodoListId(), isCompletedLng);
            todoItem.setPos(lastPosition + 1);
        }
        if (todoItem.isFieldSet(TodoItemFields.EXTERNAL_ID)) {
            Option<TodoItem> todo =
                    todoDao.findNotDeletedNotArchivedTodoItemByExternalIdAndCreatorUid(todoItem.getExternalId(), uid);
            if (todo.isPresent()) {
                throw CommandRunException.createSituation(
                        "Todo with external_id=" + todoItem.getExternalId() + ", uid=" + uid + " exists",
                        Situation.TODO_ITEM_ALREADY_EXISTS);
            }
        } else {
            todoItem.setExternalId(CalendarUtils.generateExternalId());
        }
        // XXX synchronize with position
        todoItem.setTimestampPosition(actionInfo.getNow().getMillis());

        scheduleTodoEmailsTasksIfNeeded(uid, todoItem, actionInfo.getNow());

        long id = todoDao.saveTodoItem(todoItem, actionInfo);

        todoDao.updateTodoListsStaleLastUpdateTs(Cf.list(todoItemData.getTodoListId()), actionInfo);

        return id;
    }

    public void moveTodoItemByExternalId(PassportUid uid, String externalId,
            long sourceTodoListId, long destinationTodoListId, ActionInfo actionInfo)
    {
        TodoIdOrExternalId id = TodoIdOrExternalId.externalId(externalId);
        Option<TodoItem> item = getNotDeletedTodoItemsByIdsSafe(Cf.list(id), uid)
                .find(TodoItem.getTodoListIdF().andThenEquals(sourceTodoListId));

        if (item.isPresent()) {
            Check.notEquals(item.get().getTodoListId(), destinationTodoListId);

            deleteTodoItemsByIds(uid, Cf.list(TodoIdOrExternalId.id(item.get().getId())), actionInfo);

            TodoItem data = item.get().copy();
            data.unsetField(TodoItemFields.ID);
            data.unsetField(TodoItemFields.LAST_UPDATE_TS);
            data.unsetField(TodoItemFields.MODIFICATION_SOURCE);
            data.setTodoListId(destinationTodoListId);

            createTodoItem(uid, data, actionInfo);
        }
    }

    public Instant updateTodoItem(
            PassportUid uid, TodoItem todoItemData, long todoItemId, ActionInfo actionInfo)
    {
        Validate.V.isFalse(todoItemData.isFieldSet(TodoItemFields.ID));
        Validate.V.isFalse(todoItemData.isFieldSet(TodoItemFields.EXTERNAL_ID));

        TodoItem todoItem = new TodoItem();
        todoItem.setFields(todoItemData);
        Instant lastUpdateTs = actionInfo.getNow();

        scheduleTodoEmailsTasksIfNeeded(uid, todoItem, actionInfo.getNow());

        try {
            long listId = todoDao.updateTodoItemGetListId(todoItem, todoItemId, actionInfo);
            todoDao.updateTodoListsStaleLastUpdateTs(Cf.list(listId), actionInfo);

        } catch (EmptyResultDataAccessException e) {
            throw CommandRunException.createSituation(e, Situation.TODO_ITEM_NOT_FOUND);
        }
        return lastUpdateTs;
    }

    public long updateTodoItem(PassportUid uid, DataProvider dp, DateTimeZone tz, ActionInfo actionInfo) {
        TodoItem todoItemData = RequestDataConverter.convertTodoItem(Option.of(tz), dp);
        long todoItemId = todoItemData.getId();
        todoItemData = todoItemData.withoutId();

        checkPermissionsAndUpdateTodoItem(uid, todoItemData, todoItemId, actionInfo);
        return todoItemId;
    }

    public Instant checkPermissionsAndUpdateTodoItem(
            PassportUid uid, TodoItem todoItemData, long todoItemId, ActionInfo actionInfo)
    {
        long listId;
        if (todoItemData.isFieldSet(TodoItemFields.TODO_LIST_ID)) {
            listId = todoItemData.getTodoListId();
        } else {
            listId = getNotDeletedTodoItemsByIds(
                    Cf.list(TodoIdOrExternalId.id(todoItemId)), uid).single().getTodoListId();
        }
        ensureTodoListExistsByUidAndListId(uid, ListIdOrExternalId.id(listId));
        return updateTodoItem(uid, todoItemData, todoItemId, actionInfo);
    }

    public void updateTodoItemSetCompleted(
            PassportUid uid, TodoIdOrExternalId todoItemId, boolean isCompleted, ActionInfo actionInfo)
    {
        // XXX ssytnik: move this check out to correspond to updateTodoItemSetArchived,
        // and also not to do this work twice in UpdateTodoItemAction
        TodoItem ti;
        try {
            ti = todoDao.findNotDeletedTodoItemByIdAndCreatorUid(todoItemId, uid);
        } catch (EmptyResultDataAccessException e) {
            throw CommandRunException.createSituation(e, Situation.TODO_ITEM_NOT_FOUND);
        }

        if (ti.getCompletionTs().isPresent() != isCompleted) { // does item completion change?

            TodoItem todoItemData = new TodoItem();
            if (ti.getCompletionTs().isPresent()) {
                todoItemData.setCompletionTsNull();
            } else {
                todoItemData.setCompletionTs(actionInfo.getNow());
            }
            int lastPosition = todoDao.findLastTodoItemPositionWithinList(ti.getTodoListId(), isCompleted);
            todoItemData.setPos(lastPosition + 1);
            updateTodoItem(uid, todoItemData, ti.getId(), actionInfo);

        }
    }

    public void updateTodoItemsSetCompleted(
            PassportUid uid, ListF<TodoIdOrExternalId> todoItemIds, boolean completed, ActionInfo actionInfo)
    {
        // XXX batch
        for (TodoIdOrExternalId todoItemId : todoItemIds) {
            updateTodoItemSetCompleted(uid, todoItemId, completed, actionInfo);
        }
    }

    public void updateTodoItemSetArchived(TodoIdOrExternalId todoItemId, boolean newIsArchived, ActionInfo actionInfo) {
        ListF<Long> listIds;

        if (newIsArchived) {
            listIds = todoDao.updateTodoItemsSetArchivedByIdsGetListIds(Cf.list(todoItemId), actionInfo);
        } else {
            listIds = todoDao.updateTodoItemsSetNotArchivedByIdsGetListIds(Cf.list(todoItemId), actionInfo);
        }
        todoDao.updateTodoListsStaleLastUpdateTs(listIds, actionInfo);
    }

    public void updateNotificationTs(long todoItemId, Option<Instant> notificationTs, ActionInfo actionInfo) {
        long listId;

        if (!notificationTs.isPresent()) {
            listId = todoDao.updateTodoItemClearNotificationTsGetListId(todoItemId);
        } else if (notificationTs.get().isAfter(actionInfo.getNow())) {
            listId = todoDao.updateTodoItemNotificationTsClearNotificationSentTsAndNotificationShownInWebGetListId(
                    todoItemId, notificationTs.get());
        } else {
            listId = todoDao.updateTodoItemNotificationTsGetListId(todoItemId, notificationTs.get());
        }
        todoDao.updateTodoListsStaleLastUpdateTs(Cf.list(listId), actionInfo);
    }

    private ListF<Long> obtainTodoItemIds(PassportUid uid, DataProvider dp, long todoListId) {
        ensureTodoListExistsByUidAndListId(uid, ListIdOrExternalId.id(todoListId));

        ListF<Long> requestedIds = RequestDataConverter.convertTodoItemIds(dp);
        ListF<Long> actualIds = todoDao.findNotDeletedNotArchivedNotCompletedTodoItemIdsByTodoListId(todoListId);

        Validate.sameSize(actualIds, requestedIds);
        Validate.equals(actualIds.unique(), requestedIds.unique());

        return requestedIds;
    }

    public void reorder(PassportUid uid, DataProvider dp, ActionInfo actionInfo) {
        TodoItem todoItemData = RequestDataConverter.convertTodoItem(Option.<DateTimeZone>empty(), dp);
        long todoListId = todoItemData.getTodoListId();
        ListF<Long> ids = obtainTodoItemIds(uid, dp, todoListId);

        for (int i = 0; i < ids.length(); ++i) {
            todoDao.updateTodoItemPos(ids.get(i), i + 1, actionInfo);
        }
        if (ids.isNotEmpty()) {
            todoDao.updateTodoListsStaleLastUpdateTs(Cf.list(todoListId), actionInfo);
        }
    }

    public void reorder(PassportUid uid, ListIdOrExternalId listId, Tuple2List<Long, Integer> positions, ActionInfo actionInfo) {
        ensureTodoListExistsByUidAndListId(uid, listId);

        long todoListId = getTodoListId(listId);

        if (todoDao.updateNotDeletedNotArchivedTodoItemsPosition(todoListId, positions, actionInfo) > 0) {
            todoDao.updateTodoListsStaleLastUpdateTs(Cf.list(todoListId), actionInfo);
        }
    }

    // This method is used only in pda version
    public void updateAll(PassportUid uid, DataProvider dp, ActionInfo actionInfo) {
        long todoListId = updateTodoList(uid, dp, actionInfo);
        // Get/update list items info (NOTE: open ones only!)
        ListF<Long> ids = obtainTodoItemIds(uid, dp, todoListId);

        for (int i = 0; i < ids.length(); ++i) {
            todoDao.updateTodoItemTitleAndPos(ids.get(i), dp.getText("item_" + ids.get(i), true), i + 1, actionInfo);
        }
        if (ids.isNotEmpty()) {
            todoDao.updateTodoListsStaleLastUpdateTs(Cf.list(todoListId), actionInfo);
        }
    }

    // todoListIds - can be empty, but not null
    public ListF<TodoList> getTodoLists(PassportUid uid, ListF<Long> todoListIds) {
        SqlCondition c = SqlCondition.all(
                TodoListFields.CREATOR_UID.eq(uid.getUid()),
                DbUtils.getInSetOrAllCondition(TodoListFields.ID, todoListIds)
        );
        return todoDao.findNotDeletedTodoLists(c);
    }

    public ListF<TodoItem> getTodoItemsPlainSorted(PassportUid uid, ListF<Long> todoListIds) {
        return todoDao.findNotDeletedNotArchivedTodoItemsByTodoListIds(todoListIds).sorted((a, b) -> {
            int res = Long.valueOf(a.getTodoListId()).compareTo(b.getTodoListId());
            if (res == 0) {
                res = Boolean.valueOf(a.getCompletionTs().isPresent()).compareTo(b.getCompletionTs().isPresent());
            }
            if (res == 0) {
                res = Integer.valueOf(a.getPos()).compareTo(b.getPos());
            }
            return res;
        });
    }

    public static SqlCondition createItemCondBuilder(
            DateTimeZone tz, LocalDate showDate, ViewType vt,
            DayOfWeek startWeekday)
    {
        Tuple2<DateTime, DateTime> bounds = DateTimeFormatter.getViewTypeBounds(tz, showDate, vt, startWeekday);
        DateTime startTs = bounds._1;
        DateTime endTs = bounds._2;
        SqlCondition condition = TodoItemFields.DUE_TS.column().isNotNull();
        condition = condition.and(TodoItemFields.DUE_TS.column().ge(startTs));
        return condition.and(TodoItemFields.DUE_TS.column().lt(endTs));
    }

    public void deleteTodoItemsByIds(PassportUid uid, ListF<TodoIdOrExternalId> todoItemIds, ActionInfo actionInfo) {
        ListF<TodoItem> todoItems = getNotDeletedTodoItemsByIdsSafe(todoItemIds, uid);
        if (todoItems.isNotEmpty()) {
            deleteTodoItems(uid, todoItems, actionInfo);
        }
    }

    public void deleteTodoItemById(PassportUid uid, TodoIdOrExternalId todoItemId, ActionInfo actionInfo) {
        deleteTodoItemsByIds(uid, Cf.list(todoItemId), actionInfo);
    }

    public void archiveTodoItemsByIds(PassportUid uid, ListF<TodoIdOrExternalId> todoItemIds, ActionInfo actionInfo) {
        ListF<TodoItem> todoItems = todoDao.findNotDeletedNotArchivedTodoItemByIds(todoItemIds, uid);
        if (todoItems.isNotEmpty()) {
            ListF<Long> todoListIds = todoItems.map(TodoItemFields.TODO_LIST_ID.getF());
            ensureTodoListsExistByUidAndListIds(uid, todoListIds.map(ListIdOrExternalId.idF()));

            todoDao.updateTodoItemsSetArchivedByIdsGetListIds(todoItemIds, actionInfo);
            todoDao.updateTodoListsStaleLastUpdateTs(todoListIds, actionInfo);
        }
    }

    public void restoreTodoItemsFromArchiveByIds(
            PassportUid uid, ListF<TodoIdOrExternalId> todoItemIds, ActionInfo actionInfo)
    {
        ListF<TodoItem> todoItems = getNotDeletedTodoItemsByIds(todoItemIds, uid);
        if (todoItems.isNotEmpty()) {
            ListF<Long> todoListIds = todoItems.map(TodoItemFields.TODO_LIST_ID.getF());
            ensureTodoListsExistByUidAndListIds(uid, todoListIds.map(ListIdOrExternalId.idF()));

            todoDao.updateTodoItemsSetNotArchivedAndNotCompletedByIds(todoItemIds, actionInfo);
            todoDao.updateTodoListsStaleLastUpdateTs(todoListIds, actionInfo);
        }
    }

    public void restoreTodoItemsByIds(PassportUid uid, ListF<TodoIdOrExternalId> todoItemIds, ActionInfo actionInfo) {
        ListF<TodoItem> todoItems = todoDao.findDeletedTodoItemsByTodoItemIds(todoItemIds, uid);
        if (todoItems.isNotEmpty()) {
            ListF<Long> todoListIds = todoItems.map(TodoItemFields.TODO_LIST_ID.getF());
            ensureTodoListsExistByUidAndListIds(uid, todoListIds.map(ListIdOrExternalId.idF()));

            todoDao.updateTodoItemsSetNotDeletedByIds(todoItems.map(TodoItem.getIdF()), actionInfo);
            todoDao.updateTodoListsStaleLastUpdateTs(todoListIds, actionInfo);
        }
    }

    public void deleteTodoItems(PassportUid uid, ListF<TodoItem> todoItems, ActionInfo actionInfo) {
        ListF<Long> todoListIds = todoItems.map(TodoItemFields.TODO_LIST_ID.getF());
        ensureTodoListsExistByUidAndListIds(uid, todoListIds.map(ListIdOrExternalId.idF()));

        todoDao.updateTodoItemsSetDeletedByIds(todoItems.map(TodoItemFields.ID.getF()), actionInfo);
        todoDao.updateTodoListsStaleLastUpdateTs(todoListIds, actionInfo);
    }

    public void hardDeleteTodoListsByUids(ListF<PassportUid> uids, ActionInfo actionInfo) {
        ListF<TodoList> todoLists = todoDao.findTodoListsByUid(uids);
        if (todoLists.isNotEmpty()) {
            ListF<Long> todoListIds = todoLists.map(TodoListFields.ID.getF());
            todoDao.deleteTodoItemsByTodoListIds(todoListIds);
            todoDao.deleteTodoListsByTodoListIds(todoListIds);
        }
    }

    public void restoreTodoListById(PassportUid uid, ListIdOrExternalId todoListId, ActionInfo actionInfo) {
        TodoList list = todoDao.findTodoListById(todoListId);
        Validate.equals(uid, list.getCreatorUid(), "Cannot restore another user todo list");

        if (!list.getDeleted()) return;

        todoDao.updateTodoListSetNotDeletedByIds(todoListId, actionInfo);
        todoDao.updateTodoItemsSetNotDeletedByTodoListIdAndDeletionTs(list.getId(), list.getDeletionTs().get(), actionInfo);
    }

    public Option<Long> findFirstCreatedTodoListId(PassportUid uid) {
        return todoDao.findFirstCreatedNotDeletedUserTodoList(uid).map(TodoList::getId);
    }

    public void createTodoListIfUserHasNoLists(PassportUid passportUid, ActionInfo actionInfo) {
        if (todoDao.getUserNotDeletedListsCount(passportUid) == 0) {
            createTodoListDefault(passportUid, Option.<String>empty(), actionInfo);
        }
    }

    public TodoSynchStatus getTodoSynchStatus(
            PassportUid uid, TodoItem changes, TodoIdOrExternalId todoItemId,
            Option<Boolean> completed, Option<Boolean> archived, Option<Instant> timestamp)
    {
        Option<TodoItem> existingTodo = getNotDeletedTodoItemsByIds(Cf.list(todoItemId), uid).singleO();
        if (existingTodo.isPresent()) {
            return TodoSynchStatus.found(
                    existingTodo.get(), changes, existingTodo.get().getId(), completed, archived, timestamp);
        } else {
            return TodoSynchStatus.notFound();
        }
    }

    public TodoListsAndTodoItems getTodoListsAndNotCompletedTodoItems(
            int skip, int count, PassportUid uid, Option<Instant> modifiedSince)
    {
        SqlCondition todoListCondition = SqlCondition.trueCondition();
        SqlCondition todoItemCondition = TodoItemFields.COMPLETION_TS.column().isNull();

        if (modifiedSince.isNotEmpty()) {
            todoListCondition = todoListCondition.and(
                    TodoListFields.CREATION_TS.gt(modifiedSince.get())
                    .or(TodoListFields.LAST_UPDATE_TS.gt(modifiedSince.get())));

            todoItemCondition = todoItemCondition.and(
                    TodoItemFields.CREATION_TS.gt(modifiedSince.get())
                    .or(TodoItemFields.LAST_UPDATE_TS.gt(modifiedSince.get())));
        }

        int totalTodoLists = todoDao.getUserNotDeletedListsCount(uid, todoListCondition);
        int totalTodoItems = todoDao.getUserNotDeletedNotArchivedTodoItemsCount(uid, todoItemCondition);

        ListF<TodoList> todoLists = Cf.list();
        ListF<TodoItem> todoItems = Cf.list();

        int todoListsInResponse = Math.min(totalTodoLists - skip, count);
        if (todoListsInResponse > 0) {
            todoLists = todoDao.findNotDeletedTodoListsByCreatorUid(
                    uid, skip, todoListsInResponse, todoListCondition);
        }

        int skipTodoItems = Math.max(skip - totalTodoLists, 0);
        int todoItemsInResponse = skip + count - (totalTodoLists + skipTodoItems);
        if (todoItemsInResponse > 0) {
            todoItems = todoDao.findNotDeletedNotArchivedTodoItemsByCreatorUid(
                    uid, skipTodoItems, todoItemsInResponse, todoItemCondition);
        }

        return new TodoListsAndTodoItems(todoLists, todoItems, totalTodoLists, totalTodoItems);
    }

    public long findFirstCreatedListOrCreateNewWithName(PassportUid uid, String listName, ActionInfo actionInfo) {
        Option<TodoList> todoListO = todoDao.findFirstCreatedNotDeletedUserTodoList(uid);
        if (!todoListO.isPresent()) {
            return createTodoListDefault(uid, Option.of(listName), actionInfo);
        } else {
            return todoListO.get().getId();
        }
    }

    public static Function1B<TodoItem> isCompletedF() {
        return new Function1B<TodoItem>() {
            public boolean apply(TodoItem todoItem) {
                return todoItem.getCompletionTs().isPresent();
            }
        };
    }

    public ListF<TodoItem> getNotDeletedTodoItemsByIds(ListF<TodoIdOrExternalId> todoItemIds, PassportUid uid) {
        todoItemIds = todoItemIds.stableUnique();
        ListF<TodoItem> todoItems = todoDao.findNotDeletedNotArchivedTodoItemByIds(todoItemIds, uid);
        if (todoItems.length() != todoItemIds.length()) {
            throw CommandRunException.createSituation("Todo item not found by id", Situation.TODO_ITEM_NOT_FOUND);
        }
        return todoItems;
    }

    public ListF<TodoItem> getNotDeletedTodoItemsByIdsSafe(ListF<TodoIdOrExternalId> todoItemIds, PassportUid uid) {
        todoItemIds = todoItemIds.stableUnique();
        return todoDao.findNotDeletedNotArchivedTodoItemByIds(todoItemIds, uid);
    }

    public ListF<TodoList> getNonDeletedTodoLists(PassportUid uid) {
        return todoDao.findNonDeletedTodoListsByCreatorUid(uid);
    }

    public long getTodoListId(ListIdOrExternalId todoListIdOrExternalId) {
        if (todoListIdOrExternalId.isId()) {
            return todoListIdOrExternalId.getId();
        } else {
            return todoDao.findTodoListById(todoListIdOrExternalId).getId();
        }
    }

    public Instant getTodoLastUpdateTs(PassportUid uid) {
        return todoDao.findMaxTodoListLastUpdateTs(uid).getOrElse(new Instant(0));
    }

    public Option<String> obtainTodoExportToken(PassportUid uid) {
        Option<String> found = todoExportTokenDao.findTokenByUid(uid);

        if (found.isPresent()) return found;

        MasterSlaveContextHolder.PolicyHandle handle = MasterSlaveContextHolder.push(MasterSlavePolicy.RW_M);
        try {
            String generated = UidGen.createPrivateToken();

            todoExportTokenDao.saveTokenByUid(uid, generated);
            return Option.of(generated);

        } catch (MasterSlaveUnitUnavailableException ex) {
            return Option.empty();

        } finally {
            handle.popSafely();
        }
    }

    public PassportUid getUidByExportToken(String token) {
        return todoExportTokenDao.findUidByToken(token).getOrThrow(CommandRunException.createSituationF(
                "Uid cannot be found by export token " + token, Situation.INVALID_TOKEN));
    }

    private void scheduleTodoEmailsTasksIfNeeded(PassportUid uid, TodoItem todoItemData, Instant now) {
        if (todoItemData.getFieldValueO(TodoItemFields.DUE_TS).filterNotNull().isPresent()
            && !todoItemData.getFieldValueO(TodoItemFields.COMPLETION_TS).filterNotNull().isPresent())
        {
            todoListEmailManager.scheduleTodoMailTasksIfNotYet(settingsRoutines.getSettingsByUid(uid), now);
        }
    }

}
