package ru.yandex.calendar.logic.todo;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.beans.generated.TodoItem;
import ru.yandex.commune.mapObject.MapField;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author gutman
 */
public abstract class TodoSynchStatus {
    private static final Logger logger = LoggerFactory.getLogger(TodoSynchStatus.class);

    public static Found found(
            TodoItem oldState, TodoItem changes, long todoItemId, Option<Boolean> nowCompleted,
            Option<Boolean> nowArchived, Option<Instant> timestamp)
    {
         return new Found(oldState, changes, todoItemId, nowCompleted, nowArchived, timestamp);
    }

    public static NotFound notFound() {
        return new NotFound();
    }

    public boolean isFound() {
        return this instanceof Found;
    }

    public Found found() {
        if (!isFound()) {
            throw new IllegalStateException("Todo not found");
        }
        return (Found) this;
    }

    public static class Found extends TodoSynchStatus {
        private final TodoItem oldState;
        private final TodoItem changes;
        private final long todoItemId;
        private final Option<Boolean> nowCompleted;
        private final Option<Boolean> nowArchived;
        private final Option<Instant> timestamp;

        public Found(
                TodoItem oldState, TodoItem changes, long todoItemId,
                Option<Boolean> nowCompleted, Option<Boolean> nowArchived,
                Option<Instant> timestamp) {
            this.oldState = oldState;
            this.changes = changes;
            this.todoItemId = todoItemId;
            this.nowCompleted = nowCompleted;
            this.nowArchived = nowArchived;
            this.timestamp = timestamp;
        }

        public boolean wasConflictingChange() {
            if (!timestamp.isPresent()) {
                logger.debug("No conflicting change for todo " + oldState.getTitle() + ", " + oldState.getId() +
                        ": new timestamp is empty");
                return false;
            }
            if (timestamp.get().getMillis() >= oldState.getLastUpdateTs().getMillis()) {
                logger.debug("No conflicting change for todo " + oldState.getTitle() + ", " + oldState.getId() +
                        ": new timestamp " + timestamp + ", old timestamp: "  + oldState.getLastUpdateTs());
                return false; // we got changes of latest version
            }

            boolean wasChange = wasChange();
            logger.debug("Was change for todo " + oldState.getTitle() + ", " + oldState.getId() +
                    ": " + wasChange);

            return wasChange;
        }

        public boolean wasChange() {
            if (completedChanged() || archivedChanged()) {
                return true;
            }
            for (MapField<?> mapField : changes.getSetFields()) {
                // workaround eclipse, was: equals // ssytnik@
                if (!ObjectUtils.equalsTu(oldState.getFieldValue(mapField), changes.getFieldValue(mapField))) {
                    return true;
                }
            }
            return false;
        }

        public boolean completedChanged() {
            return nowCompleted.isPresent() && nowCompleted.get() != oldState.getCompletionTs().isPresent();
        }

        public boolean changedToCompleted() {
            return completedChanged() && getNowCompleted();
        }

        public boolean getNowCompleted() {
            return nowCompleted.get();
        }

        public boolean archivedChanged() {
            return nowArchived.isPresent() && nowArchived.get() != oldState.getArchived();
        }

        public boolean getNowArchived() {
            return nowArchived.get();
        }

        public long getTodoItemId() {
            return todoItemId;
        }

    }

    private static class NotFound extends TodoSynchStatus {
    }

}
