package ru.yandex.calendar.logic.todo;

import org.joda.time.DateTimeZone;
import org.joda.time.LocalDate;

import ru.yandex.calendar.logic.event.grid.ViewType;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.dates.DayOfWeek;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.enums.StringEnum;
import ru.yandex.misc.enums.StringEnumResolver;
import ru.yandex.misc.lang.CamelWords;

/**
 * Specifies type of todo view which user looks at
 * @author akirakozov
 */
public enum TodoViewType implements StringEnum {
    ALL {
        @Override
        public SqlCondition getSearchCondition(DateTimeZone tz, DayOfWeek startWeekday) {
            return SqlCondition.trueCondition();
        }
    },
    TODAY {
        @Override
        public SqlCondition getSearchCondition(DateTimeZone tz, DayOfWeek startWeekday) {
            LocalDate showDate = new LocalDate(tz);
            return TodoRoutines.createItemCondBuilder(tz, showDate, ViewType.DAY, startWeekday);
        }
    },
    TOMORROW {
        @Override
        public SqlCondition getSearchCondition(DateTimeZone tz, DayOfWeek startWeekday) {
            LocalDate showDate = new LocalDate(tz).plusDays(1);
            return TodoRoutines.createItemCondBuilder(tz, showDate, ViewType.DAY, startWeekday);
        }
    },
    CUR_WEEK {
        @Override
        public SqlCondition getSearchCondition(DateTimeZone tz, DayOfWeek startWeekday) {
            LocalDate showDate = AuxDateTime.getStartWeekMidnight(tz).toLocalDate();
            return TodoRoutines.createItemCondBuilder(tz, showDate, ViewType.WEEK, startWeekday);
        }
    },
    NEXT_WEEK {
        @Override
        public SqlCondition getSearchCondition(DateTimeZone tz, DayOfWeek startWeekday) {
            LocalDate showDate = new LocalDate(tz).plusWeeks(1);
            return TodoRoutines.createItemCondBuilder(tz, showDate, ViewType.WEEK, startWeekday);
        }
    };

    public static final StringEnumResolver<TodoViewType> R = StringEnumResolver.r(TodoViewType.class);

    public static TodoViewType find(String s) {
        return R.valueOf(s);
    }

    public String toDbValue() { return CamelWords.parse(this.name()).toDbName(); }

    @Override
    public String value() {
        return toDbValue();
    }
    public abstract SqlCondition getSearchCondition(DateTimeZone tz, DayOfWeek startWeekday);
}
