package ru.yandex.calendar.logic.todo;

import java.nio.charset.StandardCharsets;
import java.util.List;

import javax.annotation.PreDestroy;

import com.google.common.base.Charsets;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.apache.http.client.HttpClient;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.message.BasicNameValuePair;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.calendar.tvm.TvmClient;
import ru.yandex.calendar.tvm.TvmHeaders;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;

@Slf4j
@RequiredArgsConstructor
public class YandexMailSettingsUpdater {
    @Autowired
    private TvmClient tvmClient;
    @Autowired
    private EnvironmentType environmentType;
    private final String mailSettingsUrl;
    private final int mailSettingsTVMid;
    private final HttpClient httpClient;

    /**
     * Check if environment is appropriate, set "show_todo"="on". Handle Exception inside.
     */
    void updateShowTodoOn(PassportUid uid) {
        if (environmentType != EnvironmentType.PRODUCTION && environmentType != EnvironmentType.TESTING) {
            log.info("Cannot update todo settings because of unsupported environment");
            return;
        }
        try {
            updateShowTodo(uid, true);
        } catch (Exception e) {
            log.error("Unable to update todo for client with uid = {}", uid, e);
        }
    }

    @SneakyThrows
    void updateShowTodo(PassportUid uid, boolean on) {
        log.info("Set show_todo in '{}' for user with uid = '{}'. Starting...", on ? "on" : "off", uid);
        sendRequestAndGetResponseText(makeUpdateRequest(uid, on));
    }

    @SneakyThrows
    String getCurrentShowTodoState(PassportUid uid) {
        log.info("Get show_todo value for user with uid = '{}'. Starting...", uid);
        return sendRequestAndGetResponseText(makeGetRequest(uid));
    }

    /**
     * Prepare request to update "show_todo"-setting.
     */
    private HttpPost makeUpdateRequest(PassportUid uid, boolean on) {
        val request = new HttpPost(mailSettingsUrl + "/update_profile?uid=" + uid);
        addServiceTicket(request);
        addShowTodoToBody(request, on);
        return request;
    }

    /**
     * Prepare request to get current value of "show_todo"-setting.
     */
    private HttpGet makeGetRequest(PassportUid uid) {
        val request = new HttpGet(mailSettingsUrl + "/get_profile?uid=" + uid + "&settings_list=show_todo");
        addServiceTicket(request);
        return request;
    }

    private void addShowTodoToBody(HttpPost request, boolean on) {
        val parametersList = List.of(new BasicNameValuePair("show_todo", on ? "on" : "off"));
        request.setEntity(new UrlEncodedFormEntity(parametersList, StandardCharsets.UTF_8));
    }

    private void addServiceTicket(HttpRequestBase httpRequestBase) {
        httpRequestBase.setHeader(TvmHeaders.SERVICE_TICKET, tvmClient.getServiceTicketFor(mailSettingsTVMid));
    }

    /**
     * Run request, check response code and return response body as String.
     */
    @SneakyThrows
    private String sendRequestAndGetResponseText(HttpRequestBase request) {
        val response = httpClient.execute(request);
        val statusCode = response.getStatusLine().getStatusCode();
        val responseText = new String(response.getEntity().getContent().readAllBytes(), Charsets.UTF_8);
        log.info("Obtained response with status code '{}' and body '{}'.", statusCode, responseText);
        if (statusCode < 200 || statusCode >= 300) {
            throw new RuntimeException("Bad response code: " + response.getStatusLine().getStatusCode());
        }
        return responseText;
    }

    @PreDestroy
    public void destroy() {
        ApacheHttpClientUtils.stopQuietly(httpClient);
    }
}
