package ru.yandex.calendar.logic.todo.id;

import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.logic.beans.generated.TodoListFields;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.annotation.FillWithSomething;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.reflection.ClassX;

/**
 * @author gutman
 */
public class ListIdOrExternalId extends DefaultObject {

    private final Either<Long, Tuple2<String, PassportUid>> idOrExternalId;

    @FillWithSomething
    public ListIdOrExternalId(Either<Long, Tuple2<String, PassportUid>> idOrExternalId) {
        this.idOrExternalId = idOrExternalId;
    }

    public long getId() {
        return idOrExternalId.getLeft();
    }

    public String getExternalId() {
        return idOrExternalId.getRight()._1;
    }

    public PassportUid getUid() {
        return idOrExternalId.getRight()._2;
    }

    public boolean isId() {
        return idOrExternalId.isLeft();
    }

    public SqlCondition idEq() {
        if (isId()) {
            return TodoListFields.ID.eq(idOrExternalId.getLeft());
        } else {
            String externalId = idOrExternalId.getRight()._1;
            PassportUid creatorUid = idOrExternalId.getRight()._2;
            return TodoListFields.EXTERNAL_ID.eq(externalId).and(TodoListFields.CREATOR_UID.eq(creatorUid));
        }
    }

    public String toIdOrExternalIdString() {
        if (isId()) {
            return Long.toString(idOrExternalId.getLeft());
        } else {
            return idOrExternalId.getRight()._1;
        }
    }

    public static ListIdOrExternalId id(long todoListId) {
        return new ListIdOrExternalId(Either.<Long, Tuple2<String, PassportUid>>left(todoListId));
    }

    public static ListIdOrExternalId externalId(String todoListExternalId, PassportUid uid) {
        return new ListIdOrExternalId(Either.<Long, Tuple2<String, PassportUid>>right(
                Tuple2.tuple(todoListExternalId, uid)));
    }

    public static SqlCondition idsInSetCondition(ListF<ListIdOrExternalId> idOrExternalIds) {
        if (idOrExternalIds.isEmpty()) {
            return SqlCondition.falseCondition();
        }
        Validate.hasSize(1, idOrExternalIds.map(ClassX::getClass).unique());

        Tuple2<ListF<ListIdOrExternalId>, ListF<ListIdOrExternalId>> x =
                idOrExternalIds.partition(isIdF());
        ListF<ListIdOrExternalId> ids = x.get1();
        ListF<ListIdOrExternalId> externalIds = x.get2();

        SqlCondition idsCondition = TodoListFields.ID.column().inSet(ids.map(getIdF()));

        SqlCondition externalIdsCondition;
        if (externalIds.isEmpty()) {
            externalIdsCondition = SqlCondition.falseCondition();
        } else {
            Validate.hasSize(1, externalIds.map(getExternalIdUidF()).unique());
            externalIdsCondition =
                    TodoListFields.EXTERNAL_ID.column().inSet(externalIds.map(getExternalIdF()))
                            .and(TodoListFields.CREATOR_UID.eq(externalIds.first().idOrExternalId.getRight()._2));
        }

        return idsCondition.or(externalIdsCondition);
    }

    public static Function<Long, ListIdOrExternalId> idF() {
        return new Function<Long, ListIdOrExternalId>() {
            public ListIdOrExternalId apply(Long id) {
                return id(id);
            }
        };
    }

    public static Function<String, ListIdOrExternalId> externalIdF(final PassportUid uid) {
        return new Function<String, ListIdOrExternalId>() {
            public ListIdOrExternalId apply(String id) {
                return externalId(id, uid);
            }
        };
    }

    public static Function<ListIdOrExternalId, Long> getIdF() {
        return new Function<ListIdOrExternalId, Long>() {
            public Long apply(ListIdOrExternalId todoItemId) {
                return todoItemId.idOrExternalId.getLeft();
            }
        };
    }

    private static Function<ListIdOrExternalId, String> getExternalIdF() {
        return new Function<ListIdOrExternalId, String>() {
            public String apply(ListIdOrExternalId todoItemId) {
                return todoItemId.idOrExternalId.getRight()._1;
            }
        };
    }

    private static Function<ListIdOrExternalId, PassportUid> getExternalIdUidF() {
        return new Function<ListIdOrExternalId, PassportUid>() {
            public PassportUid apply(ListIdOrExternalId todoItemId) {
                return todoItemId.idOrExternalId.getRight()._2;
            }
        };
    }

    private static Function1B<ListIdOrExternalId> isIdF() {
        return new Function1B<ListIdOrExternalId>() {
            public boolean apply(ListIdOrExternalId todoItemId) {
                return todoItemId.isId();
            }
        };
    }

    @Override
    public String toString() {
        if (isId()) {
            return String.valueOf(getId());
        } else {
            return "external id = " + getExternalId() + ", uid = " + getUid();
        }
    }
}
